package org.jasig.cas.web.flow;

import java.net.URLEncoder;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;

import org.jasig.cas.logout.LogoutManager;
import org.jasig.cas.logout.LogoutRequest;
import org.jasig.cas.logout.LogoutRequestStatus;
import org.jasig.cas.web.support.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;

public final class FrontChannelLogoutAction extends AbstractLogoutAction {
	public static final String DEFAULT_LOGOUT_PARAMETER = "SAMLRequest";
	public static final String DEFAULT_FLOW_ATTRIBUTE_LOGOUT_URL = "logoutUrl";
	private static final Logger LOGGER = LoggerFactory.getLogger(FrontChannelLogoutAction.class);

	private String logoutRequestParameter = "SAMLRequest";

	@NotNull
	private final LogoutManager logoutManager;

	public FrontChannelLogoutAction(LogoutManager logoutManager) {
		this.logoutManager = logoutManager;
	}

	protected Event doInternalExecute(HttpServletRequest request, HttpServletResponse response, RequestContext context) throws Exception {
		List<LogoutRequest> logoutRequests = WebUtils.getLogoutRequests(context);
		Integer startIndex = Integer.valueOf(getLogoutIndex(context));
		if (logoutRequests != null) {
			for (int i = startIndex.intValue(); i < logoutRequests.size(); i++) {
				LogoutRequest logoutRequest = (LogoutRequest) logoutRequests.get(i);
				if (logoutRequest.getStatus() == LogoutRequestStatus.NOT_ATTEMPTED) {
					logoutRequest.setStatus(LogoutRequestStatus.SUCCESS);

					putLogoutIndex(context, i + 1);

					String logoutUrl = logoutRequest.getLogoutUrl().toExternalForm();
					LOGGER.debug("Using logout url [{}] for front-channel logout requests", logoutUrl);

					String logoutMessage = this.logoutManager.createFrontChannelLogoutMessage(logoutRequest);
					LOGGER.debug("Front-channel logout message to send under [{}] is [{}]", this.logoutRequestParameter,
							logoutMessage);

					UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(logoutUrl);
					builder.queryParam(this.logoutRequestParameter,
							new Object[] { URLEncoder.encode(logoutMessage, "UTF-8") });

					return result("redirectApp", "logoutUrl", builder.build().toUriString());
				}
			}

		}

		return new Event(this, "finish");
	}

	public LogoutManager getLogoutManager() {
		return this.logoutManager;
	}

	public void setLogoutRequestParameter(String logoutRequestParameter) {
		this.logoutRequestParameter = logoutRequestParameter;
	}
}