package org.jasig.cas.web.support;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.jasig.cas.util.CipherExecutor;
import org.jasig.cas.util.NoOpCipherExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DefaultCasCookieValueManager implements CookieValueManager {
	private static final Logger LOGGER = LoggerFactory.getLogger(DefaultCasCookieValueManager.class);
	private static final char COOKIE_FIELD_SEPARATOR = '@';
	private static final int COOKIE_FIELDS_LENGTH = 3;
	private final CipherExecutor cipherExecutor;

	public DefaultCasCookieValueManager() {
		this(new NoOpCipherExecutor());
	}

	public DefaultCasCookieValueManager(CipherExecutor cipherExecutor) {
		this.cipherExecutor = cipherExecutor;
		LOGGER.debug("Using cipher [{} to encrypt and decode the cookie", this.cipherExecutor.getClass());
	}

	public String buildCookieValue(String givenCookieValue, HttpServletRequest request) {
		StringBuilder builder = new StringBuilder(givenCookieValue);

		String remoteAddr = request.getRemoteAddr();
		if (StringUtils.isBlank(remoteAddr)) {
			throw new IllegalStateException("Request does not specify a remote address");
		}
		builder.append(COOKIE_FIELD_SEPARATOR);
		builder.append(remoteAddr);

		String userAgent = request.getHeader("user-agent");
		if (StringUtils.isBlank(userAgent)) {
			throw new IllegalStateException("Request does not specify a user-agent");
		}
		builder.append(COOKIE_FIELD_SEPARATOR);
		builder.append(userAgent);

		String res = builder.toString();
		LOGGER.debug("Encoding cookie value [{}]", res);
		return this.cipherExecutor.encode(res);
	}

	public String obtainCookieValue(Cookie cookie, HttpServletRequest request) {
		String cookieValue = this.cipherExecutor.decode(cookie.getValue());

		LOGGER.debug("Decoded cookie value is [{}]", cookieValue);
		if (StringUtils.isBlank(cookieValue)) {
			LOGGER.debug("Retrieved decoded cookie value is blank. Failed to decode cookie [{}]", cookie.getName());
			return null;
		}
		String[] cookieParts = cookieValue.split(String.valueOf(COOKIE_FIELD_SEPARATOR));
		if (cookieParts.length != COOKIE_FIELDS_LENGTH) {
			throw new IllegalStateException("Invalid cookie. Required fields are missing");
		}
		String value = cookieParts[0];
		String remoteAddr = cookieParts[1];
		String userAgent = cookieParts[2];

		if ((StringUtils.isBlank(value)) || (StringUtils.isBlank(remoteAddr)) || (StringUtils.isBlank(userAgent))) {
			throw new IllegalStateException("Invalid cookie. Required fields are empty");
		}

		LOGGER.debug(new StringBuilder().append("obtainCookieValue request.getRemoteAddr")
				.append(request.getRemoteAddr()).toString());
		LOGGER.debug(new StringBuilder().append("obtainCookieValue remoteAddr").append(remoteAddr).toString());

		if (!userAgent.equals(request.getHeader("user-agent"))) {
			throw new IllegalStateException(
					new StringBuilder().append("Invalid cookie. Required user-agent does not match ")
							.append(request.getHeader("user-agent")).toString());
		}

		return value;
	}
}