package uap.web.cache;

import java.io.Serializable;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springside.modules.nosql.redis.JedisTemplate;
import org.springside.modules.nosql.redis.JedisUtils;

import com.yonyou.uap.tenant.web.filter.PerformanceLoggerCollector;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.exceptions.JedisConnectionException;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.exceptions.JedisException;
import uap.web.cache.CacheManager.JedisAction;
import uap.web.cache.utils.SerializUtil;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class CacheManager {
	public static final String TOKEN_SEED = "token_seed";
	public static final String DEFAULT_CHARSET = "UTF-8";
	private final Logger logger;
	private int sessionTimeout;
	private JedisTemplate jedisTemplate;

	public CacheManager() {
		this.logger = LoggerFactory.getLogger(getClass());

		this.sessionTimeout = 3600;
	}

	public JedisTemplate getJedisTemplate() {
		return this.jedisTemplate;
	}

	public void setJedisTemplate(JedisTemplate jedisTemplate) {
		this.jedisTemplate = jedisTemplate;
	}

	public int getSessionTimeout() {
		return this.sessionTimeout;
	}

	public void setSessionTimeout(int sessionTimeout) {
		this.sessionTimeout = sessionTimeout;
	}

	public Map<String, Object> getAllSessionAttrCache(String sid) {
		HashMap hashMap = new HashMap();
		Map redisMap = hgetAll(sid);
		for (Iterator iterator = redisMap.keySet().iterator(); iterator.hasNext();) {
			byte[] byteKey = (byte[]) iterator.next();
			String key = new String(byteKey, Charset.forName("UTF-8"));
			Object obj = SerializUtil.byteToObject((byte[]) redisMap.get(byteKey));
			hashMap.put(key, obj);
		}
		expire(sid, this.sessionTimeout);
		return hashMap;
	}

	public Map<String, Object> getAllTicketAttrCache(String sid) {
		HashMap hashMap = new HashMap();
		Map redisMap = hgetAll(sid);
		for (Iterator iterator = redisMap.keySet().iterator(); iterator.hasNext();) {
			byte[] byteKey = (byte[]) iterator.next();
			String key = new String(byteKey, Charset.forName("UTF-8"));
			Object obj = SerializUtil.byteToObjectByKryo((byte[]) redisMap.get(byteKey));
			hashMap.put(key, obj);
		}
		expire(sid, this.sessionTimeout);
		return hashMap;
	}

	public void removeSessionCache(String sid) {
		this.jedisTemplate.del(new String[] { sid });
	}

	public <T extends Serializable> void putSessionCacheAttribute(String sid, String key, T value) {
		putSessionCacheAttribute(sid, key, value, this.sessionTimeout);
	}

	public <T extends Serializable> void putTicketCacheAttribute(String sid, String key, T value) {
		putTicketCacheAttribute(sid, key, value, this.sessionTimeout);
	}

	public <T extends Serializable> void putTicketCacheAttribute(String sid, String key, T value, int timeout) {
		hsetByKryo(sid, key, value);
		if (timeout > 0)
			expire(sid, timeout);
	}

	public <T extends Serializable> void putSessionCacheAttribute(String sid, String key, T value, int timeout) {
		hset(sid, key, value);
		if (timeout > 0)
			expire(sid, timeout);
	}

	public <T extends Serializable> void updateSessionCacheAttribute(String sid, String key, T value) {
		if (this.jedisTemplate.hexists(sid, key).booleanValue())
			putSessionCacheAttribute(sid, key, value);
	}

	public <T extends Serializable> T getSessionCacheAttribute(String sid, String key) {
		T result = null;
		boolean isExist = exists(sid).booleanValue();
		if (isExist) {
			expire(sid, this.sessionTimeout);
			result = hget(sid, key);
		}
		return result;
	}

	public <T extends Serializable> T getTicketCacheAttribute(String sid, String key) {
		T result = null;
		boolean isExist = exists(sid).booleanValue();
		if (isExist) {
			expire(sid, this.sessionTimeout);
			result = hgetByKryo(sid, key);
		}
		return result;
	}

	public void removeSessionCacheAttribute(String sid, String key) {
		if ((StringUtils.isNotBlank(sid)) && (StringUtils.isNotBlank(key)))
			this.jedisTemplate.hdel(sid, new String[] { key });
	}

	public boolean removeCacheAttribute(String sid, String key) {
		if ((StringUtils.isNotBlank(sid)) && (StringUtils.isNotBlank(key))) {
			long count = this.jedisTemplate.hdel(sid, new String[] { key }).longValue();
			if (count > 0L) {
				return true;
			}
		}
		return false;
	}

	public <T extends Serializable> T getUserCache(String key) {
		boolean isExist = exists(key).booleanValue();
		T result = null;
		if (isExist) {
			expire(key, this.sessionTimeout);
			result = get(key);
		}
		return result;
	}

	public <T extends Serializable> T getCurUser(String prefix, String uname) {
		return (T) (StringUtils.isNotBlank(uname) ? getUserCache(createUserCacheKey(prefix, uname)) : null);
	}

	public <T extends Serializable> void cacheUser(String prefix, String uname, T user) {
		putUserCache(createUserCacheKey(prefix, uname), user);
	}

	public <T extends Serializable> void cacheUser(String uname, T user) {
		putUserCache(createUserCacheKey(uname), user);
	}

	public <T extends Serializable> void putUserCache(String key, T value) {
		putTimedCache(key, value, this.sessionTimeout);
	}

	private String createUserCacheKey(String prefix, String uname) {
		return new StringBuilder().append("user.info.login").append(prefix).toString() + ":" + uname;
	}

	private String createUserCacheKey(String uname) {
		return "user.info.login.tenant" + ":" + uname;
	}

	public <T extends Serializable> void disCacheUser(String uname) {
		removeCache(createUserCacheKey(uname));
	}

	public <T extends Serializable> void disCacheUser(String prefix, String uname) {
		removeCache(createUserCacheKey(prefix, uname));
	}

	public String getSeedValue(String key) {
		return this.jedisTemplate.get(key);
	}

	public <T extends Serializable> void putTimedCache(String key, T value, int timeout) {
		setex(key, value, timeout);
	}

	public <T extends Serializable> void set(final String key, final T value) {
		execute(new JedisActionNoResult() {
			public void action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				byte[] valueBytes = SerializUtil.objectToByte(value);
				jedis.set(keyBytes, valueBytes);
			}
		});
	}

	public <T extends Serializable> void setex(final String key, final T value, final int timeout) {
		execute(new JedisActionNoResult() {
			public void action(Jedis jedis) {
				byte[] valueBytes = SerializUtil.objectToByte(value);
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				jedis.setex(keyBytes, timeout, valueBytes);
			}
		});
	}

	public void expire(final String key, final int timeout) {
		execute(new JedisActionNoResult() {
			public void action(Jedis jedis) {
				jedis.expire(key, timeout);
			}
		});
	}

	public Boolean exists(final String key) {
		return (Boolean) execute(new JedisAction() {
			public Boolean action(Jedis jedis) {
				return jedis.exists(key);
			}
		});
	}

	public <T extends Serializable> T get(final String key) {
		return (T) execute(new JedisAction() {
			public T action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				if (keyBytes == null) {
					return null;
				}
				byte[] valueBytes = jedis.get(keyBytes);
				if (valueBytes == null) {
					return null;
				}
				return (T) SerializUtil.byteToObject(valueBytes);
			}
		});
	}

	public <T extends Serializable> T hget(final String key, final String fieldName) {
		return (T) execute(new JedisAction() {
			public T action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				byte[] fieldBytes = fieldName.getBytes(Charset.forName("UTF-8"));
				byte[] attrBytes = jedis.hget(keyBytes, fieldBytes);
				if (attrBytes == null) {
					return null;
				}
				return (T) SerializUtil.byteToObject(attrBytes);
			}
		});
	}

	private <T extends Serializable> T hgetByKryo(final String key, final String fieldName) {
		return (T) execute(new JedisAction() {
			public T action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				byte[] fieldBytes = fieldName.getBytes(Charset.forName("UTF-8"));
				byte[] attrBytes = jedis.hget(keyBytes, fieldBytes);
				if (attrBytes == null) {
					return null;
				}
				return (T) SerializUtil.byteToObjectByKryo(attrBytes);
			}
		});
	}

	public Map<byte[], byte[]> hgetAll(final String key) {
		return (Map) execute(new JedisAction() {
			public Map<byte[], byte[]> action(Jedis jedis) {
				return jedis.hgetAll(key.getBytes(Charset.forName("UTF-8")));
			}
		});
	}

	public <T extends Serializable> void hset(final String key, final String fieldName, final T value) {
		execute(new JedisActionNoResult() {
			public void action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				byte[] fieldBytes = fieldName.getBytes(Charset.forName("UTF-8"));
				byte[] valueBytes = SerializUtil.objectToByte(value);
				jedis.hset(keyBytes, fieldBytes, valueBytes);
			}
		});
	}

	private <T extends Serializable> void hsetByKryo(final String key, final String fieldName, final T value) {
		execute(new JedisActionNoResult() {
			public void action(Jedis jedis) {
				byte[] keyBytes = key.getBytes(Charset.forName("UTF-8"));
				byte[] fieldBytes = fieldName.getBytes(Charset.forName("UTF-8"));
				byte[] valueBytes = SerializUtil.objectToByteByKryo(value);
				jedis.hset(keyBytes, fieldBytes, valueBytes);
			}
		});
	}

	public void removeCache(String key) {
		if (StringUtils.isNotBlank(key))
			this.jedisTemplate.del(new String[] { key });
	}

	public boolean remove(String key) {
		if (StringUtils.isNotBlank(key)) {
			return this.jedisTemplate.del(new String[] { key }).booleanValue();
		}
		return false;
	}

	public <T> T execute(JedisAction<T> jedisAction) throws JedisException {
		PerformanceLoggerCollector.start("redis");

		Jedis jedis = null;
		boolean broken = false;
		T result = null;
		try {
			jedis = (Jedis) this.jedisTemplate.getJedisPool().getResource();
			result = jedisAction.action(jedis);
		} catch (JedisException e) {
			broken = handleJedisException(e);
			throw e;
		} finally {
			closeResource(jedis, broken);
			PerformanceLoggerCollector.stop("redis");
		}

		return result;
	}

	public void execute(JedisActionNoResult jedisAction) throws JedisException {
		PerformanceLoggerCollector.start("redis");
		Jedis jedis = null;
		boolean broken = false;
		try {
			jedis = (Jedis) this.jedisTemplate.getJedisPool().getResource();
			jedisAction.action(jedis);
		} catch (JedisException e) {
			broken = handleJedisException(e);
			throw e;
		} finally {
			closeResource(jedis, broken);
			PerformanceLoggerCollector.stop("redis");
		}
	}

	protected boolean handleJedisException(JedisException jedisException) {
		if ((jedisException instanceof JedisConnectionException))
			this.logger.error("Redis connection " + this.jedisTemplate.getJedisPool().getAddress() + " lost.",
					jedisException);
		else if ((jedisException instanceof JedisDataException)) {
			if ((jedisException.getMessage() != null) && (jedisException.getMessage().indexOf("READONLY") != -1))
				this.logger.error(
						"Redis connection " + this.jedisTemplate.getJedisPool().getAddress() + " are read-only slave.",
						jedisException);
			else
				return false;
		} else {
			this.logger.error("Jedis exception happen.", jedisException);
		}
		return true;
	}

	protected void closeResource(Jedis jedis, boolean conectionBroken) {
		try {
			if (conectionBroken)
				this.jedisTemplate.getJedisPool().returnBrokenResource(jedis);
			else
				this.jedisTemplate.getJedisPool().returnResource(jedis);
		} catch (Exception e) {
			this.logger.error("return back jedis failed, will fore close the jedis.", e);
			JedisUtils.destroyJedis(jedis);
		}
	}

	public static abstract interface JedisActionNoResult {
		public abstract void action(Jedis paramJedis);
	}

	public static abstract interface JedisAction<T> {
		public abstract T action(Jedis paramJedis);
	}

	public long getTTL(final String key) {
		return ((Long) execute(new JedisAction() {
			public Long action(Jedis jedis) {
				return jedis.ttl(key);
			}
		})).longValue();
	}
}