package uap.web.utils;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.net.ssl.SSLContext;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.FormBodyPart;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;

class AnyTrustStrategy implements TrustStrategy {
	@Override
	public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
		return true;
	}
}

public class HttpUtil {

	private static final Log log = LogFactory.getLog(HttpUtil.class);
	
	/**
	 * 默认代理名称
	 */
	public static final String DEFAULT_AGENT = "iuap.httpclient";

	private static int bufferSize = 1024;

	private static volatile HttpUtil instance;

	private ConnectionConfig connConfig;

	private SocketConfig socketConfig;

	private ConnectionSocketFactory plainSF;

	private KeyStore trustStore;

	private SSLContext sslContext;

	private LayeredConnectionSocketFactory sslSF;

	private Registry<ConnectionSocketFactory> registry;

	private PoolingHttpClientConnectionManager connManager;

	private CloseableHttpClient client;

	// private static final String PARAM_SYSID = "u_sysid" ;
	// private static final String PARAM_TENANTID="tenantid";
	// private static final String PARAM_USERCODE="u_usercode";
	// private static final String PARAM_CALLID="u_callid";
	// private static final String PARAM_LOCALE="u_locale";
	// private static final String PARAM_TOKEN="token" ;
	// private static final String PARAM_THEME="u_theme" ;

	// private volatile BasicCookieStore cookieStore;

	public static String defaultEncoding = "utf-8";

	private static List<NameValuePair> paramsConverter(Map<String, String> params) {
		List<NameValuePair> nvps = new LinkedList<NameValuePair>();
		Set<Entry<String, String>> paramsSet = params.entrySet();
		for (Entry<String, String> paramEntry : paramsSet) {
			nvps.add(new BasicNameValuePair(paramEntry.getKey(), paramEntry.getValue()));
		}
		return nvps;
	}

	/**
	 * Deprecated, use {@link EntityUtils}.toString({@link HttpEntity}.getEntity) instead.
	 * 
	 * @param in
	 * @param encoding
	 * @return
	 */
	@Deprecated
	public static String readStream(InputStream in, String encoding) {
		if (in == null) {
			return null;
		}
		try {
			InputStreamReader inReader = null;
			if (encoding == null) {
				inReader = new InputStreamReader(in, defaultEncoding);
			} else {
				inReader = new InputStreamReader(in, encoding);
			}
			char[] buffer = new char[bufferSize];
			int readLen = 0;
			StringBuffer sb = new StringBuffer();
			while ((readLen = inReader.read(buffer)) != -1) {
				sb.append(buffer, 0, readLen);
			}
			inReader.close();
			return sb.toString();
		} catch (IOException e) {
			log.error("读取返回内容出错", e);
		} finally {
			if (null != in) {
				try {
					in.close();
				} catch (Exception e) {
				}
			}
		}
		return null;
	}

	private HttpUtil() {
		// 设置连接参数
		connConfig = ConnectionConfig.custom().setCharset(Charset.forName(defaultEncoding)).build();
		socketConfig = SocketConfig.custom().setSoTimeout(10000).build();
		RegistryBuilder<ConnectionSocketFactory> registryBuilder = RegistryBuilder.<ConnectionSocketFactory> create();
		plainSF = new PlainConnectionSocketFactory();
		registryBuilder.register("http", plainSF);
		// 指定信任密钥存储对象和连接套接字工厂
		try {
			trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
			sslContext = SSLContexts.custom().useTLS().loadTrustMaterial(trustStore, new AnyTrustStrategy()).build();
			sslSF = new SSLConnectionSocketFactory(sslContext, SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
			registryBuilder.register("https", sslSF);
		} catch (KeyStoreException e) {
			throw new RuntimeException(e);
		} catch (KeyManagementException e) {
			throw new RuntimeException(e);
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}
		registry = registryBuilder.build();
		// 设置连接管理器
		connManager = new PoolingHttpClientConnectionManager(registry);
		connManager.setDefaultConnectionConfig(connConfig);
		connManager.setDefaultSocketConfig(socketConfig);
		connManager.setDefaultMaxPerRoute(200);
		connManager.setMaxTotal(200);
		// 指定cookie存储对象
		// cookieStore = new BasicCookieStore();
		// 构建客户端
		// client= HttpClientBuilder.create().setDefaultCookieStore(cookieStore).setConnectionManager(connManager).build();
		client = HttpClientBuilder.create().setConnectionManager(connManager)
				.setDefaultRequestConfig(RequestConfig.custom().setConnectionRequestTimeout(2000).setConnectTimeout(3000).setSocketTimeout(5000).build()).setUserAgent(DEFAULT_AGENT).build();
	}

	public static HttpUtil getInstance() {
		if (HttpUtil.instance == null) {
			synchronized (HttpUtil.class) {
				if (HttpUtil.instance == null) {
					instance = new HttpUtil();
				}
			}
		}
		return instance;
	}

	public InputStream doGet(String url) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, null);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doGetForString(String url) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, null);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	public InputStream doGetForStream(String url, Map<String, String> queryParams) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, queryParams);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doGetForString(String url, Map<String, String> queryParams) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, queryParams);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	public InputStream doGetForStream(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, queryParams, headers);
		return response != null ? response.getEntity().getContent() : null;
	}

	public InputStream doDeleteForStream(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doDelete(url, queryParams, headers);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doGetForString(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doGet(url, queryParams, headers);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	public String doDeleteForString(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doDelete(url, queryParams, headers);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	/**
	 * 基本的Get请求
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @return HttpResponse
	 * @throws URISyntaxException
	 * @throws IOException
	 * @throws ClientProtocolException
	 */
	public HttpResponse doGet(String url, Map<String, String> queryParams) throws URISyntaxException, ClientProtocolException, IOException {
		return doGet(url, queryParams, null);
	}

	/**
	 * 加入header的Get请求,在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @param headers
	 *            http header
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse doGet(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpGet gm = new HttpGet();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		gm.setURI(builder.build());
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				gm.addHeader(name, headers.get(name));
			}
		}
		return client.execute(gm);
	}

	/**
	 * 在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 * @param queryParams
	 * @param headers
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse doDelete(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpDelete hd = new HttpDelete();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		hd.setURI(builder.build());
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				hd.addHeader(name, headers.get(name));
			}
		}
		return client.execute(hd);
	}

	/**
	 * 在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 * @param queryParams
	 * @param headers
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse doPut(String url, Map<String, String> queryParams, Map<String, String> headers) throws URISyntaxException, ClientProtocolException, IOException {
		HttpPut hp = new HttpPut();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		hp.setURI(builder.build());
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				hp.addHeader(name, headers.get(name));
			}
		}
		return client.execute(hp);
	}

	public InputStream doPostForStream(String url, Map<String, String> queryParams) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, null);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doPostForString(String url, Map<String, String> queryParams) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, null);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	public InputStream doPostForStream(String url, Map<String, String> queryParams, Map<String, String> formParams)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, formParams);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doPostRetString(String url, Map<String, String> queryParams, Map<String, String> formParams) throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, formParams);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	public InputStream doPostForStream(String url, Map<String, String> queryParams, Map<String, String> formParams, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, formParams, headers);
		return response != null ? response.getEntity().getContent() : null;
	}

	public String doPostRetString(String url, Map<String, String> queryParams, Map<String, String> formParams, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpResponse response = this.doPost(url, queryParams, formParams, headers);
		String ret = HttpUtil.readStream(response != null ? response.getEntity().getContent() : null, null);
		HttpUtil.releaseQuietly(response);
		return ret;
	}

	/**
	 * 基本的Post请求
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @param formParams
	 *            post表单的参数
	 * @return HttpResponse
	 * @throws URISyntaxException
	 * @throws IOException
	 * @throws ClientProtocolException
	 */
	public HttpResponse doPost(String url, Map<String, String> queryParams, Map<String, String> formParams) throws URISyntaxException, ClientProtocolException, IOException {
		return doPost(url, queryParams, formParams, null);
	}

	/**
	 * 加入header的Post请求, 在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @param formParams
	 *            post表单的参数
	 * @param headers
	 *            http header
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse doPost(String url, Map<String, String> queryParams, Map<String, String> formParams, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpPost pm = new HttpPost();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		pm.setURI(builder.build());
		// 填入表单参数
		if (formParams != null && !formParams.isEmpty()) {
			pm.setEntity(new UrlEncodedFormEntity(HttpUtil.paramsConverter(formParams), defaultEncoding));
		}
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				pm.addHeader(name, headers.get(name));
			}
		}
		return client.execute(pm);
	}
	
	public HttpResponse doPostForString(String url, Map<String, String> queryParams, String requestBody, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpPost pm = new HttpPost();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		pm.setURI(builder.build());
		// 填入requestbody
		if (requestBody != null && !requestBody.isEmpty()) {
			pm.setEntity(new StringEntity(requestBody, defaultEncoding));
		}
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				pm.addHeader(name, headers.get(name));
			}
		}
		return client.execute(pm);
	}

	/**
	 * 在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 * @param queryParams
	 * @param formParams
	 * @param headers
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse doPut(String url, Map<String, String> queryParams, Map<String, String> formParams, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpPut pm = new HttpPut();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		pm.setURI(builder.build());
		// 填入表单参数
		if (formParams != null && !formParams.isEmpty()) {
			pm.setEntity(new UrlEncodedFormEntity(HttpUtil.paramsConverter(formParams), defaultEncoding));
		}
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				pm.addHeader(name, headers.get(name));
			}
		}
		return client.execute(pm);
	}

	/**
	 * 多块Post请求
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @param formParts
	 *            post表单的参数,支持字符串-文件(FilePart)和字符串-字符串(StringPart)形式的参数
	 * @param maxCount
	 *            最多尝试请求的次数
	 * @return HttpResponse
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws HttpException
	 * @throws IOException
	 */
	public HttpResponse multipartPost(String url, Map<String, String> queryParams, List<FormBodyPart> formParts) throws URISyntaxException, ClientProtocolException, IOException {
		return multipartPost(url, queryParams, formParts, null);
	}

	/**
	 * 多块Post请求，加入了header, 在HttpResponse使用完毕后务必调用在finally中调用com.yonyou.iuap.utils.HttpUtil.releaseQuietly(HttpResponse)释放response所占连接, 否则会导致连接不释放而阻塞调用线程.
	 * 
	 * @param url
	 *            请求url
	 * @param queryParams
	 *            请求头的查询参数
	 * @param formParts
	 *            post表单的参数,支持字符串-文件(FilePart)和字符串-字符串(StringPart)形式的参数
	 * @param headers
	 *            http header
	 * @return
	 * @throws URISyntaxException
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public HttpResponse multipartPost(String url, Map<String, String> queryParams, List<FormBodyPart> formParts, Map<String, String> headers)
			throws URISyntaxException, ClientProtocolException, IOException {
		HttpPost pm = new HttpPost();
		URIBuilder builder = new URIBuilder(url);
		// 填入查询参数
		if (queryParams != null && !queryParams.isEmpty()) {
			builder.setParameters(HttpUtil.paramsConverter(queryParams));
		}
		pm.setURI(builder.build());
		// 填入表单参数
		if (formParts != null && !formParts.isEmpty()) {
			MultipartEntityBuilder entityBuilder = MultipartEntityBuilder.create();
			entityBuilder = entityBuilder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
			for (FormBodyPart formPart : formParts) {
				entityBuilder = entityBuilder.addPart(formPart.getName(), formPart.getBody());
			}
			pm.setEntity(entityBuilder.build());
		}
		// 填入http请求的header
		if (headers != null) {
			for (String name : headers.keySet()) {
				pm.addHeader(name, headers.get(name));
			}
		}
		return client.execute(pm);
	}

	/**
	 * 提供后端调用带有requestBody方法的POST服务
	 * 
	 * @param url 调用接口的地址
	 * @param bodyParams 请求接口的body参数
	 * @param headers http头信息
	 */
	public HttpResponse postFormForBody(String url, String bodyParams, List<Header> headers) {
		HttpPost httpPost = null;
		CloseableHttpResponse httpResponse = null;

		try {
			httpPost = new HttpPost(url);

			if (CollectionUtils.isNotEmpty(headers)) {
				for (Header header : headers) {
					httpPost.addHeader(header);
				}
			}

			if (null != bodyParams) {
				httpPost.setEntity(new StringEntity(bodyParams, Charset.forName("UTF-8")));
			}

			httpResponse = client.execute(httpPost);

			return httpResponse;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * 提供后端调用带有requestBody方法的PUT服务
	 * 
	 * @param url 调用接口的地址
	 * 
	 * @param bodyParams 请求接口的body参数
	 * 
	 * @param headers http头信息
	 * 
	 * @return
	 */
	public HttpResponse putFormForBody(String url, String bodyParams, List<Header> headers) {
		HttpPut httpPut = null;
		CloseableHttpResponse httpResponse = null;
		
		try {
			httpPut = new HttpPut(url);
			if (CollectionUtils.isNotEmpty(headers)) {
				for (Header header : headers) {
					httpPut.addHeader(header);
				}
			}
			
			if(null!=bodyParams){
				httpPut.setEntity(new StringEntity(bodyParams, Charset.forName("UTF-8")));
			}
			
			httpResponse = client.execute(httpPut);
			return httpResponse;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} 
	}
	
	/**
	 * httpclient调用时候，加入上下文信息
	 */

	public static void releaseQuietly(HttpResponse response) {
		if (null != response) {
			EntityUtils.consumeQuietly(response.getEntity());
			if (response instanceof CloseableHttpResponse) {
				try {
					((CloseableHttpResponse) response).close();
				} catch (IOException e) {
				}
			}
		}
	}

}
