package com.yyjz.icop.base.dao.impl;

import java.io.Serializable;
import java.sql.Timestamp;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;

import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.query.QueryUtils;
import org.springframework.data.jpa.repository.support.JpaEntityInformation;
import org.springframework.data.jpa.repository.support.SimpleJpaRepository;
import org.springframework.transaction.annotation.Transactional;

import com.yyjz.icop.base.dao.BaseDao;
import com.yyjz.icop.base.tenant.TenantHelper;
import com.yyjz.icop.base.tenant.TenantSpecification;
import com.yyjz.icop.database.entity.SuperEntity;
/**
 * 继承SimpleJpaRepository，重写删除逻辑为逻辑删除
 * @author shl
 *
 * @param <T>
 */
@Transactional
public class BaseDaoImpl<T extends SuperEntity> extends SimpleJpaRepository<T, String> implements BaseDao<T> {

	private final EntityManager entityManager;
	
	private  JpaEntityInformation<T, ?> entityInformation;

	public BaseDaoImpl(Class<T> domainClass, EntityManager entityManager) {
		super(domainClass, entityManager);
		this.entityManager = entityManager;
	}

	public BaseDaoImpl(JpaEntityInformation<T, Serializable> information, EntityManager entityManager) {
		super(information, entityManager);
		this.entityManager = entityManager;
		this.entityInformation = information;
	}

	@Override
	public <S extends T> S save(S entity) {

		if(entity!=null){
			TenantHelper.setTenantIdToEntity(entity);
		}
		entity.setAttributeValue("modificationTimestamp", new Timestamp(System.currentTimeMillis()));
		return super.save(entity);
	}
	
	@Override
	public void delete(String id) {
		T entity = findOne(id);
		entity.setDr(1);
		this.save(entity);
	}

	/**
	 * 只做逻辑删除
	 */
	@Override
	public void delete(T entity) {
		entity.setDr(1);
		save(entity);
	}
	
	@Transactional
	@Override
	public void deleteInBatch(Iterable<T> entities) {
		if (!(entities.iterator().hasNext())) {
			return;
		}

		QueryUtils.applyAndBind(QueryUtils.getQueryString("update %s x set dr = 1 ", this.entityInformation.getEntityName()),
				entities, this.entityManager).executeUpdate();
	}

	@Transactional
	@Override
	public void deleteAll() {
		for (T t : findAll()) {
			t.setDr(1);
			save(t);
		}
	}
	
	@Override
	protected TypedQuery<T> getQuery(Specification<T> spec, Sort sort) {
		return super.getQuery(new TenantSpecification<T>(spec), sort);
	}
	@Override
	protected TypedQuery<Long> getCountQuery(Specification<T> spec){
		return super.getCountQuery(new TenantSpecification<T>(spec));
	}
	
	@Override
	public T getOne(String id) {
		return findOne(id);
	}
}
