package com.yyjz.icop.base.tenant;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.Path;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.persistence.metamodel.EntityType;

import com.yonyou.iuap.context.InvocationInfoProxy;

/**
 * 多租户辅助类
 * 
 * @author jiangsf
 *
 */
public class TenantHelper {

	private static HashMap<String, TenantMethodCacheItem> tenandIdCache = new HashMap<String, TenantMethodCacheItem>();

	// private static final String TENANT_ID_FIELD="TenantId";

	/**
	 * 获取上下文中的租户ID
	 * 
	 * @return 上下文中的租户ID
	 */
	public static String getTenantId() {
		return InvocationInfoProxy.getTenantid();
	}

	/**
	 * 获取租户id对应的Field
	 * 
	 * @param obj
	 * @return
	 */
	private static TenantMethodCacheItem getTeantIdMethod(Object obj) {
		if (obj == null) {
			return null;
		}
		Class<?> cls = obj.getClass();
		return getTeantIdMethodByClass(cls);
	}

	private static TenantMethodCacheItem getTeantIdMethodByClass(Class<?> cls) {
		String clsName = cls.getName();
		if (!tenandIdCache.containsKey(clsName)) {
			TenantMethodCacheItem item = new TenantMethodCacheItem();
			item.setReadMethod(getTenantMethod(cls, "get"));
			item.setWriteMethod(getTenantMethod(cls, "set",String.class));
			tenandIdCache.put(clsName, item);
		}
		return tenandIdCache.get(clsName);
	}

	private static Method getTenantMethod(Class<?> cls, String method,Class<?>... paramArrayOfMethod) {
		try {
			return cls.getMethod(method + "TenantId",paramArrayOfMethod);
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (SecurityException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 设置实体中的租户ID属性，如果实体租户ID有值则不进行设置，否则设置为上下文中的租户ID
	 * 
	 * @param entity
	 *            需要设置的实体
	 */
	public static void setTenantIdToEntity(Object entity) {
		TenantMethodCacheItem item = getTeantIdMethod(entity);
		if (item == null)
			return;
		Method setMethod = item.getWriteMethod();
		Method getMethod = item.getReadMethod();
		if (setMethod == null || getMethod == null)
			return;

		try {
			Object oldValue = getMethod.invoke(entity);
			if (oldValue == null || oldValue.toString() == "") {
				setMethod.invoke(entity, getTenantId());
			}
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 检查当前实体的租户ID和上下文中的租户ID是否一致，实体中没有租户id则认为是一致的
	 * 
	 * @param entity
	 *            需要检查的实体
	 * @return 一致返回true，否则返回false
	 */
	public static boolean checkEntityTeanantId(Object entity) {
		if (entity == null)
			return true;
		TenantMethodCacheItem method = getTeantIdMethod(entity);
		if (method.getReadMethod() == null)
			return true;
		Object tenantId = null;
		try {
			tenantId = method.getReadMethod().invoke(entity);
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}
		if (tenantId == null) {
			return false;
		}
		return tenantId == getTenantId();
	}

	/**
	 * 获取租户ID的查询条件
	 * 
	 * @param root
	 * @param cb
	 * @return 实体中存在租户ID则返回对应的条件，否则返回null
	 */
	public static Predicate getTenantIdPredicate(Root<?> root,
			CriteriaBuilder cb) {
		Predicate tenantPredicate = null;
		try {
			EntityType<?> model = root.getModel();
			Class<?> cls = model.getJavaType();
			TenantMethodCacheItem item = getTeantIdMethodByClass(cls);
			if (item.getReadMethod() != null && item.getWriteMethod() != null) {
				Path<Object> t = root.get("tenantId");
				if (t != null) {
					tenantPredicate = cb.equal(t, getTenantId());
				}
			}
		} catch (Exception e) {
			//
		}
		return tenantPredicate;
	}

	/**
	 * 租户ID缓存信息
	 * 
	 * @author jiangsf
	 *
	 */
	private static class TenantMethodCacheItem {
		/**
		 * 读方法
		 */
		private Method readMethod;
		/**
		 * 写方法
		 */
		private Method writeMethod;

		/**
		 * 获取读方法
		 * 
		 * @return
		 */
		public Method getReadMethod() {
			return readMethod;
		}

		/**
		 * 设置读方法
		 * 
		 * @param readMethod
		 */
		public void setReadMethod(Method readMethod) {
			this.readMethod = readMethod;
		}

		/**
		 * 获取写方法
		 * 
		 * @return
		 */
		public Method getWriteMethod() {
			return writeMethod;
		}

		/**
		 * 设置写方法
		 * 
		 * @param writeMethod
		 */
		public void setWriteMethod(Method writeMethod) {
			this.writeMethod = writeMethod;
		}

	}
}
