package com.yyjz.icop.orgcenter.company.service.base;

import java.util.List;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.yyjz.icop.base.exception.BusinessException;
import com.yyjz.icop.orgcenter.company.vo.orgunit.OrgUnitVO;

/**
 * 基本查询服务接口
 *
 * @param <T> 继承自 {@code OrgUnitVO} 的实体
 * @author GavinWood
 * @date 2018年7月25日 上午10:33:21
 * @JDK jdk1.8
 */
public interface BaseQryService<V extends OrgUnitVO> {

    /**
     * 根据主键信息查询实体
     *
     * @param id       实体主键
     * @param tenantId 租户标识
     * @return 实体信息{@code V}
     * @throws BusinessException
     */
	V getById(String id, String tenantId) throws BusinessException;

    /**
     * 根据主键和dr查询
     *
     * @param id
     * @param tenantId
     * @param dr
     * @return 实体信息{@code V}，查询不到则返回null
     * @throws BusinessException
     */
    V getByIdAndDr(String id, int dr, String tenantId) throws BusinessException;

    /**
     * 根据编码查询实体信息
     *
     * @param code     实体编码，对应字段为code
     * @param tenantId 租户标识
     * @return 实体信息{@code V}，查询不到则返回null
     * @throws BusinessException
     */
    V getByCode(String code, String tenantId) throws BusinessException;
    
    /**
     * 根据名称查询实体信息列表 
     *
     * @param name     实体名称，对应字段为name
     * @param tenantId 租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    List<V> listByName(String name, String tenantId) throws BusinessException;

    /**
     * 查询实体信息列表
     *
     * @param statusList 数据状态，三态，详见 {@code ICommonConst#ENABLE_STATUS_INIT}
     *                   {@code ICommonConst#ENABLE_STATUS_ENABLED},
     *                   {@code ICommonConst#ENABLE_STATUS_DISABLED}
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    List<V> listByStatus(List<Integer> statusList, String tenantId) throws BusinessException;
    
    /**
     * 查询实体信息列表
     *
     * @param statusList 数据状态，三态，详见 {@code ICommonConst#ENABLE_STATUS_INIT}
     *                   {@code ICommonConst#ENABLE_STATUS_ENABLED},
     *                   {@code ICommonConst#ENABLE_STATUS_DISABLED}
     * @param tenantId   租户标识
     * @param dr 删除标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
	List<V> listByStatusAndDr(List<Integer> statusList, int dr, String tenantId) throws BusinessException;

    /**
     * 根据主键批量查询
     *
     * @param ids        主键集合
     * @param statusList 数据状态，三态，详见 {@code ICommonConst#ENABLE_STATUS_INIT}
     *                   {@code ICommonConst#ENABLE_STATUS_ENABLED},
     *                   {@code ICommonConst#ENABLE_STATUS_DISABLED}
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
	List<V> listByIds(List<String> ids, List<Integer> statusList, String tenantId) throws BusinessException;

    /**
     * 根据主键和dr批量查询
     *
     * @param ids
     * @param statusList
     * @param tenantId
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
	List<V> listByIdsAndDr(List<String> ids, List<Integer> statusList, int dr, String tenantId) throws BusinessException;
	
    /**
     * 根据编码或者名称进行模糊查询 code对应字段名code，name对应字段名name
     *
     * @param keyword    查询关键字
     * @param statusList 数据状态，三态，详见 {@code ICommonConst#ENABLE_STATUS_INIT}
     *                   {@code ICommonConst#ENABLE_STATUS_ENABLED},
     *                   {@code ICommonConst#ENABLE_STATUS_DISABLED}
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    List<V> listByCodeOrNameLike(String keyword, List<Integer> statusList, String tenantId) throws BusinessException;
    
    /**
     * 根据关联字段sourceId查询实体
     *
     * @param sourceId 关联字段，对应字段source_id
     * @param tenantId
     * @return 实体信息{@code V}，查询不到则返回null
     * @throws BusinessException
     */
    V getBySourceId(String sourceId, String tenantId) throws BusinessException;
    
    /**
     * 根据组织单元获取职能组织
     *
     * @param orgUnitId 组织单元主键
     * @param tenantId 租户标识
     * @return 实体信息{@code V}
     * @throws BusinessException
     */
    V getByOrgUnit(String orgUnitId, String tenantId) throws BusinessException;
    
    /**
     * 根据组织单元获取上级职能组织
     *
     * @param orgUnitId 组织单元主键
     * @param tenantId 租户标识
     * @return 实体信息{@code V}
     * @throws BusinessException
     */
    V getParentByOrgUnit(String orgUnitId, String tenantId) throws BusinessException;
    
    /**
     * 查询实体信息列表
     *
     * @param statusList 数据状态，三态，详见 {@code ICommonConst#ENABLE_STATUS_INIT}
     *                   {@code ICommonConst#ENABLE_STATUS_ENABLED},
     *                   {@code ICommonConst#ENABLE_STATUS_DISABLED}
     * @param searchText 搜索关键字（基于名称、编码等）
     * @param pid 上级组织
     * @param direction 查询方向
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    Page<V> page(String pid, Pageable pageable, String searchText, int direction, List<Integer> enabledStatus, String tenantId) throws BusinessException;
    
    /**
     * 获取职能树
     *
     * @param orgUnitId 所属组织单元id
     * @param keyword 搜索关键字（基于名称、编码等）
     * @param tenantId   租户标识
     * @return 职能树String
     * @throws BusinessException
     */
    String orgTree(String orgUnitId, String keyword, String tenantId) throws BusinessException;
}
