package com.yyjz.icop.orgcenter.company.service.orgunit;

import java.util.List;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.yyjz.icop.base.exception.BusinessException;
import com.yyjz.icop.base.vo.SuperVO;
import com.yyjz.icop.orgcenter.company.vo.orgunit.OrgUnitVO;

/**
 * 基本组织单元查询服务接口
 *
 * @param <T> 继承自 {@code OrgUnitVO} 的实体
 * @author GavinWood
 * @date 2018年7月25日 上午10:33:21
 * @JDK jdk1.8
 */
public interface OrgUnitQryService<V extends OrgUnitVO<SuperVO>> {

    /**
     * 根据主键信息查询组织单元信息
     *
     * @param id       组织单元主键
     * @param tenantId 租户标识
     * @return 实体信息{@code V}
     * @throws BusinessException
     */
	V getById(String id, String tenantId) throws BusinessException;

    /**
     * 根据编码查询组织单元信息
     *
     * @param code     实体编码，对应字段为code
     * @param tenantId 租户标识
     * @return 实体信息{@code V}，查询不到则返回null
     * @throws BusinessException
     */
    V getByCode(String code, String tenantId) throws BusinessException;
    
    /**
     * 根据名称查询组织单元信息列表 
     *
     * @param name     实体名称，对应字段为name
     * @param tenantId 租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    List<V> listByName(String name, String tenantId) throws BusinessException;

    /**
     * 根据主键批量查询
     *
     * @param ids        主键集合
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
	List<V> listByIds(List<String> ids, String tenantId) throws BusinessException;

    /**
     * 根据编码或者名称进行模糊查询 code对应字段名code，name对应字段名name
     *
     * @param keyword    查询关键字
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    List<V> listByCodeOrNameLike(String keyword, String tenantId) throws BusinessException;
    
    /**
     * 根据关联字段sourceId查询实体
     *
     * @param sourceId 关联字段，对应字段source_id
     * @param tenantId
     * @return 实体信息{@code V}，查询不到则返回null
     * @throws BusinessException
     */
    V getBySourceId(String sourceId, String tenantId) throws BusinessException;
    
    /**
     * 分页查询组织单元列表 
     *
     * @param searchText 搜索关键字（基于名称、编码等）
     * @param tenantId   租户标识
     * @return 实体信息{@code List<V>}，查询不到则返回空列表 List.size() == 0
     * @throws BusinessException
     */
    Page<V> page(Pageable pageable, String searchText, String tenantId) throws BusinessException;
}
