package com.ejianc.business.rmat.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.rmat.bean.CalculateDailyEntity;
import com.ejianc.business.rmat.bean.CalculateEntity;
import com.ejianc.business.rmat.consts.BillTypeEnum;
import com.ejianc.business.rmat.consts.MaterialConstant;
import com.ejianc.business.rmat.consts.MaterialStateEnum;
import com.ejianc.business.rmat.mapper.CalculateMapper;
import com.ejianc.business.rmat.service.ICalculateService;
import com.ejianc.business.rmat.service.IMaterialService;
import com.ejianc.business.rmat.util.DateUtil;
import com.ejianc.business.rmat.util.ValidateUtil;
import com.ejianc.business.rmat.vo.CalculateDailyVO;
import com.ejianc.business.rmat.vo.CalculateVO;
import com.ejianc.business.rmat.vo.MaterialFlowVO;
import com.ejianc.business.rmat.vo.MaterialVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;

/**
 * 租金计算单
 * 
 * @author generator
 * 
 */
@Service("calculateService")
public class CalculateServiceImpl extends BaseServiceImpl<CalculateMapper, CalculateEntity> implements ICalculateService{
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IMaterialService materialService;

//    @Autowired
//    private IRestituteService restituteService;

    private static final String BILL_CODE = "CSCEC_RMAT_CALCULATE";//此处需要根据实际修改
    private static final String BILL_NAME = BillTypeEnum.租金计算单.getName();

    @Override
    public CalculateVO saveOrUpdate(CalculateVO saveOrUpdateVO) {
        // 同一个合同只能存在一个自由态或审批中的单据
        materialService.validateContract(saveOrUpdateVO.getContractId(), BILL_NAME, saveOrUpdateVO.getId(), MaterialConstant.保存);
        // 校验必须大于最大单据日期
        this.validateTime(saveOrUpdateVO, MaterialConstant.保存);
        CalculateEntity entity = BeanMapper.map(saveOrUpdateVO, CalculateEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE,InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        super.saveOrUpdate(entity, false);
        return BeanMapper.map(entity, CalculateVO.class);
    }

    @Override
    public String validateTime(CalculateVO vo, String type) {
        Map<String, Object> params = new HashMap<>();
        params.put("contractId", vo.getContractId());
        if(vo.getId() != null){
            params.put("billType", BILL_NAME);
            params.put("billId", vo.getId());
        }
        Date lastDate = materialService.getLastDate(params);
        Map<Date, Date> maxTimeMap = materialService.getMaxTime(params);

        if (lastDate == null) return "未获取最大单据日期！";
        if(ValidateUtil.compareDate(vo.getRentDate(), lastDate, maxTimeMap, vo.getCreateTime())){
            throw new BusinessException(DateUtil.formatDate(vo.getRentDate())
                    + "小于最大单据日期【" + DateUtil.formatDate(lastDate) + "】，不允许" + type + "!");
        }
        return "校验通过！";
    }

    @Override
    public CalculateVO autoCalculate(CalculateVO vo) {
        if (vo == null || vo.getContractId() == null) {
            throw new BusinessException("合同信息不能为空");
        }
        if (vo.getRentDate() == null) {
            throw new BusinessException("租金计算日期不能为空");
        }
        String contractId = vo.getContractId(); // 合同id
        Date endDate = DateUtil.concatDate(vo.getRentDate(), new Date()); // 本次租金计算日期
        logger.info("开始租金自动计算，合同id：{}，租金计算日期：{}。>>>>>>>>>>>>>>>>>>>>>>>>>>", contractId, endDate);
        QueryParam param = new QueryParam();
        param.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        param.getParams().put("billState", new Parameter(QueryParam.NOT_IN, "1,3"));
        if (null != vo.getId()) {
            param.getParams().put("id", new Parameter(QueryParam.NE, vo.getId()));
        }
        List<CalculateEntity> checkList = super.queryList(param, false);
        if (CollectionUtils.isNotEmpty(checkList)) {
            throw new BusinessException("当前合同存在未生效的租金计算单");
        }
        // 最新租金计算单
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("rentDate", QueryParam.DESC);
        List<CalculateEntity> lastList = super.queryList(param, false);
        Date startDate = null;
        if(CollectionUtils.isNotEmpty(lastList)){
            CalculateEntity last = lastList.get(0);
            startDate = DateUtil.concatDate(DateUtil.dayAddOne(last.getRentDate()), last.getCreateTime());
        }
        /*
         * 1. 查询当前合同下物料台账信息
         */
        logger.info(">>>>查询已入场物料信息start");
        List<MaterialVO> reportList = materialService.queryCalculateList(contractId, startDate, endDate);
        if (CollectionUtils.isEmpty(reportList)) throw new BusinessException("未查询到当前合同下物料入场信息");
        logger.info("<<<<查询已入场台账信息end，查询结果：{}", JSONObject.toJSONString(reportList));
        // 3. 根据租赁方式及变更日期，对操作日期再次进行排序
        logger.info(">>>>对查询数据进行操作");
        CalculateEntity entity = new CalculateEntity();
        List<CalculateDailyEntity> dailyList = new ArrayList<>();
        List<CalculateDailyVO> dailyVOList = new ArrayList<>();
        // 循环比对日期
        for (MaterialVO materialVO : reportList) {
            this.packParameterDetailAndContract(dailyVOList, startDate, endDate, materialVO);
        }
        this.setRentalDayDetailValue(dailyVOList, dailyList);
        // 4. 处理租金计算单
        entity.setDailyList(dailyList);
        this.setRentalValue(entity);
        logger.info(">>>>租金自动计算结束，输出结果：{}", JSONObject.toJSONString(entity));
        return BeanMapper.map(entity, CalculateVO.class);
    }

    /**
     * 日租明细赋值
     *
     * @param resultList        日租组装数据
     * @param dailyList 日租明细结果
     */
    private void setRentalDayDetailValue(List<CalculateDailyVO> resultList, List<CalculateDailyEntity> dailyList) {
        if (CollectionUtils.isEmpty(resultList)) return;
        for (CalculateDailyVO vo : resultList) {
            CalculateDailyEntity entity = new CalculateDailyEntity();
//            entity.setSourceId(vo.getSourceId()); // 来源合同明细主键
            entity.setMaterialTypeId(vo.getMaterialTypeId());
            entity.setMaterialTypeName(vo.getMaterialTypeName());
            entity.setMaterialId(vo.getMaterialId());
            entity.setMaterialCode(vo.getMaterialCode());
            entity.setMaterialName(vo.getMaterialName());
            entity.setSpec(vo.getSpec());
            entity.setUnitId(vo.getUnitId());
            entity.setUnitName(vo.getUnitName());
            entity.setUnitTaxPrice(vo.getUnitTaxPrice());
            entity.setUnitPrice(vo.getUnitPrice());
            entity.setStopUnitTaxPrice(vo.getStopUnitTaxPrice());
            entity.setStopUnitPrice(vo.getStopUnitPrice());
            entity.setTaxRate(vo.getTaxRate());

            entity.setStartDate(vo.getStartDate());
            entity.setEndDate(vo.getEndDate());
            entity.setNum(vo.getNum()); // 修改数量需要重新计算下面的金额
            entity.setUseStatus(String.valueOf(vo.getMaterialState() - 2));
            entity.setRentDayDate(DateUtil.getSubDay(vo.getEndDate(), vo.getStartDate())); // 租赁天数
            if (MaterialStateEnum.启用.getCode().equals(vo.getMaterialState())) {// 启用
                entity.setDailyRentMny(this.getMny(entity.getUnitPrice(), entity.getRentDayDate(), entity.getNum())); // 租赁金额(无税)
                entity.setDailyRentTaxMny(this.getMny(entity.getUnitTaxPrice(), entity.getRentDayDate(), entity.getNum())); // // 租赁金额
            } else if (MaterialStateEnum.停用.getCode().equals(vo.getMaterialState())) {// 停用
                entity.setDailyRentMny(this.getMny(entity.getStopUnitPrice(), entity.getRentDayDate(), entity.getNum())); // 租赁金额(无税)
                entity.setDailyRentTaxMny(this.getMny(entity.getStopUnitTaxPrice(), entity.getRentDayDate(), entity.getNum())); // // 租赁金额
            }
            entity.setDailyTax(ComputeUtil.safeSub(entity.getDailyRentTaxMny(), entity.getDailyRentMny())); // 租赁税额
            entity.setId(IdWorker.getId());
            entity.setRowState("add");
            dailyList.add(entity);
        }
        // 按照物料主键、使用状态、开始时间排序
        if (CollectionUtils.isNotEmpty(dailyList)) {
            dailyList.sort(Comparator.comparing(CalculateDailyEntity::getMaterialId)
                    .thenComparing(CalculateDailyEntity::getStartDate));
        }
    }

    /**
     * 租金计算主单据赋默认值
     *
     * @param entity         租金计算单据
     */
    private void setRentalValue(CalculateEntity entity) {
        BigDecimal rentMny = BigDecimal.ZERO; // 租赁金额无税
        BigDecimal rentTaxMny = BigDecimal.ZERO;
        BigDecimal rentTax = BigDecimal.ZERO;
        if (CollectionUtils.isNotEmpty(entity.getDailyList())) {
            for (CalculateDailyEntity detail : entity.getDailyList()) {
                rentMny = ComputeUtil.safeAdd(rentMny, detail.getDailyRentMny());
                rentTaxMny = ComputeUtil.safeAdd(rentTaxMny, detail.getDailyRentTaxMny());
                rentTax = ComputeUtil.safeAdd(rentTax, detail.getDailyTax());
            }
        }
        entity.setRentMny(rentMny);
        entity.setRentTaxMny(rentTaxMny);
        entity.setRentTax(rentTax);
    }

    /**
     * 组装物料台账记录明细及合同变更明细
     *
     * @param resultList      组装后结果
     * @param lastDate  上一次计租日期，可能为空
     * @param rentDate        本次计租日期
     * @param materialVO     物料台账
     */
    private void packParameterDetailAndContract(List<CalculateDailyVO> resultList, Date lastDate, Date rentDate, MaterialVO materialVO) {
        // 获取物料操作记录
        List<MaterialFlowVO> flowList = materialVO.getFlowList();
        List<CalculateDailyVO> result = new ArrayList<>();
        // 计算期初
        List<MaterialFlowVO> beginList = materialVO.getBeginList();
        // 验收单模拟期初数据，没有期初则下一操作必为验收
        if(CollectionUtils.isEmpty(beginList)){
            MaterialFlowVO start = BeanMapper.map(flowList.get(0), MaterialFlowVO.class);
            start.setNum(BigDecimal.ZERO);
            start.setStartedNum(BigDecimal.ZERO);
            start.setStopedNum(flowList.get(0).getNum());
            beginList = new ArrayList<>(Arrays.asList(start));
            MaterialFlowVO stop = BeanMapper.map(start, MaterialFlowVO.class);
            stop.setMaterialState(MaterialStateEnum.停用.getCode());
            stop.setUseStatus(MaterialConstant.NO);
            beginList.add(stop);
        }
        // 生成期初数据
        for (MaterialFlowVO begin : beginList) {
            this.dealFlowList(materialVO, lastDate, rentDate, begin, CollectionUtils.isNotEmpty(flowList) ? flowList.get(0) : null, result, null);
        }
        //流水拆分
        if(CollectionUtils.isNotEmpty(flowList)){
            // 按照操作日期升序排序
            flowList.sort(Comparator.comparing(MaterialFlowVO::getOperationDate));
            for (MaterialFlowVO begin : beginList) {
                for (int i = 0; i < flowList.size(); i++) {
                    MaterialFlowVO detailVO = flowList.get(i);
                    MaterialFlowVO detailVO1 = i == flowList.size() - 1 ? null : flowList.get(i + 1);
                    this.dealFlowList(materialVO, lastDate, rentDate, detailVO, detailVO1, result, MaterialConstant.YES.equals(begin.getUseStatus()));
                }
            }
        }
        result.sort(Comparator.comparing(CalculateDailyVO::getStartDate));
        resultList.addAll(result);
        logger.info("组装结果：{}", JSONObject.toJSONString(resultList));
    }

    /**
     * 按操作日期和操作数量拆分
     * @param materialVO
     * @param lastDate 上一次计租日期，可能为空
     * @param rentDate   本次计租日期
     * @param detailVO
     * @param detailVO1
     * @param result
     * @param beginFlag
     * 关键字段注释 A-本次操作日期，B-下次操作日期（本次操作结束日期），T1-合同拆分后开始日期（合同签订日期/合同变更日期），T2-合同拆分后结束日期（合同变更日期 - 1/计算结束日期）
     */
    private void dealFlowList(MaterialVO materialVO, Date lastDate, Date rentDate, MaterialFlowVO detailVO, MaterialFlowVO detailVO1,
                              List<CalculateDailyVO> result, Boolean beginFlag) {
        // 给下一vo赋值期初数量
        if(detailVO1 != null) {
            this.setBeginNum(detailVO, detailVO1);
        }
        // 根据操作日期和数量拆分
        CalculateDailyVO vo = this.transferCacheVO(materialVO, detailVO, beginFlag);
        // 默认起始日期为本次操作日期，如果本次退场，则止租日期 + 1 作为分段起始日期
        Date startDate = MaterialStateEnum.退场.getCode().equals(detailVO.getMaterialState()) ?
                DateUtil.beginOfDate(DateUtil.dayAddOne(detailVO.getOperationDate())) : detailVO.getOperationDate();
        // 最后一条操作
        if(detailVO1 == null){
            if(lastDate == null){
                this.setValue(detailVO.getOperationDate(), rentDate, vo, result);// A-T2
            } else {
                // 操作日期大于区间开始日期 A > T1
                if(DateUtil.compareDate(detailVO.getOperationDate(), lastDate) > 0){
                    // 操作日期大于区间结束日期无效
                    if(DateUtil.compareDate(detailVO.getOperationDate(), rentDate) > 0){
                        return;
                    }
                    // 操作日期小于等于区间结束日期 T1 < A <= T2
                    this.setValue(startDate, rentDate, vo, result);// A-T2
                }
                // 操作日期小于等于区间开始日期 A <= T1
                else {
                    this.setValue(lastDate, rentDate, vo, result);// T1-T2
                }
            }
        }
        // 非最后一条其他操作
        else {
            // 退场止租日期计算当天
            Date endDate = MaterialStateEnum.退场.getCode().equals(detailVO1.getMaterialState()) ?
                    detailVO1.getOperationDate() : DateUtil.daySubOne(detailVO1.getOperationDate());
            if(lastDate == null){
                // 操作日期大于区间结束日期无效
                if(DateUtil.compareDate(detailVO.getOperationDate(), rentDate) > 0){
                    return;
                }
                // 操作日期小于等于区间结束日期 T1 < A <= T2
                // 下一操作日期大于区间结束日期 T1 < A <= T2 < B
                if(DateUtil.compareDate(detailVO1.getOperationDate(), rentDate) > 0){
                    this.setValue(startDate, rentDate, vo, result);// A-T2
                }
                // 下一操作日期小于等于区间结束日期 T1 < A < B <= T2
                else {
                    this.setValue(startDate, endDate, vo, result);// A-B-1，缺失B-T2
                }
            } else {
                // 操作日期大于区间开始日期 A > T1
                if(DateUtil.compareDate(detailVO.getOperationDate(), lastDate) > 0){
                    // 操作日期大于区间结束日期无效
                    if(DateUtil.compareDate(detailVO.getOperationDate(), rentDate) > 0){
                        return;
                    }
                    // 操作日期小于等于区间结束日期 T1 < A <= T2
                    // 下一操作日期大于区间结束日期 T1 < A <= T2 < B
                    if(DateUtil.compareDate(detailVO1.getOperationDate(), rentDate) > 0){
                        this.setValue(startDate, rentDate, vo, result);// A-T2
                    }
                    // 下一操作日期小于等于区间结束日期 T1 < A < B <= T2
                    else {
                        this.setValue(startDate, endDate, vo, result);// A-B-1，缺失B-T2
                    }
                }
                // 操作日期小于等于区间开始日期 A <= T1
                else {
                    // 下一操作日期大于区间结束日期 A <= T1 < T2 < B
                    if(DateUtil.compareDate(detailVO1.getOperationDate(), rentDate) > 0){
                        this.setValue(lastDate, rentDate, vo, result);// T1-T2
                    }
                    // 下一操作日期小于等于区间结束日期 A <= T1 < B <= T2
                    else {
                        this.setValue(lastDate, endDate, vo, result);// T1-B
                    }
                }
            }
        }
    }

    /**
     * 赋值期初数量
     * @param detailVO
     * @param detailVO1
     */
    private void setBeginNum(MaterialFlowVO detailVO, MaterialFlowVO detailVO1) {
        if(MaterialStateEnum.退场.getCode().equals(detailVO1.getMaterialState())){
            // 减去退赔数量
            if(MaterialConstant.YES.equals(detailVO1.getUseStatus())){
                detailVO1.setStartedNum(ComputeUtil.safeSub(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(detailVO.getStopedNum());
            } else {
                detailVO1.setStartedNum(detailVO.getStartedNum());
                detailVO1.setStopedNum(ComputeUtil.safeSub(detailVO.getStopedNum(), detailVO1.getNum()));
            }
        } else {
            if(MaterialConstant.YES.equals(detailVO1.getUseStatus())){
                detailVO1.setStartedNum(ComputeUtil.safeAdd(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(ComputeUtil.safeSub(detailVO.getStopedNum(), detailVO1.getNum()));
            } else {
                detailVO1.setStartedNum(ComputeUtil.safeSub(detailVO.getStartedNum(), detailVO1.getNum()));
                detailVO1.setStopedNum(ComputeUtil.safeAdd(detailVO.getStopedNum(), detailVO1.getNum()));
            }
        }
    }

    /**
     * 赋值开始时间、结束时间
     * @param startDate
     * @param endDate
     * @param vo
     * @param result
     */
    private void setValue(Date startDate, Date endDate, CalculateDailyVO vo, List<CalculateDailyVO> result) {
        if(DateUtil.compareDate(startDate, endDate) > 0){
            return;
        }
        vo.setStartDate(startDate);
        vo.setEndDate(endDate);
        if(ComputeUtil.isLessOrEqual(vo.getNum(), BigDecimal.ZERO)){
            return;
        }
        result.add(vo);
    }

    /**
     * 根据租赁方式拼装不同的缓存VO
     *
     * @param materialVO    台账VO
     * @param detailVO     操作VO
//     * @param cache     查询结果
     * @param flag      是否启用
     * @return
     */
    private CalculateDailyVO transferCacheVO(MaterialVO materialVO, MaterialFlowVO detailVO, Boolean flag) {
        CalculateDailyVO vo = new CalculateDailyVO();
        vo.setMaterialId(materialVO.getMaterialId());
        vo.setMaterialCode(materialVO.getMaterialCode());
        vo.setMaterialName(materialVO.getMaterialName());
        vo.setMaterialSourceId(materialVO.getMaterialSourceId());
        vo.setMaterialTypeId(materialVO.getMaterialTypeId());
        vo.setMaterialTypeName(materialVO.getMaterialTypeName());
        vo.setSpec(materialVO.getSpec());
        vo.setUnitId(materialVO.getUnitId());
        vo.setUnitName(materialVO.getUnitName());
        vo.setRealUnitId(materialVO.getRealUnitId());
        vo.setRealUnitName(materialVO.getRealUnitName());
        vo.setRealTransScale(materialVO.getRealTransScale());
        vo.setRentUnitId(materialVO.getRentUnitId());
        vo.setRentUnitName(materialVO.getRentUnitName());
        vo.setRentTransScale(materialVO.getRentTransScale());
        vo.setParameterId(materialVO.getId());
        if(flag == null) {
            vo.setNum(detailVO.getNum());
            vo.setMaterialState(detailVO.getMaterialState());
        } else if(flag){
            vo.setNum(detailVO.getStartedNum());
            vo.setMaterialState(MaterialStateEnum.启用.getCode());
        } else {
            vo.setNum(detailVO.getStopedNum());
            vo.setMaterialState(MaterialStateEnum.停用.getCode());
        }
        return vo;
    }

    /**
     * 计算金额
     *
     * @param price 单价
     * @param days       天数
     * @param num       数量
     * @return 计算结果
     */
    private BigDecimal getMny(BigDecimal price, Integer days, BigDecimal num) {
        return ComputeUtil.safeMultiply(price, new BigDecimal(days), num);
    }

//    @Override
//    public List<RestituteVO> queryRestituteData(String contractId, String lastRentDate, String rentDate) {
//        LambdaQueryWrapper<RestituteEntity> wrapper = new LambdaQueryWrapper<>();
//        wrapper.eq(RestituteEntity::getContractId, contractId);
//        wrapper.eq(RestituteEntity::getSettleFlag, 0);
//        wrapper.in(RestituteEntity::getBillState, Arrays.asList(1, 3));
//        wrapper.orderByDesc(RestituteEntity::getCreateTime);
//        if (StringUtils.isBlank(lastRentDate)) {
//            wrapper.le(RestituteEntity::getRestituteDate, rentDate);
//        }else {
//            wrapper.between(RestituteEntity::getRestituteDate, lastRentDate, rentDate);
//        }
//        List<RestituteEntity> list = restituteService.list(wrapper);
//        if (ListUtil.isEmpty(list)) {
//            return new ArrayList<>();
//        }else {
//            return BeanMapper.mapList(list, RestituteVO.class);
//        }
//    }

    @Override
    public Date getLastRentDate(String contractId) {
        QueryParam param = new QueryParam();
        param.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("rentDate", QueryParam.DESC);
        List<CalculateEntity> lastList = super.queryList(param, false);
        if(CollectionUtils.isNotEmpty(lastList)){
            return lastList.get(0).getRentDate();
        }
        return null;
    }
}
