package com.ejianc.business.storecloud.utils;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
public class IdCardUtil {

  /**
   * 数字
   */
  public final static Pattern NUMBERS = Pattern.compile("\\d+");

  /**
   * 中国公民身份证号码最小长度。
   */
  private static final int CHINA_ID_MIN_LENGTH = 15;

  /**
   * 中国公民身份证号码最大长度。
   */
  private static final int CHINA_ID_MAX_LENGTH = 18;

  public static Exception isValidatedAllIdcard(String idcard) throws Exception {
    boolean ret = isIdcard(idcard);
    if (!ret) {
      throw new Exception("身份证格式有误");
    }
    return null;
  }

  final static Map<Integer, String> zoneNum = new HashMap<>();
  /**
   * 身份证省份编码
   * */
  static {
    zoneNum.put(11, "北京");
    zoneNum.put(12, "天津");
    zoneNum.put(13, "河北");
    zoneNum.put(14, "山西");
    zoneNum.put(15, "内蒙古");
    zoneNum.put(21, "辽宁");
    zoneNum.put(22, "吉林");
    zoneNum.put(23, "黑龙江");
    zoneNum.put(31, "上海");
    zoneNum.put(32, "江苏");
    zoneNum.put(33, "浙江");
    zoneNum.put(34, "安徽");
    zoneNum.put(35, "福建");
    zoneNum.put(36, "江西");
    zoneNum.put(37, "山东");
    zoneNum.put(41, "河南");
    zoneNum.put(42, "湖北");
    zoneNum.put(43, "湖南");
    zoneNum.put(44, "广东");
    zoneNum.put(45, "广西");
    zoneNum.put(46, "海南");
    zoneNum.put(50, "重庆");
    zoneNum.put(51, "四川");
    zoneNum.put(52, "贵州");
    zoneNum.put(53, "云南");
    zoneNum.put(54, "西藏");
    zoneNum.put(61, "陕西");
    zoneNum.put(62, "甘肃");
    zoneNum.put(63, "青海");
    zoneNum.put(64, "宁夏");
    zoneNum.put(65, "新疆");
    zoneNum.put(71, "台湾");
    zoneNum.put(81, "香港");
    zoneNum.put(82, "澳门");
    zoneNum.put(91, "国外");
  }

  /**
   * 校验码
   */
  final static int[] PARITYBIT = { '1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2' };

  /**
   * 加权因子wi
   */
  final static int[] POWER_LIST = { 7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2 };

  /**
   * 验证身份证号有效性
   *
   * @param idCard:身份证号
   * @return true/false
   */
  public static boolean isIdcard(String idCard) {
    // 号码长度应为15位或18位
    if (idCard == null || (idCard.length() != 15 && idCard.length() != 18)) {
      return false;
    }
    // 校验区位码
    if (!zoneNum.containsKey(Integer.valueOf(idCard.substring(0, 2)))) {
      return false;
    }
    // 校验年份
    String year = idCard.length() == 15 ? "19" + idCard.substring(6, 8) : idCard.substring(6, 10);
    final int iyear = Integer.parseInt(year);
    if (iyear < 1900 || iyear > Calendar.getInstance().get(Calendar.YEAR)) {
      // 1900年的PASS，超过今年的PASS
      return false;
    }
    // 校验月份
    String month = idCard.length() == 15 ? idCard.substring(8, 10) : idCard.substring(10, 12);
    final int imonth = Integer.parseInt(month);
    if (imonth < 1 || imonth > 12) {
      return false;
    }
    // 校验天数
    String day = idCard.length() == 15 ? idCard.substring(10, 12) : idCard.substring(12, 14);
    final int iday = Integer.parseInt(day);
    if (iday < 1 || iday > 31) {
      return false;
    }
    // 校验一个合法的年月日
    if (!isValidDate(year + month + day)) {
      return false;
    }
    // 校验位数
    int power = 0;
    final char[] cs = idCard.toUpperCase().toCharArray();
    for (int i = 0; i < cs.length; i++) {// 循环比正则表达式更快
      if (i == cs.length - 1 && cs[i] == 'X') {
        break;// 最后一位可以是X或者x
      }
      if (cs[i] < '0' || cs[i] > '9') {
        return false;
      }
      if (i < cs.length - 1) {
        power += (cs[i] - '0') * POWER_LIST[i];
      }
    }
    // 校验“校验码”
    if (idCard.length() == 15) {
      return true;
    }
    return cs[cs.length - 1] == PARITYBIT[power % 11];
  }

  /**
   * 判断字符串是否为日期格式(合法)
   *
   * @param inDate:字符串时间
   * @return true/false
   */
  public static boolean isValidDate(String inDate) {
    if (inDate == null) {
      return false;
    }
    // 或yyyy-MM-dd
    SimpleDateFormat dataFormat = new SimpleDateFormat("yyyyMMdd");
    if (inDate.trim().length() != dataFormat.toPattern().length()) {
      return false;
    }
    // 该方法用于设置Calendar严格解析字符串;默认为true,宽松解析
    dataFormat.setLenient(false);
    try {
      dataFormat.parse(inDate.trim());
    } catch (Exception e) {
      return false;
    }
    return true;
  }



  /**
   * 给定内容是否匹配正则
   *
   * @param pattern 模式
   * @param content 内容
   * @return 正则为null或者""则不检查，返回true，内容为null返回false
   */
  private static boolean isMatch(Pattern pattern, CharSequence content) {
    if (content == null || pattern == null) {
      // 提供null的字符串为不匹配
      return false;
    }
    return pattern.matcher(content).matches();
  }

  /**
   * 将power和值与11取模获得余数进行校验码判断
   *
   * @param iSum 加权和
   * @return 校验位
   */
  private static char getCheckCode18(int iSum) {
    switch (iSum % 11) {
      case 10:
        return '2';
      case 9:
        return '3';
      case 8:
        return '4';
      case 7:
        return '5';
      case 6:
        return '6';
      case 5:
        return '7';
      case 4:
        return '8';
      case 3:
        return '9';
      case 2:
        return 'x';
      case 1:
        return '0';
      case 0:
        return '1';
      default:
        return ' ';
    }
  }
  /**
   * 获得18位身份证校验码
   * 计算方式：
   * 将前面的身份证号码17位数分别乘以不同的系数。从第一位到第十七位的系数分别为：7 9 10 5 8 4 2 1 6 3 7 9 10 5 8 4 2
   * 将这17位数字和系数相乘的结果相加
   * 用加出来和除以11，看余数是多少
   * 余数只可能有0 1 2 3 4 5 6 7 8 9 10这11个数字。其分别对应的最后一位身份证的号码为1 0 X 9 8 7 6 5 4 3 2
   * 通过上面得知如果余数是2，就会在身份证的第18位数字上出现罗马数字的Ⅹ。如果余数是10，身份证的最后一位号码就是2
   * @param code17 18位身份证号中的前17位
   * @return 第18位
   */
  private static char getCheckCode18(String code17) {
    int sum = getPowerSum(code17.toCharArray());
    return getCheckCode18(sum);
  }

  /**
   * 将身份证的每位和对应位的加权因子相乘之后，再得到和值
   *
   * @param iArr 身份证号码的数组
   * @return 身份证编码
   */
  private static int getPowerSum(char[] iArr) {
    int iSum = 0;
    if (POWER_LIST.length == iArr.length) {
      for (int i = 0; i < iArr.length; i++) {
        iSum += Integer.valueOf(String.valueOf(iArr[i])) * POWER_LIST[i];
      }
    }
    return iSum;
  }
}
