package com.ejianc.business.assist.rmat.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.assist.rmat.bean.*;
import com.ejianc.business.assist.rmat.enums.BillTypeEnum;
import com.ejianc.business.assist.rmat.enums.SettleEnum;
import com.ejianc.business.assist.rmat.mapper.SettleMapper;
import com.ejianc.business.assist.rmat.service.*;
import com.ejianc.business.assist.rmat.utils.DateUtil;
import com.ejianc.business.assist.rmat.utils.PushSupUtil;
import com.ejianc.business.assist.rmat.vo.LoseVO;
import com.ejianc.business.assist.rmat.vo.RentCalculateVO;
import com.ejianc.business.assist.rmat.vo.RestituteVO;
import com.ejianc.business.assist.rmat.vo.SettleVO;
import com.ejianc.business.assist.rmat.vo.record.SettleRecordVO;
import com.ejianc.business.contractbase.pool.contractpool.api.IContractPoolApi;
import com.ejianc.business.contractbase.pool.contractpool.vo.ContractPoolVO;
import com.ejianc.business.contractbase.pool.settlepool.api.ISettlePoolApi;
import com.ejianc.business.contractbase.pool.settlepool.vo.SettlePoolVO;
import com.ejianc.business.contractpub.util.BeanConvertorUtil;
import com.ejianc.business.targetcost.enums.BillCategoryEnum;
import com.ejianc.business.targetcost.enums.BussinessTypeEnum;
import com.ejianc.business.targetcost.vo.ExecutionVO;
import com.ejianc.business.targetcost.vo.TotalExecutionVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 周转材租赁结算单
 * 
 * @author generator
 * 
 */
@Service("settleService")
public class SettleServiceImpl extends BaseServiceImpl<SettleMapper, SettleEntity> implements ISettleService{

    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    
    private static final String ASSIST_RMAT_RENT_IN_CONTRACT_SETTLE = "ASSIST_RMAT_RENT_IN_CONTRACT_SETTLE";//此处需要根据实际修改
    private static final String OPERATE = "SETTLE_BILL_SYNC";
    private static final String BILL_TYPE = BillTypeEnum.辅料中心租入合同结算单.getCode();
    private static final String BILL_NAME = BillTypeEnum.辅料中心租入合同结算单.getName();

    private static final String PUSH_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistrmat/settle/billSync";
    private static final String DEL_SUP_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistrmat/settle/billDel";
    
    @Autowired
    private IBillTypeApi billTypeApi;
    
    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IRentCalculateService calculateService;
    
    @Autowired
    private IRestituteService restituteService;

    @Autowired
    private ILoseService loseService;

    @Autowired
    private IContractService contractService;

    @Autowired
    private ISettlePoolApi settlePoolApi;

    @Value("${common.env.base-host}")
    private String baseHost;

    @Autowired
    private IContractPoolApi contractPoolApi;

    @Autowired
    private PushSupUtil pushSupUtil;
    
    @Override
    public SettleVO saveOrUpdate(SettleVO saveOrUpdateVO) {
        SettleEntity entity = BeanMapper.map(saveOrUpdateVO, SettleEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(ASSIST_RMAT_RENT_IN_CONTRACT_SETTLE, InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
            
            // 签字状态
            entity.setSupplierSignStatus(0);
        }
        
        // 校验合同
        validateContract(entity.getContractId(), entity.getId());
        // 校验结算日期
        SettleVO lastVo = queryLastSettleTaxMny(entity.getContractId(), entity.getId());
        if (lastVo.getId() != null) {
            int compareDate = DateUtil.compareDate(lastVo.getSettleDate(), entity.getSettleDate());
            if (compareDate > 0) {
                throw new BusinessException("本次结算日期" + DateUtil.formatDate(entity.getSettleDate()) + "不能小于上次结算日期" + DateUtil.formatDate(lastVo.getSettleDate()));
            }
        }
        
        this.saveOrUpdate(entity, false);
        
        return BeanMapper.map(selectById(entity.getId()), SettleVO.class);
    }

    @Override
    public String validateContract(Long contractId, Long billId) {
        // 同一个合同只能存在一个自由态或审批中的单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NOT_IN, "1,3"));
        if(billId != null){
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, billId));
        }
        List<SettleEntity> list = super.queryList(queryParam, false);
        if (ListUtil.isNotEmpty(list)) {
            throw new BusinessException("当前合同存在非审批通过态的结算单，不允许新增!");
        }
        return "校验通过！";
    }

    @Override
    public List<RentCalculateVO> queryCalculateData(Long contractId, String lastSettleDate, String settleDate) {
        LambdaQueryWrapper<RentCalculateEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RentCalculateEntity::getContractId, contractId);
        wrapper.eq(RentCalculateEntity::getSettleFlag, 0);
        wrapper.in(RentCalculateEntity::getBillState, Arrays.asList(1, 3));
        wrapper.orderByDesc(RentCalculateEntity::getCreateTime);
        if (StringUtils.isBlank(lastSettleDate)) {
            wrapper.le(RentCalculateEntity::getRentDate, settleDate);
        }else {
            wrapper.between(RentCalculateEntity::getRentDate, lastSettleDate, settleDate);
        }

        List<RentCalculateEntity> list = calculateService.list(wrapper);

        if (ListUtil.isEmpty(list)) {
            return new ArrayList<>();
        }else {
            return BeanMapper.mapList(list, RentCalculateVO.class);
        }
    }

    @Override
    public List<RestituteVO> queryRestituteData(Long contractId, String lastSettleDate, String settleDate) {
        LambdaQueryWrapper<RestituteEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RestituteEntity::getContractId, contractId);
        wrapper.eq(RestituteEntity::getSettleFlag, 0);
        wrapper.in(RestituteEntity::getBillState, Arrays.asList(1, 3));
        wrapper.orderByDesc(RestituteEntity::getCreateTime);
        if (StringUtils.isBlank(lastSettleDate)) {
            wrapper.le(RestituteEntity::getRestituteDate, settleDate);
        }else {
            wrapper.between(RestituteEntity::getRestituteDate, lastSettleDate, settleDate);
        }

        List<RestituteEntity> list = restituteService.list(wrapper);

        if (ListUtil.isEmpty(list)) {
            return new ArrayList<>();
        }else {
            return BeanMapper.mapList(list, RestituteVO.class);
        }
    }

    @Override
    public List<LoseVO> queryLoseData(Long contractId, String lastSettleDate, String settleDate) {
        LambdaQueryWrapper<LoseEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LoseEntity::getContractId, contractId);
        wrapper.eq(LoseEntity::getSettleFlag, 0);
        wrapper.in(LoseEntity::getBillState, Arrays.asList(1, 3));
        wrapper.orderByDesc(LoseEntity::getCreateTime);
        if (StringUtils.isBlank(lastSettleDate)) {
            wrapper.le(LoseEntity::getBillDate, settleDate);
        }else {
            wrapper.between(LoseEntity::getBillDate, lastSettleDate, settleDate);
        }
        
        List<LoseEntity> list = loseService.list(wrapper);

        if (ListUtil.isEmpty(list)) {
            return new ArrayList<>();
        }else {
            return BeanMapper.mapList(list, LoseVO.class);
        }
        
    }

    @Override
    public SettleVO queryLastSettleTaxMny(Long contractId, Long billId) {
        LambdaQueryWrapper<SettleEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SettleEntity::getContractId, contractId);
        wrapper.in(SettleEntity::getBillState, Arrays.asList(1, 3));
        if(billId != null){
            wrapper.ne(SettleEntity::getId, billId);
        }
        wrapper.orderByDesc(SettleEntity::getCreateTime);
        
        List<SettleEntity> list = this.list(wrapper);

        if (ListUtil.isEmpty(list)) {
            return new SettleVO();
        }else {
            return BeanMapper.map(list.get(0), SettleVO.class);
        }
        
    }

    @Override
    public SettleRecordVO querySettleRecord(Long id) {
        ContractEntity entity = contractService.selectById(id);

        SettleRecordVO vo = new SettleRecordVO();
        vo.setId(id);
        vo.setMainContractId(id);
        vo.setContractMny(entity.getContractMny());
        vo.setContractTaxMny(entity.getContractTaxMny());
        vo.setPerformanceStatus(entity.getPerformanceStatus());
        vo.setChangeStatus(entity.getChangeStatus());
        vo.setSignatureStatus(entity.getSignatureStatus());

        // 查询变更记录
        LambdaQueryWrapper<SettleEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SettleEntity::getContractId, id);
        wrapper.orderByDesc(SettleEntity::getCreateTime);
        List<SettleEntity> settleList = super.list(wrapper);
        
        if (ListUtil.isNotEmpty(settleList)) {
            vo.setTotalSettleTaxMny(settleList.get(0).getTotalSettleTaxMny());
            vo.setTotalSettleMny(settleList.get(0).getTotalSettleMny());
            vo.setSettleRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalSettleTaxMny(), vo.getTotalSettleMny()), new BigDecimal("100")));
        }

        vo.setDetailList(BeanMapper.mapList(settleList, SettleVO.class));
        
        return vo;
    }

    @Override
    public Boolean pushSettleToPool(SettleVO vo) {
        SettlePoolVO poolVO = new SettlePoolVO();
        try {
            logger.info("结算单对象 -> 结算池对象自动转换开始");
            // 对象自动转换
            BeanConvertorUtil.convert(vo, poolVO);

            // 个别字段需要手动封装
            poolVO.setSourceId(vo.getId());
            poolVO.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            poolVO.setSourceType(vo.getSettleType());
            poolVO.setSourceTypeName(SettleEnum.getEnumByCode(vo.getSettleType()).getName());
            poolVO.setSettleProperty(Integer.valueOf(SettleEnum.支出.getCode()));
            poolVO.setSettlePropertyName(SettleEnum.支出.getName());

            poolVO.setBillCodeUrl("/ejc-assistrmat-frontend/#/settle/card?id=" + vo.getId());
            
            // 查询合同
            ContractEntity contractEntity = contractService.selectById(vo.getContractId());
            poolVO.setSupplementFlag(contractEntity.getSupplementFlag());
            poolVO.setMaiContractId(contractEntity.getId());
            poolVO.setMaiContractCode(contractEntity.getBillCode());
            poolVO.setMaiContractName(contractEntity.getContractName());
            poolVO.setSignDate(contractEntity.getSignDate());

            poolVO.setCreateTime(vo.getCreateTime());
            poolVO.setCreateUserCode(vo.getCreateUserCode());
            poolVO.setUpdateTime(vo.getUpdateTime());
            poolVO.setUpdateUserCode(vo.getUpdateUserCode());


            CommonResponse<SettlePoolVO> res = settlePoolApi.saveOrUpdateSettle(poolVO);
            if(res.isSuccess()) {
                logger.info("结算单推送至结算池成功！结算单id-{}", vo.getId());
                return true;
            } else {
                logger.error("结算单推送合同池失败！结算单id-{}，{}", vo.getId(), res.getMsg());
            }
        } catch (Exception e) {
            logger.error("结算单推送合同池失败！结算单id-{}", vo.getId(), e);
        }
        return false;

    }

    @Override
    public Boolean delSettleFromPool(Long id) {
        SettlePoolVO poolVO = new SettlePoolVO();
        poolVO.setSourceId(id);

        CommonResponse<SettlePoolVO> res = settlePoolApi.deleteSettle(poolVO);
        if (res.isSuccess()) {
            logger.info("将结算单从结算池中删除成功！结算单id-{}", id);
            return true;
        }

        logger.error("将结算单从结算池中删除失败！结算单id-{}，{}", id, res.getMsg());
        return false;
    }

    @Override
    public ExecutionVO targetCost(SettleEntity entity) {
        // 查询结算单
        LambdaQueryWrapper<SettleEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SettleEntity::getContractId, entity.getContractId());
        wrapper.in(SettleEntity::getBillState, Arrays.asList(1,3));
        List<SettleEntity> list = this.list(wrapper);
        // 计算所有结算累计结算金额 
        BigDecimal totalSettleTaxMny = list.stream().filter(e -> e.getSettleTaxMny() != null).map(SettleEntity::getSettleTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal totalSettleMny = list.stream().filter(e -> e.getSettleMny() != null).map(SettleEntity::getSettleMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        // 查询合同
        ContractEntity contractEntity = contractService.getById(entity.getContractId());

        ExecutionVO executionVO = new ExecutionVO();
        TotalExecutionVO totalVO = new TotalExecutionVO();
        totalVO.setSourceId(entity.getId());
        totalVO.setTenantId(entity.getTenantId());
        totalVO.setBillCode(entity.getBillCode());
        totalVO.setBillType(BillTypeEnum.辅料中心租入合同结算单.getCode());
        totalVO.setBillCategory(BillCategoryEnum.合同.getCode());
        totalVO.setBussinessType(BussinessTypeEnum.周转材租赁合同.getCode());
        totalVO.setProjectId(entity.getProjectId());
        totalVO.setOrgId(entity.getOrgId());
        totalVO.setMoney(ComputeUtil.safeDiv(totalSettleMny, contractEntity.getContractMny()));
        totalVO.setTaxMoney(ComputeUtil.safeDiv(totalSettleTaxMny, contractEntity.getContractTaxMny()));
        totalVO.setLinkUrl(baseHost + "ejc-assistrmat-frontend/#/settle/card?id=" + entity.getId());

        executionVO.setTotalVO(totalVO);

        return executionVO;
    }

    @Override
    public Boolean pushContract(SettleVO vo, String updateLevel) {
        ContractPoolVO data = new ContractPoolVO();
        try {
            
//            data.set

            CommonResponse<ContractPoolVO> transDataResp = contractPoolApi.saveOrUpdateContract(data);

            if(transDataResp.isSuccess()) {
                return true;
            } else {
                logger.error("合同id-{}推送合同池失败，{}",vo.getId(), transDataResp.getMsg());
            }
        } catch (Exception e) {
            logger.error("合同-{}推送合同池失败，", vo.getId(), e);
        }

        return null;
    }

    /**
     * 单据推送到供方协同服务
     * @param entity
     * @return
     */
    @Override
    public Boolean pushBillToSupCenter(SettleEntity entity) {
        Boolean syncFlag = pushSupUtil.pushBillToSupCenter((JSONObject)JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, PUSH_BILL_SERVER_URL);
        return syncFlag;
        
    }

    /**
     * 将推送至供方的单据作废
     * @param entity
     * @return
     */
    @Override
    public Boolean delPushBill(SettleEntity entity) {
        Boolean delSuc = pushSupUtil.delPushBill((JSONObject) JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, DEL_SUP_BILL_SERVER_URL);
        return delSuc;

    }

    @Override
    public String updateBillSupSignSyncInfo(HttpServletRequest request) {
        String billId = request.getParameter("billId");
        logger.info("辅料中心结算单billId:" + billId);
        SettleEntity entity = super.selectById(billId);
        logger.info("辅料中心结算单实体:" + entity);
        String msg = pushSupUtil.updateBillSupSignSyncInfo(request, (JSONObject) JSONObject.toJSON(entity),
                SettleEntity.class, OPERATE, BILL_TYPE, BILL_NAME);
        logger.info("返回消息msg" + msg);
        return msg;
    }
    
    
}
