package com.ejianc.business.assist.material.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.assist.material.bean.MaterialDeliveryEntity;
import com.ejianc.business.assist.material.bean.MaterialOrderDetailEntity;
import com.ejianc.business.assist.material.bean.MaterialOrderEntity;
import com.ejianc.business.assist.material.enums.BillTypeEnum;
import com.ejianc.business.assist.material.mapper.MaterialDeliveryMapper;
import com.ejianc.business.assist.material.mapper.MaterialOrderMapper;
import com.ejianc.business.assist.material.service.IMaterialOrderService;
import com.ejianc.business.assist.rmat.utils.PushSupUtil;
import com.ejianc.business.common.CommonConstant;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletRequest;
import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 租赁订单
 * 
 * @author generator
 * 
 */
@Service("materialOrderService")
public class MaterialOrderServiceImpl extends BaseServiceImpl<MaterialOrderMapper, MaterialOrderEntity> implements IMaterialOrderService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /** 无权访问 */
    private static final String noPower = "无权限访问该服务，请先联系管理员进行授权！";

    @Autowired
    private PushSupUtil pushSupUtil;

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IPushMessageApi pushMessageApi;

    @Autowired
    private ISystemDataPushService systemDataPushService;

    @Autowired
    private MaterialDeliveryMapper materialDeliveryMapper;



    private static final String OPERATE = "ORDER_BILL_SYNC";
    private static final String BILL_TYPE = BillTypeEnum.租赁订单.getCode();
    private static final String BILL_NAME = BillTypeEnum.租赁订单.getName();

    private static final String PUSH_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistmaterial/order/billSync";
    private static final String DEL_SUP_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistmaterial/order/billDel";


    private static final String CHANGE_CLOSE_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistmaterial/order/changeCloseState";


    @Override
    public boolean pushBillToSupCenter(MaterialOrderEntity entity) {
        Boolean syncFlag = pushSupUtil.pushBillToSupCenter((JSONObject)JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, PUSH_BILL_SERVER_URL);
        return syncFlag;
    }

    @Override
    public String updateBillSupSignSyncInfo(HttpServletRequest request) {

        String msg = null;

        Jedis jedis = null;
        boolean locked = false;

        String billId = request.getParameter("billId");
        String supOperatorName = request.getParameter("supOperatorName");
        String supOperatorPhone = request.getParameter("supOperatorPhone");
        String supOperatorUserCode = request.getParameter("supOperatorUserCode");
        Date supOperateTime = new Date(Long.valueOf(request.getParameter("supOperateTime")));

        String denialReason = request.getParameter("denialReason");
        String receiveState = request.getParameter("receiveState");
        String deliverState = request.getParameter("deliverState");
        String transData = request.getParameter("orderDetailList");

        List<MaterialOrderDetailEntity> orderDetailList = JSONArray.parseArray(transData, MaterialOrderDetailEntity.class);
        Map<Long, MaterialOrderDetailEntity> collect = orderDetailList.stream().collect(Collectors.toMap(MaterialOrderDetailEntity::getId, Function.identity(), (k1, k2) -> k2));

        logger.info("接收到单据成功接收通知：id-{}, supOperatorName-{}, supOperatorPhone-{}, supOperatorUserCode-{}, supOperateTime-{}, orderDetailList-{}",
                billId, supOperatorName, supOperatorPhone, supOperatorUserCode, supOperateTime, orderDetailList.toString());

        MaterialOrderEntity entity = super.selectById(billId);
        //设置供方签字信息
        entity.setSupOperateTime(supOperateTime);
        entity.setSupOperatorName(supOperatorName);
        entity.setSupOperatorPhone(supOperatorPhone);
        entity.setSupOperatorUserCode(supOperatorUserCode);

        String key = BILL_TYPE + "::" + entity.getId().toString();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            if(!locked) {
                logger.error("单据作废失败，单据锁获取失败！");
                releaseLock(jedis, false, key, OPERATE);
                return "单据接收状态更新失败，单据锁获取失败！";
            }

            entity.setDenialReason(denialReason);
            entity.setReceiveState(receiveState);
            entity.setDeliverState(deliverState);
            for (MaterialOrderDetailEntity detailEntity:entity.getOrderDetailList()){
                if (collect.get(detailEntity.getId())!=null){
                    MaterialOrderDetailEntity detail = collect.get(detailEntity.getId());
                    detailEntity.setReceiveNumsSum(detail.getReceiveNumsSum());
                    detailEntity.setNotDeliveredNumsSum(detail.getReceiveNumsSum());
                }
            }


            //更新单据
            super.saveOrUpdate(entity, false);

            //向单据制单人和经办人推送该消息
            String msgSendResult = sendMsg(entity, "供方已确认提醒", "租赁订单[" + entity.getBillCode() + "]供方已确认完成");
            if (null != msgSendResult) {
                String createUserId = PushSupUtil.getCreateUserId(entity.getCreateUserCode());
                logger.error("向用户-{}发送单据id-{}接收提醒失败，原因：{}", StringUtils.join(createUserId, entity.getLinkId()),
                        entity.getId(), msgSendResult);
            }

        } catch (Exception e) {
            logger.error("单据id-{}接收信息回写异常，", entity.getId(), e);
            msg = "单据接收信息回写失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return msg;
    }

    @Override
    public boolean delPushBill(MaterialOrderEntity entity) {
        Boolean delSuc = pushSupUtil.delPushBill((JSONObject) JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, DEL_SUP_BILL_SERVER_URL);
        return delSuc;
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String OPERATE) {
        try {
            if(locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }

    /**
     * 通知用户消息
     *
     * @param entity
     */
    private String sendMsg(MaterialOrderEntity entity, String subject, String content) {

        String createUserId = PushSupUtil.getCreateUserId(entity.getCreateUserCode());

        String[] msgRecUserIds = new String[]{createUserId , entity.getLinkId().toString()};

        logger.info("消息接收人: {}", StringUtils.join(msgRecUserIds, ","));
        PushMsgParameter msgParameter = new PushMsgParameter();
        //消息接收人
        msgParameter.setReceivers(msgRecUserIds);
        //消息内容
        msgParameter.setContent(content);
        //消息主题
        msgParameter.setSubject(subject);
        //消息类型
        msgParameter.setMsgType("notice");
        msgParameter.setTenantId(entity.getTenantId().toString());
        //消息保存
        msgParameter.setSaveFlag(true);
        //消息发送人
        msgParameter.setSendUserId(InvocationInfoProxy.getUserid());
        //消息发送渠道
        msgParameter.setChannel(new String[]{PushMsgParameter.CHANNEL_TYPE_SYS});

        CommonResponse<String> msgSendResp = pushMessageApi.pushMessage(msgParameter);
        if (!msgSendResp.isSuccess()) {
            return msgSendResp.getMsg();
        }
        return null;
    }

    @Override
    public String validateClose(MaterialOrderEntity entity){

        String msg = null;

        if (CommonConstant.ORDER_RECEIVE_WAIT.equals(entity.getReceiveState())||CommonConstant.ORDER_RECEIVE_NOT.equals(entity.getReceiveState())){
            msg = "订单状态为待接收或不予接收，订单不可关闭！";
        }

        if (CommonConstant.ORDER_DELIVER_ALL.equals(entity.getDeliverState())){
            msg = "订单状态为全部送货，订单不可关闭！";
        }

        List<MaterialDeliveryEntity> deliveryEntities = materialDeliveryMapper.selectDeliveryList(entity.getId(), CommonConstant.DELIVERY_ACCEPTANCE_ALL, CommonConstant.ORDER_DISABLE_NO);
        if (CollectionUtils.isNotEmpty(deliveryEntities)){
            msg = "订单存在未全部验收的送货单或存在未关闭的送货单，订单不可关闭！";
        }

        return msg;
    }


    @Override
    public String ChangeCloseState(MaterialOrderEntity entity){

        String msg = null;
        Jedis jedis = null;
        boolean locked = false;

        String key = BILL_TYPE + "::" + entity.getId().toString() + "::sup";

        try {
            //对单据进行加锁
            jedis = jedisPool.getResource();
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            if(!locked) {//加锁失败
                releaseLock(jedis, false, key, OPERATE);
                return "单据已被修改，请勿重复操作！";
            }
            // 回写参数
            Map<String, String> params = new HashMap<>();
            params.put("billId", entity.getId().toString());

            Map<String, Map<String, InputStream>> files = new HashMap<>();

            logger.info("参数-{}，供方-{}",JSONObject.toJSONString(params),entity.getSupplierId().toString());

            CommonResponse<String> backResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(CHANGE_CLOSE_BILL_SERVER_URL,params, entity.getSupplierId().toString(),files);

            logger.error("单据-{}发送请求结果，{}", BILL_NAME, JSONObject.toJSONString(backResp));
            if(!backResp.isSuccess()) {
                return BILL_NAME + "校验关闭发送请求失败";
            }

            if(noPower.equals(backResp.getData())){
                logger.error("发送请求URL-{}给供应商-{}失败, {}", CHANGE_CLOSE_BILL_SERVER_URL, entity.getSupplierId(), backResp.getData());
                return backResp.getData();
            }

            LambdaUpdateWrapper<MaterialOrderEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.eq(MaterialOrderEntity::getId,entity.getId());
            updateWrapper.set(MaterialOrderEntity::getDisableState,CommonConstant.ORDER_DISABLE_YES);
            super.update(updateWrapper);

        } catch (Exception e) {
            msg = "操作失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }
        return msg;
    }

}
