package com.ejianc.business.outrmat.contract.service.impl;

import cn.hutool.http.HttpRequest;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.outrmat.contract.bean.OutRmatContractChangeEntity;
import com.ejianc.business.outrmat.contract.bean.OutRmatContractEntity;
import com.ejianc.business.outrmat.contract.service.IOutRmatContractAsyncService;
import com.ejianc.business.outrmat.contract.service.IOutRmatContractChangeService;
import com.ejianc.business.outrmat.contract.service.IOutRmatContractService;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Recover;
import org.springframework.retry.annotation.Retryable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.concurrent.TimeUnit;

/**
 * @author baipengyan
 * @date 2022/5/20
 * @description 合同加水印异步http请求
 */
@Service("OutRmatContractContractAsyncService")
public class OutRmatContractAsyncServiceImpl implements IOutRmatContractAsyncService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BaseHost;
    @Autowired
    private IOutRmatContractService service;

    @Autowired
    private IOutRmatContractChangeService changeService;

    /**
     * 获取水印文件附件信息
     *
     * @param watermarkVO 水印配置参数
     * @return 水印文件信息
     */
    @Override
    @Async(value = "commonTask")
    @Retryable(maxAttempts = 3, backoff = @Backoff(delay = 2000, multiplier = 1.5), stateful = true)
    public void fetchWatermarkAttachment(HashMap<String, String> headers, WatermarkVO watermarkVO) {
        logger.info("异步获取水印文件附件信息开始，入参：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        Stopwatch start = Stopwatch.createStarted();
        String body = HttpRequest.post(BaseHost + "ejc-signaturemanage-web/common/fetchWatermarkAttachment")
                .addHeaders(headers)
                .body(JSON.toJSONString(watermarkVO))
                .timeout(60000)
                .execute().body();
        AttachmentVO attachmentVO = JSON.parseObject(body, AttachmentVO.class);
        logger.info("异步获取水印文件附件信息结束，请求签章中心耗时「{}」秒，文件中心的水印文件信息：{}", start.stop().elapsed(TimeUnit.SECONDS), JSON.toJSONString(attachmentVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        start.reset().start();

        boolean flag;
        if (watermarkVO.getSourceType().contains("Change")) {
            // 更新合同变更水印文件id
            LambdaUpdateWrapper<OutRmatContractChangeEntity> changeUpdateWrapper = new LambdaUpdateWrapper<>();
            changeUpdateWrapper.eq(OutRmatContractChangeEntity::getId, watermarkVO.getBillId());
            changeUpdateWrapper.set(OutRmatContractChangeEntity::getWatermarkContractChangeFileId, attachmentVO.getId());
            flag = changeService.update(changeUpdateWrapper);
        }else {
            // 更新合同水印文件id
            LambdaUpdateWrapper<OutRmatContractEntity> contractUpdateWrapper = new LambdaUpdateWrapper<>();
            contractUpdateWrapper.eq(OutRmatContractEntity::getId, watermarkVO.getBillId());
            contractUpdateWrapper.set(OutRmatContractEntity::getWatermarkContractFileId, attachmentVO.getId());
            flag = service.update(contractUpdateWrapper);
        }

        if (flag) {
            logger.info("获取水印文件附件信息并更新合同成功，写入数据库耗时「{}」秒，合同id：{}，原合同文件id：{}，水印合同文件id：{}", start.stop().elapsed(TimeUnit.SECONDS), watermarkVO.getBillId(), watermarkVO.getFileId(), attachmentVO.getId());
        }
    }

    @Recover
    public void recordWatermark(Exception e, HashMap<String, String> headers, WatermarkVO watermarkVO) {
        logger.error("获取水印文件附件信息并更新合同失败，请求参数：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
    }
}
