package com.ejianc.business.assist.store.service.impl;

import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.assist.store.bean.FlowEntity;
import com.ejianc.business.assist.store.bean.SurplusEntity;
import com.ejianc.business.assist.store.consts.InOutTypeEnum;
import com.ejianc.business.assist.store.consts.StoreCommonConsts;
import com.ejianc.business.assist.store.mapper.FlowMapper;
import com.ejianc.business.assist.store.service.IFlowService;
import com.ejianc.business.assist.store.service.ISurplusService;
import com.ejianc.business.assist.store.vo.*;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 仓库流水表
 *
 * @author generator
 */
@Service("flowService")
public class FlowServiceImpl extends BaseServiceImpl<FlowMapper, FlowEntity> implements IFlowService {

    @Autowired
    ISurplusService surplusService;

    /**
     * @param storeId
     * @description: 校验入库单是否被使用
     * @return: com.ejianc.framework.core.response.CommonResponse<cn.hutool.json.JSONObject>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<List<FlowEntity>> validateIsUse(Long sourceId, Long storeId) {
        QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("store_id", storeId);
        queryWrapper.eq("source_id", sourceId);
        queryWrapper.eq("in_out_flag", StoreCommonConsts.IN_OUT_TYPE_IN);
        List<FlowEntity> instoreList = this.list(queryWrapper);
        if (CollectionUtils.isEmpty(instoreList)) {
            return CommonResponse.error("未查询到该入库单流水信息！");
        }

        for (FlowEntity flow : instoreList) {
            Integer outUseFlag = flow.getOutUseFlag();
            //直入直出撤回不需要校验被使用，直入直出的入库必然是使用完的
            if (!InOutTypeEnum.直入直出入库.getInOutType().equals(flow.getInOutType())) {
                if (StoreCommonConsts.UseOutFlag.USEING.equals(outUseFlag)
                        || StoreCommonConsts.UseOutFlag.USE_FINISH.equals(outUseFlag)
                ) {
                    return CommonResponse.error("物资[名称：" + flow.getMaterialName() + "，规格：" + StrUtil.emptyToDefault(flow.getMaterialSpec(), "无") + "]已被使用，不允许撤回！");
                }
            }

            Integer accountFlag = flow.getAccountFlag();
            if (StoreCommonConsts.ONE.equals(accountFlag)) {
                return CommonResponse.error("单据已对账，不允许撤回！");
            }
            Integer settleFlag = flow.getSettleFlag();
            if (StoreCommonConsts.ONE.equals(settleFlag)) {
                return CommonResponse.error("单据已结算，不允许撤回！");
            }
        }
        return CommonResponse.success(instoreList);
    }


    @Override
    public List<FlowVO> queryInstoreFlowData(IPage<FlowVO> page, QueryWrapper wrapper) {
        return baseMapper.queryInstoreFlowData(page, wrapper);
    }

    /**
     * @param computeStoreVO
     * @description: 根据入库流水计算库存
     * @return: java.util.List<com.ejianc.business.assist.store.vo.SurplusVO>
     * @author songlx
     * @date: 2022/2/18
     */
    @Override
    public List<SurplusVO> computeStore(ComputeStoreVO computeStoreVO) {
        Long storeId = computeStoreVO.getStoreId();
        List<Long> materialIds = computeStoreVO.getMaterialIds();
        QueryWrapper<SurplusEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("store_id", storeId);
        wrapper.in(CollectionUtils.isNotEmpty(materialIds), "material_id", materialIds);
        List<SurplusEntity> oldList = surplusService.list(wrapper);
        boolean updateFlag = computeStoreVO.getUpdateFlag();
        List<SurplusVO> newSurplusVOList = baseMapper.computeStore(storeId, materialIds);

        Map<Long, List<SurplusVO>> newSurplusVOMap = newSurplusVOList.stream().collect(Collectors.groupingBy(SurplusVO::getMaterialId));
        ArrayList<Long> delIds = new ArrayList<>();
        Iterator<SurplusEntity> it = oldList.iterator();
        while (it.hasNext()) {
            SurplusEntity t = it.next();
            Long materialId = t.getMaterialId();
            List<SurplusVO> surplusVOS = newSurplusVOMap.get(materialId);
            if (CollectionUtils.isNotEmpty(surplusVOS)) {
                SurplusVO newSurplus = surplusVOS.get(0);
                t.setOutLockNum(newSurplus.getOutLockNum());
                t.setOutLockMny(newSurplus.getOutLockMny());
                t.setOutLockTaxMny(newSurplus.getOutLockTaxMny());
                t.setSurplusMny(newSurplus.getSurplusMny());
                t.setSurplusTaxMny(newSurplus.getSurplusTaxMny());
                t.setSurplusNum(newSurplus.getSurplusNum());
                t.setInstoreMny(newSurplus.getInstoreMny());
                t.setInstoreTaxMny(newSurplus.getInstoreTaxMny());
                t.setInstoreNum(newSurplus.getInstoreNum());
                t.setReturnGoodsNum(newSurplus.getReturnGoodsNum());
            } else {
                delIds.add(t.getId());
                it.remove();
            }

        }
        if (updateFlag) {
            surplusService.saveOrUpdateBatch(oldList);
            if (CollectionUtils.isNotEmpty(delIds)) {
                surplusService.removeByIds(delIds);
            }
        }
        List<SurplusVO> oldSurplusList = BeanMapper.mapList(oldList, SurplusVO.class);
        return oldSurplusList;
    }

    /**
     * @param storeManageVO
     * @description: 更新摊销金额
     * @return: com.ejianc.business.assist.store.vo.StoreManageVO
     * @author songlx
     * @date: 2022/3/2
     */
    @Override
    public StoreManageVO updateShareMny(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();

        if (CollectionUtils.isNotEmpty(flowVOList)) {
            Map<Long, List<FlowVO>> inFlowMap = flowVOList.stream().collect(Collectors.groupingBy(FlowVO::getSourceDetailId));
            QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
//            queryWrapper.eq("store_id", storeId);
            queryWrapper.in("source_detail_id", new ArrayList<>(inFlowMap.keySet()));
            List<FlowEntity> list = this.list(queryWrapper);
            list.forEach(
                    t -> {
                        List<FlowVO> flowVOS = inFlowMap.get(t.getSourceDetailId());
                        FlowVO flowVO = flowVOS.get(0);
                        BigDecimal shareMny = flowVO.getShareMny();
                        BigDecimal shareTaxMny = flowVO.getShareTaxMny();
                        Date lastShareDate = flowVO.getLastShareDate();
                        // 摊销日期
                        t.setLastShareDate(lastShareDate);
                        // 如果是生效就把摊销金额加上,反之减去
                        // 如果是生效就把当前净值减去,反之加上去
                        if (storeManageVO.getOutEffectiveON()) {
                            t.setShareMny(ComputeUtil.safeAdd(t.getShareMny(), shareMny));
                            t.setShareTaxMny(ComputeUtil.safeAdd(t.getShareTaxMny(), shareTaxMny));
                            t.setNetMny(ComputeUtil.safeSub(t.getNetMny(), shareMny));
                            t.setNetTaxMny(ComputeUtil.safeSub(t.getNetTaxMny(), shareTaxMny));
                        } else {
                            t.setShareMny(ComputeUtil.safeSub(t.getShareMny(), shareMny));
                            t.setShareTaxMny(ComputeUtil.safeSub(t.getShareTaxMny(), shareTaxMny));
                            t.setNetMny(ComputeUtil.safeAdd(t.getNetMny(), shareMny));
                            t.setNetTaxMny(ComputeUtil.safeAdd(t.getNetTaxMny(), shareTaxMny));
                        }
                    }
            );
            this.saveOrUpdateBatch(list);
            List<FlowVO> flowVOS = BeanMapper.mapList(list, FlowVO.class);
            storeManageVO.setFlowVOList(flowVOS);
        }
        return storeManageVO;
    }

    /**
     * @param sourceId
     * @param storeId
     * @description: 周传材料入库弃审校验是否出库或被摊销
     * @return: com.ejianc.framework.core.response.CommonResponse<cn.hutool.json.JSONObject>
     * @author songlx
     * @date: 2022/3/2
     */
    @Override
    public CommonResponse<JSONObject> validateTurnIsUseOutFlag(Long sourceId, Long storeId) {
        QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("source_id", sourceId);
        //已出库或者已摊销的入库不允许撤回
        queryWrapper.and(wrapper ->
                wrapper.in("out_use_flag", Arrays.asList(StoreCommonConsts.UseOutFlag.USEING, StoreCommonConsts.UseOutFlag.USE_FINISH)).or().gt("share_tax_mny", BigDecimal.ZERO));

        List<FlowEntity> list = this.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            FlowEntity flowEntity = list.get(0);
            return CommonResponse.error("材料[名称:" + flowEntity.getMaterialName() + ",规格:" + StrUtil.emptyToDefault(flowEntity.getMaterialSpec(), "无") + "]已出库或摊销,不允许撤回!");
        }

        queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("source_id", sourceId);
        queryWrapper.eq("in_out_flag", StoreCommonConsts.ONE);
        queryWrapper.eq("account_flag", StoreCommonConsts.ONE);
        list = this.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("入库单据已对账,不允许撤回!");
        }
        queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("source_id", sourceId);
        queryWrapper.eq("in_out_flag", StoreCommonConsts.ONE);
        queryWrapper.eq("settle_flag", StoreCommonConsts.ONE);
        list = this.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("入库单据已结算,不允许撤回!");
        }


        return CommonResponse.success();
    }

    /**
     * @param page
     * @param wrapper
     * @description: 对账/结算入库材料参照
     * @return: java.util.List<com.ejianc.business.assist.store.vo.StoreInstoreVO>
     * @author songlx
     * @date: 2022/3/9
     */
    @Override
    public List<StoreApiVO> queryStoreInstoreData(Page<StoreApiVO> page, QueryWrapper wrapper) {
        return baseMapper.queryStoreInstoreData(page, wrapper);
    }

    /**
     * @param page
     * @param wrapper
     * @description: 周转材材料参照
     * @return: java.util.List<com.ejianc.business.assist.store.vo.FlowVO>
     * @author songlx
     * @date: 2022/2/11
     */
    @Override
    public List<FlowVO> queryTurnInstoreFlowData(Page<FlowVO> page, QueryWrapper wrapper) {
        return baseMapper.queryTurnInstoreFlowData(page, wrapper);
    }

    @Override
    public Integer settleCount(HashMap<String, Object> map) {
        return baseMapper.settleCount(map);
    }

    @Override
    public void changeSettleFlag(HashMap<String, Object> map) {
        baseMapper.changeSettleFlag(map);
    }

    /**
     * @param page
     * @param wrapper
     * @description: 根据流水计算出来的库存列表
     * @return: java.util.List<com.ejianc.business.assist.store.vo.SurplusVO>
     * @author songlx
     * @date: 2022/4/1
     */
    @Override
    public List<SurplusVO> computeStoreList(Page<SurplusVO> page, QueryWrapper wrapper) {
        return baseMapper.computeStoreList(page, wrapper);
    }
}
