package com.ejianc.framework.auth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.Filter;

import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.mgt.DefaultSessionStorageEvaluator;
import org.apache.shiro.mgt.DefaultSubjectDAO;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.session.mgt.DefaultSessionManager;
import org.apache.shiro.spring.LifecycleBeanPostProcessor;
import org.apache.shiro.spring.web.ShiroFilterFactoryBean;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.apache.shiro.web.mgt.DefaultWebSubjectFactory;
import org.apache.shiro.web.session.mgt.DefaultWebSessionManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.MethodInvokingFactoryBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springside.modules.nosql.redis.pool.JedisPool;

import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.SpringContextTools;
import com.ejianc.framework.auth.shiro.StatelessAuthcFilter;
import com.ejianc.framework.auth.shiro.StatelessDefaultSubjectFactory;
import com.ejianc.framework.auth.shiro.StatelessRealm;
import com.ejianc.framework.auth.token.DefaultTokenProcessor;
import com.ejianc.framework.auth.token.ITokenProcessor;
import com.ejianc.framework.auth.token.TokenFactory;
import com.ejianc.framework.cache.redis.RedisPoolFactory;

@Configuration
@EnableConfigurationProperties(value = { FilterChainDefinitionsConfiguration.class })
@ConfigurationProperties(prefix = "ejc.shiro")
public class EjcAuthConfiguration {

	private int expr = 3600;
	private String sysid = "icop";
	private String loginUrl = "/sso/login";
	private String redisSessionUrl;
	private boolean sessionMutex = false;
	private String filterExcludes;

	@Autowired(required = false)
	private FilterChainDefinitionsConfiguration filterChainDefinitionsConfig;
	
	@Bean("springContextTools")
	public SpringContextTools getContextUtils() {
		SpringContextTools springContextTools = new SpringContextTools();
		return springContextTools;
	}

	@Bean("statelessRealm")
	public StatelessRealm getStatelessRealm() {
		StatelessRealm statelessRealm = new StatelessRealm();
		statelessRealm.setCachingEnabled(false);
		return statelessRealm;
	}

	@Bean("subjectFactory")
	public DefaultWebSubjectFactory  getStatelessDefaultSubjectFactory() {
		StatelessDefaultSubjectFactory  statelessDefaultSubjectFactory = new StatelessDefaultSubjectFactory();
		return statelessDefaultSubjectFactory;
	}

	@Bean("webTokenProcessor")
	public DefaultTokenProcessor getWebTokenProcessor() {
		DefaultTokenProcessor defaultTokenProcessor = new DefaultTokenProcessor();
		defaultTokenProcessor.setId("web");
		defaultTokenProcessor.setPath("/");
		defaultTokenProcessor.setExpr(expr);

		List<String> exactList = new ArrayList<>();
		exactList.add("tenantid");
		exactList.add("userId");
		exactList.add("userType");
		exactList.add("typeAlias");
		defaultTokenProcessor.setExacts(exactList);
		return defaultTokenProcessor;
	}

	@Bean("maTokenProcessor")
	public DefaultTokenProcessor getMaTokenProcessor() {
		DefaultTokenProcessor defaultTokenProcessor = new DefaultTokenProcessor();
		defaultTokenProcessor.setId("ma");
		defaultTokenProcessor.setPath("/");
		defaultTokenProcessor.setExpr(-1);

		List<String> exactList = new ArrayList<>();
		exactList.add("tenantid");
		exactList.add("userId");
		exactList.add("userType");
		exactList.add("typeAlias");
		defaultTokenProcessor.setExacts(exactList);
		return defaultTokenProcessor;
	}

	@Bean("tokenFactory")
	public TokenFactory getTokenFactory() {
		TokenFactory tokenFactory = new TokenFactory();

		List<ITokenProcessor> processList = new ArrayList<>();
		processList.add(getWebTokenProcessor());
		processList.add(getMaTokenProcessor());

		tokenFactory.setProcessors(processList);
		return tokenFactory;
	}

	/**
	 * 会话管理器
	 * 
	 * @return
	 */
	@Bean("sessionManager")
	public DefaultSessionManager getDefaultSessionManager() {
		DefaultSessionManager defaultSessionManager = new DefaultWebSessionManager();
		defaultSessionManager.setSessionValidationSchedulerEnabled(false);
		return defaultSessionManager;
	}

	/**
	 * 安全管理器
	 * 
	 * @return
	 */
	@Bean("securityManager")
	public DefaultWebSecurityManager getDefaultWebSecurityManager() {
		DefaultWebSecurityManager defaultWebSecurityManager = new DefaultWebSecurityManager();
		List<Realm> realms = new ArrayList<>();
		realms.add(getStatelessRealm());
		defaultWebSecurityManager.setRealms(realms);

		defaultWebSecurityManager.setSubjectFactory(getStatelessDefaultSubjectFactory());
		defaultWebSecurityManager.setSessionManager(getDefaultSessionManager());
		
		((DefaultSessionStorageEvaluator)((DefaultSubjectDAO)defaultWebSecurityManager.getSubjectDAO()).getSessionStorageEvaluator()).setSessionStorageEnabled(false);

		return defaultWebSecurityManager;
	}

	@Bean
	public MethodInvokingFactoryBean getMethodInvokingFactoryBean() {
		MethodInvokingFactoryBean methodInvokingFactoryBean = new MethodInvokingFactoryBean();
		methodInvokingFactoryBean.setStaticMethod("org.apache.shiro.SecurityUtils.setSecurityManager");
		methodInvokingFactoryBean.setArguments(getDefaultWebSecurityManager());
		return methodInvokingFactoryBean;
	}
	
	private StatelessAuthcFilter getStatelessAuthcFilter() {
		StatelessAuthcFilter statelessAuthcFilter = new StatelessAuthcFilter();
		statelessAuthcFilter.setSysid(sysid);
		statelessAuthcFilter.setLoginUrl(loginUrl);
		statelessAuthcFilter.setTokenFactory(getTokenFactory());
		return statelessAuthcFilter;
	}
	
	@Bean("statelessAuthcFilter")  
	public FilterRegistrationBean<StatelessAuthcFilter> registrationBean(){  
	   FilterRegistrationBean<StatelessAuthcFilter> registration = new FilterRegistrationBean<StatelessAuthcFilter>(getStatelessAuthcFilter());  
	   registration.setEnabled(false);//取消  Filter自动注册,不会添加到FilterChain中.  
	   return registration;  
	}  

	@Bean("shiroFilter")
	public ShiroFilterFactoryBean getShiroFilterFactoryBean() {
		ShiroFilterFactoryBean shiroFilterFactoryBean = new ShiroFilterFactoryBean();
		shiroFilterFactoryBean.setSecurityManager(getDefaultWebSecurityManager());
		shiroFilterFactoryBean.setLoginUrl(loginUrl);
		
		Map<String, String> filterChainDefinitionMap = new LinkedHashMap<>();
		filterChainDefinitionMap.put("/actuator/**", "anon"); //springboot admin 监控
 		filterChainDefinitionMap.put("/no_auth/**", "anon"); //rpc服务
		if(StringUtils.isNotBlank(redisSessionUrl)) {
			if (filterChainDefinitionsConfig != null) {
				Map<String, String[]> anonMap = filterChainDefinitionsConfig.getFilterChainDefinitions();
				String[] anons = anonMap.get("anons");
				if(anons != null && anons.length > 0) {
					for (String anon : anons) {
						filterChainDefinitionMap.put(anon, "anon");
					}
				}
				filterChainDefinitionMap.put("/**", "statelessAuthc");
			} else {
				filterChainDefinitionMap.put("/**", "statelessAuthc");
			}
		}else{
			filterChainDefinitionMap.put("/**", "anon");
		}
		shiroFilterFactoryBean.setFilterChainDefinitionMap(filterChainDefinitionMap);
		
		Map<String, Filter> filterMap = new HashMap<>();
		filterMap.put("statelessAuthc", getStatelessAuthcFilter());
		shiroFilterFactoryBean.setFilters(filterMap);

		return shiroFilterFactoryBean;
	}

	@Bean("lifecycleBeanPostProcessor")
	public static LifecycleBeanPostProcessor getLifecycleBeanPostProcessor() {
		LifecycleBeanPostProcessor lifecycleBeanPostProcessor = new LifecycleBeanPostProcessor();
		return lifecycleBeanPostProcessor;
	}

	@Bean("sessionJedisPool")
	public JedisPool getRedisPoolFactory() {
		RedisPoolFactory redisPoolFactory = new RedisPoolFactory();
		if(StringUtils.isBlank(redisSessionUrl)) {
			return redisPoolFactory.createJedisPool("direct://127.0.0.1:6379?poolSize=50&poolName=mypool");
		}else{
			return redisPoolFactory.createJedisPool(redisSessionUrl);
		}
	}

	@Bean("sessionMgr")
	public SessionManager getSessionManager() {
		SessionManager sessionManager = new SessionManager();
		sessionManager.setSessionJedisPool(getRedisPoolFactory());
		sessionManager.setSessionMutex(sessionMutex);
		return sessionManager;
	}

	public int getExpr() {
		return expr;
	}

	public void setExpr(int expr) {
		this.expr = expr;
	}

	public String getSysid() {
		return sysid;
	}

	public void setSysid(String sysid) {
		this.sysid = sysid;
	}

	public String getLoginUrl() {
		return loginUrl;
	}

	public void setLoginUrl(String loginUrl) {
		this.loginUrl = loginUrl;
	}

	public String getRedisSessionUrl() {
		return redisSessionUrl;
	}

	public void setRedisSessionUrl(String redisSessionUrl) {
		this.redisSessionUrl = redisSessionUrl;
	}

	public boolean isSessionMutex() {
		return sessionMutex;
	}

	public void setSessionMutex(boolean sessionMutex) {
		this.sessionMutex = sessionMutex;
	}

	public String getFilterExcludes() {
		return filterExcludes;
	}

	public void setFilterExcludes(String filterExcludes) {
		this.filterExcludes = filterExcludes;
	}

}
