package com.ejianc.business.bidprice.material.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.bidprice.material.bean.MaterialQuoteEntity;
import com.ejianc.business.bidprice.material.service.IMaterialQuoteService;
import com.ejianc.business.bidprice.material.vo.MaterialQuoteVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 报价记录
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("materialQuote")
@Api(tags = {"材料报价记录"})
public class MaterialQuoteController implements Serializable {
	private static final long serialVersionUID = -6476701269896286249L;
	private static final String RULE_CODE = "MATERIAL_QUOTE";
	private static final String BILL_CODE = "BT202302000002";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Resource
	private IBillTypeApi billTypeApi;
	@Resource
	private IBillCodeApi billCodeApi;
	@Resource
	private IOrgApi iOrgApi;
	@Resource
	private IMaterialQuoteService service;
	@Autowired
	private SessionManager sessionManager;


	/**
	 * 保存或更新
	 *
	 * @param materialQuoteVO 报价记录
	 *
	 * @return {@link CommonResponse}<{@link MaterialQuoteVO}>
	 */
	@ApiOperation(value = "保存或更新")
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<MaterialQuoteVO> saveOrUpdate(@RequestBody MaterialQuoteVO materialQuoteVO) {
		MaterialQuoteEntity entity = BeanMapper.map(materialQuoteVO, MaterialQuoteEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), materialQuoteVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		service.saveOrUpdate(entity, false);
		MaterialQuoteVO vo = BeanMapper.map(entity, MaterialQuoteVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}


	/**
	 * 查询详情
	 *
	 * @param id id
	 *
	 * @return {@link CommonResponse}<{@link MaterialQuoteVO}>
	 */
	@ApiOperation(value = "查询详情")
	@GetMapping(value = "/queryDetail")
	public CommonResponse<MaterialQuoteVO> queryDetail(Long id) {
		MaterialQuoteEntity entity = service.selectById(id);
		if (entity == null) {
			return CommonResponse.error("未查询到报价记录数据！");
		}
		MaterialQuoteVO vo = BeanMapper.map(entity, MaterialQuoteVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}


	/**
	 * 批量删除单据
	 *
	 * @param vos vos
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@ApiOperation(value = "批量删除单据")
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<MaterialQuoteVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (MaterialQuoteVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(MaterialQuoteVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}


	/**
	 * 查询列表
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link MaterialQuoteVO}>>
	 */
	@ApiOperation(value = "查询列表")
	@PostMapping(value = "/queryList")
	public CommonResponse<IPage<MaterialQuoteVO>> queryList(@RequestBody QueryParam param) {
		IPage<MaterialQuoteEntity> page = service.queryPage(param, false);
		IPage<MaterialQuoteVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), MaterialQuoteVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}

	/**
	 * 分享供应商报价二维码
	 *
	 * @param inquiryId 询价单id
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "/shareQrCode")
	public CommonResponse<JSONObject> shareQrCode(@RequestParam(value = "inquiryId") Long inquiryId) {
		return CommonResponse.success("供应商报价二维码生成成功！", service.generateQrCode(inquiryId));
	}


	/**
	 * 获取验证码
	 *
	 * @param inquiryId   询价单id
	 * @param phoneNumber 电话号码
	 * @param dialog      是否是登录弹窗
	 *
	 * @return {@link CommonResponse}<{@link Void}>
	 *
	 * @throws Exception 异常
	 */
	@GetMapping(value = "/fetchVerifyCode")
	public CommonResponse<Void> fetchVerifyCode(@RequestParam(value = "inquiryId") Long inquiryId,
	                                            @RequestParam(value = "phoneNumber") String phoneNumber,
	                                            @RequestParam(value = "dialog", required = false) Boolean dialog) throws Exception {
		service.fetchVerifyCode(inquiryId, phoneNumber, dialog);
		return CommonResponse.success("验证码发送成功！");
	}


	/**
	 * 登录
	 *
	 * @param phoneNumber 电话号码
	 * @param validate    验证码
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "/login")
	public CommonResponse<String> login(@RequestParam(value = "phoneNumber") String phoneNumber,
	                                    @RequestParam(value = "validate") String validate) {
		service.login(phoneNumber, validate);
		return CommonResponse.success("登录成功！", phoneNumber);
	}

}
