package com.ejianc.framework.cache.redis;

import java.io.Serializable;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springside.modules.nosql.redis.JedisTemplate;
import org.springside.modules.nosql.redis.JedisTemplate.PipelineAction;
import org.springside.modules.nosql.redis.JedisTemplate.PipelineActionNoResult;
import org.springside.modules.nosql.redis.JedisUtils;

import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.exceptions.JedisConnectionException;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.exceptions.JedisException;


/**
 * 缓存管理类，采用redis实现
 */
public class CacheManager {

    public static final String DEFAULT_CHARSET = "UTF-8";

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private Serializer serializer = new DefaultJDKSerializer();

    private JedisTemplate jedisTemplate;

    public JedisTemplate getJedisTemplate() {
        return jedisTemplate;
    }

    public void setJedisTemplate(JedisTemplate jedisTemplate) {
        this.jedisTemplate = jedisTemplate;
    }

    /**
     * 放置缓存
     *
     * @param key 缓存key
     * @param value 缓存对象
     */
    public <T extends Serializable> void set(final String key, final T value) {
        execute(new JedisActionNoResult() {
            @Override
            public void action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] valueBytes = serializer.marshalToByte(value);
                jedis.set(keyBytes, valueBytes);
            }
        });
    }

    /**
     * 放置指定有效期的缓存
     *
     * @param key 缓存key
     * @param value 缓存对象
     * @param timeout 缓存有效期
     */
    public <T extends Serializable> void setex(final String key, final T value, final int timeout) {
        execute(new JedisActionNoResult() {
            @Override
            public void action(Jedis jedis) {
                byte[] valueBytes = serializer.marshalToByte(value);
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                jedis.setex(keyBytes, timeout, valueBytes);
            }
        });
    }

    /**
     * 设置key值对应的缓存的有效期
     *
     * @param key
     * @param timeout
     */
    public void expire(final String key, final int timeout) {
        execute(new JedisActionNoResult() {
            @Override
            public void action(Jedis jedis) {
                jedis.expire(key, timeout);
            }
        });
    }

    public <T extends Serializable> void setAndExpireInPipeline(final String key, final T value, final int timeout) {
        jedisTemplate.execute(new PipelineActionNoResult() {
            @Override
            public void action(Pipeline pipeline) {
                byte[] valueBytes = serializer.marshalToByte(value);
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                pipeline.set(keyBytes, valueBytes);

                pipeline.expire(key, timeout);
            }
        });
    }

    public void piplineExecute(PipelineActionNoResult action) {
        execute(action);
    }

    @SuppressWarnings("unchecked")
	public <T> List<T> piplineExecute(PipelineAction action) {
        return (List<T>) execute(action);
    }

    /**
     * 判断key值对应的缓存是否存在
     *
     * @param key
     * @return 是否存在标志
     */
    public Boolean exists(final String key) {
        return execute(new JedisAction<Boolean>() {
            @Override
            public Boolean action(Jedis jedis) {
                return jedis.exists(key);
            }
        });
    }

    /**
     * 获取缓存对象
     *
     * @param key 缓存key
     * @return Object类型的缓存对象
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> T get(final String key) {
        return execute(new JedisAction<T>() {
            @Override
            public T action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                if (keyBytes == null) {
                    return null;
                }
                byte[] valueBytes = jedis.get(keyBytes);
                if (valueBytes == null) {
                    return null;
                } else {
                    return (T) serializer.unMarshal(valueBytes);
                }
            }
        });
    }

    /**
     * 获取HashMap中对应的子key的缓存值
     *
     * @param key map的key
     * @param fieldName 子key
     * @return hashmap的子项缓存
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> T hget(final String key, final String fieldName) {
        return execute(new JedisAction<T>() {
            @Override
            public T action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] fieldBytes = fieldName.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] attrBytes = jedis.hget(keyBytes, fieldBytes);
                if (attrBytes == null) {
                    return null;
                } else {
                    return (T) serializer.unMarshal(attrBytes);
                }
            }
        });
    }

    /**
     * 获取HashMap中对应的子key的缓存值
     *
     * @param key map的key
     * @param fieldName 子key
     * @return hashmap的子项缓存
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> List<T> hmget(final String key, final String... fieldName) {
        return execute(new JedisAction<List<T>>() {
            @Override
            public List<T> action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[][] fieldBytes = new byte[fieldName.length][];
                for (int i = 0; i < fieldName.length; i++) {
                    fieldBytes[i] = fieldName[i].getBytes(Charset.forName(DEFAULT_CHARSET));
                }
                List<byte[]> resultList = jedis.hmget(keyBytes, fieldBytes);
                if (resultList == null || resultList.isEmpty()) {
                    return null;
                } else {
                    return (List<T>) serializer.unMarshal(resultList);
                }
            }
        });
    }

    /**
     * 判断hash中field值对应的缓存是否存在
     *
     * @param key
     * @param field
     * @return 是否存在标志
     */
    public Boolean hexists(final String key, final String field) {
        return execute(new JedisAction<Boolean>() {
            @Override
            public Boolean action(Jedis jedis) {
                return jedis.hexists(key, field);
            }
        });
    }

    /**
     * 获取HashMap集合所有缓存的对象
     *
     * @param key map对应的key
     */
    public Map<byte[], byte[]> hgetAll(final String key) {
        return execute(new JedisAction<Map<byte[], byte[]>>() {
            @Override
            public Map<byte[], byte[]> action(Jedis jedis) {
                return jedis.hgetAll(key.getBytes(Charset.forName(DEFAULT_CHARSET)));
            }
        });
    }

    /**
     * 设置hashmap中的某一项
     */
    public <T extends Serializable> void hset(final String key, final String fieldName, final T value) {
        execute(new JedisActionNoResult() {
            @Override
            public void action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] fieldBytes = fieldName.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] valueBytes = serializer.marshalToByte(value);
                jedis.hset(keyBytes, fieldBytes, valueBytes);
            }
        });
    }


    /**
     * 设置hash
     */
    public <T extends Serializable> void hmset(final String key, final Map<String, T> valueMap) {
        execute(new JedisActionNoResult() {
            @Override
            public void action(Jedis jedis) {
                byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                Map<byte[], byte[]> byteDataMap = new HashMap<byte[], byte[]>();
                for (Map.Entry<String, T> entry : valueMap.entrySet()) {
                    byteDataMap.put(entry.getKey().getBytes(Charset.forName(DEFAULT_CHARSET)),
                            serializer.marshalToByte(entry.getValue()));
                }
                jedis.hmset(keyBytes, byteDataMap);
            }
        });
    }



    /**
     * 根据key删除缓存
     *
     * @param key 缓存的key
     */
    public void removeCache(String key) {
        if (StringUtils.isNotBlank(key)) {
            jedisTemplate.del(key);
        }
    }

    /**
     * 根据key和field删除hash缓存中的field
     *
     * @param key hash的key
     * @param key hash中的field
     */
    public void hdel(String key, String field) {
        if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(field)) {
            jedisTemplate.hdel(key, field);
        }
    }

    /**
     * 根据key和field删除hash缓存中的field
     *
     * @param key hash的key
     * @param key hash中的field
     */
    public void hdel(String key, String... field) {
        if (field!=null && field.length>0) {
            jedisTemplate.hdel(key, field);
        }
    }

    public void initNumForIncr(String key, long initValue) {
        jedisTemplate.set(key, String.valueOf(initValue));
    }

    public Long incr(final String key) {
        return jedisTemplate.incr(key);
    }

    public Long decr(final String key) {
        return jedisTemplate.decr(key);
    }

    public interface JedisAction<T> {
        T action(Jedis jedis);
    }

    public interface JedisActionNoResult {
        void action(Jedis jedis);
    }

    public <T> T execute(JedisAction<T> jedisAction) throws JedisException {
        Jedis jedis = null;
        boolean broken = false;
        try {
            jedis = jedisTemplate.getJedisPool().getResource();
            return jedisAction.action(jedis);
        } catch (JedisException e) {
            broken = handleJedisException(e);
            throw e;
        } finally {
            closeResource(jedis, broken);
        }
    }

    public void execute(JedisActionNoResult jedisAction) throws JedisException {
        Jedis jedis = null;
        boolean broken = false;
        try {
            jedis = jedisTemplate.getJedisPool().getResource();
            jedisAction.action(jedis);
        } catch (JedisException e) {
            broken = handleJedisException(e);
            throw e;
        } finally {
            closeResource(jedis, broken);
        }
    }

    public void execute(PipelineActionNoResult pipelineAction) throws JedisException {
        Jedis jedis = null;
        boolean broken = false;
        try {
            jedis = jedisTemplate.getJedisPool().getResource();
            Pipeline pipeline = jedis.pipelined();
            pipelineAction.action(pipeline);
            pipeline.sync();
        } catch (JedisException e) {
            broken = handleJedisException(e);
            throw e;
        } finally {
            closeResource(jedis, broken);
        }
    }

    public List<Object> execute(PipelineAction pipelineAction) throws JedisException {
        Jedis jedis = null;
        boolean broken = false;
        try {
            jedis = jedisTemplate.getJedisPool().getResource();
            Pipeline pipeline = jedis.pipelined();
            pipelineAction.action(pipeline);
            return pipeline.syncAndReturnAll();
        } catch (JedisException e) {
            broken = handleJedisException(e);
            throw e;
        } finally {
            closeResource(jedis, broken);
        }
    }

    protected boolean handleJedisException(JedisException jedisException) {
        if (jedisException instanceof JedisConnectionException) {
            logger.error("Redis connection " + jedisTemplate.getJedisPool().getAddress() + " lost.", jedisException);
        } else if (jedisException instanceof JedisDataException) {
            if ((jedisException.getMessage() != null) && (jedisException.getMessage().indexOf("READONLY") != -1)) {
                logger.error("Redis connection " + jedisTemplate.getJedisPool().getAddress() + " are read-only slave.",
                        jedisException);
            } else {
                return false;
            }
        } else {
            logger.error("Jedis exception happen.", jedisException);
        }
        return true;
    }

    protected void closeResource(Jedis jedis, boolean conectionBroken) {
        try {
            if (conectionBroken) {
                jedisTemplate.getJedisPool().returnBrokenResource(jedis);
            } else {
                jedisTemplate.getJedisPool().returnResource(jedis);
            }
        } catch (Exception e) {
            logger.error("return back jedis failed, will fore close the jedis.", e);
            JedisUtils.destroyJedis(jedis);
        }
    }

    public Serializer getSerializer() {
        return serializer;
    }

    public void setSerializer(Serializer serializer) {
        this.serializer = serializer;
    }
}
