package com.ejianc.business.contractbase.controller.api;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.contractbase.entity.TemplateEntity;
import com.ejianc.business.contractbase.entity.TemplateVersionEntity;
import com.ejianc.business.contractbase.service.ITemplateService;
import com.ejianc.business.contractbase.service.ITemplateVersionService;
import com.ejianc.business.contractbase.vo.TemplateVO;
import com.ejianc.business.contractbase.vo.TemplateVersionVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 模板API控制器
 *
 * @author CJ
 * @Description: 模板API控制器
 * @date 2021/11/11 11:39
 */
@RestController
@RequestMapping
public class TemplateApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ITemplateService templateService;

    @Autowired
    private ITemplateVersionService templateVersionService;

    /**
     * 根据分类Id查询分类下启用的合同模板信息
     *
     * @param categoryId 模板分类Id
     * @return
     */
    @GetMapping(value = "/api/template/getByCategoryId")
    public CommonResponse<TemplateVO> getByCategoryId(@RequestParam(value = "categoryId") Long categoryId) {
        QueryWrapper<TemplateEntity> query = new QueryWrapper<>();
        //设置分类Id
        query.eq("category_id", categoryId);
        //查询启用的模板
        query.eq("enable_status", 1);
        List<TemplateEntity> eList = templateService.list(query);
        if(CollectionUtils.isEmpty(eList)) {
            return CommonResponse.error("未查询到模板信息", null);
        }
        return CommonResponse.success("查询成功！", BeanMapper.map(eList.get(0), TemplateVO.class));
    }

    /**
     * 根据模板Id查询模板信息
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/api/template/queryDetailById")
    public CommonResponse<TemplateVO> queryDetailById(@RequestParam(value = "id") Long id) {
        TemplateVO resp = null;
        TemplateEntity tmpl = templateService.selectById(id);
       if(null != tmpl) {
           resp = BeanMapper.map(tmpl, TemplateVO.class);
       }
        return CommonResponse.success("查询成功！", resp);
    }

    /**
     * 根据模板Id和版本号查询对应版本模板信息
     *
     * @param templateId 模板Id
     * @param version 版本号
     * @return
     */
    @GetMapping(value = "/api/template/getTemplateVersionInfo")
    public CommonResponse<TemplateVersionVO> getTemplateVersionInfo(@RequestParam(value = "templateId") Long templateId,
                                                 @RequestParam(value = "version", required = false) Long version) {
        TemplateVersionEntity tmp = null;
        if(null == version) {
            TemplateEntity e = templateService.selectById(templateId);
            if(null == e) {
                return CommonResponse.error("未查询到值指定的模板信息！");
            }

            tmp = templateVersionService.selectById(e.getCurVersionId());
        } else {
            QueryWrapper<TemplateVersionEntity> query = new QueryWrapper<>();
            query.eq("template_id", templateId);
            query.eq("template_version", version);
            List<TemplateVersionEntity> vList = templateVersionService.list(query);
            if(CollectionUtils.isEmpty(vList)) {
                return CommonResponse.error("未查询到指定的模板信息！");
            }
            tmp = vList.get(0);
        }

        return CommonResponse.success("查询成功！", BeanMapper.map(tmp, TemplateVersionVO.class));
    }

    /**
     * 创建新的模板版本
     *
     * @param templateVersionVO
     * @return
     */
    @PostMapping(value = "/api/template/createNewTmplVersion")
    public CommonResponse<TemplateVersionVO> createNewTmplVersion(@RequestBody TemplateVersionVO templateVersionVO) {
       //保存模板版本信息
        TemplateVersionVO resp = templateVersionService.newTemplateVersion(templateVersionVO);
        return CommonResponse.success("操作成功！", resp);
    }

    /**
     * 根据模板Id查询模板编辑版本列表
     *
     * @param templateId
     * @return
     */
    @GetMapping(value = "/api/template/getHisListByTemplateId")
    public CommonResponse<List<TemplateVersionVO>> getHisListByTemplateId(@RequestParam String templateId,
                                                                          @RequestParam Integer start,
                                                                          @RequestParam Integer size) {
        List<TemplateVersionVO> resp = new ArrayList<>();
        if(null != start) {
            QueryWrapper<TemplateVersionEntity> countQuery = new QueryWrapper<>();
            countQuery.eq("template_id", templateId);
            countQuery.eq("dr", BaseVO.DR_UNDELETE);
            countQuery.eq("tenant_id", InvocationInfoProxy.getTenantid());

            int hisCount = templateVersionService.count(countQuery);
            if(hisCount < start) {
                return CommonResponse.success(resp);
            }
            if(hisCount < start+size) {
                size = hisCount - start;
            }
        }
         QueryWrapper<TemplateVersionEntity> listQuery = new QueryWrapper<>();
        listQuery.eq("template_id", templateId);
        listQuery.eq("dr", BaseVO.DR_UNDELETE);
        listQuery.eq("tenant_id", InvocationInfoProxy.getTenantid());
        listQuery.orderByDesc("template_version");
        if(null != start) {
            listQuery.last("limit " + start + "," + size);
        }
        List<TemplateVersionEntity> list = templateVersionService.list(listQuery);
        if(CollectionUtils.isNotEmpty(list)) {
            resp = BeanMapper.mapList(list, TemplateVersionVO.class);
        }

        return CommonResponse.success("查询成功！", resp);
    }
}
