package com.ejianc.business.contractbase.pool.settlepool.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.contractbase.pool.settlepool.bean.SettlePoolEntity;
import com.ejianc.business.contractbase.pool.settlepool.service.ISettlePoolService;
import com.ejianc.business.contractbase.pool.settlepool.vo.SettlePoolVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 结算池主实体---controller
 *
 * @author generator
 */
@RestController
@RequestMapping
public class SettlePoolApiController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ISettlePoolService service;

    @Autowired
    private IBillTypeApi billTypeApi;

    // 有合同结算
    private static final String BILL_CODE = "SETTLE_POOL_TEMP";

    // 无合同结算
    private static final String BILL_CODE_NO_CONTRACT = "SETTLE_POOL_NO_CONTRACT_TEMP";

    /**
     * 新增&更新
     *
     * @param saveOrUpdateVO
     * @return
     */
    @PostMapping(value = "/api/settlePool/saveOrUpdateSettle")
    public CommonResponse<SettlePoolVO> saveOrUpdate(@RequestBody SettlePoolVO saveOrUpdateVO) {
        logger.info("接收到结算单同步数据-{}", JSONObject.toJSONString(saveOrUpdateVO));
        SettlePoolVO vo = service.saveOrUpdateSettle(saveOrUpdateVO);
        return CommonResponse.success("结算单保存成功！", vo);
    }

    /**
     *  获取当前 合同下的累计结算金额
     * @param settlePoolVO
     * @return
     */
    @RequestMapping(value = "/api/settlePool/queryTotalMny", method = RequestMethod.POST)
    public CommonResponse<SettlePoolVO> queryTotalMny(@RequestBody SettlePoolVO settlePoolVO) {
        return CommonResponse.success("查询成功",service.queryTotalMny(settlePoolVO));
    }

    /**
     *  获取当前 合同下的累计结算金额++1
     * @param queryParam
     * @return
     */
    @RequestMapping(value = "/api/settlePool/queryContractsTotalSettleMny", method = RequestMethod.POST)
    public CommonResponse<Map<String, BigDecimal>> queryContractsTotalSettleMny(@RequestBody QueryParam queryParam) {
        return CommonResponse.success("查询成功",service.queryContractsTotalSettleMny(queryParam));
    }

    @RequestMapping(value = "/api/settlePool/queryNoContractTotalMny", method = RequestMethod.POST)
    public CommonResponse<JSONObject> queryNoContractTotalMny(@RequestBody SettlePoolVO settlePoolVO){
        return service.queryNoContractTotalMny(settlePoolVO);

    }


    /**
     * 批量新增&更新
     *
     * @param saveOrUpdateVOs
     * @return
     */
    @PostMapping(value = "/api/settlePool/saveOrUpdateBatch")
    public CommonResponse<List<SettlePoolVO>> saveOrUpdateBatch(@RequestBody List<SettlePoolVO> saveOrUpdateVOs) {
        logger.info("接收到结算单同步数据-{}", JSONObject.toJSONString(saveOrUpdateVOs));
        List<SettlePoolVO> vos = service.saveOrUpdateSettleBatch(saveOrUpdateVOs);
        return CommonResponse.success("结算单保存成功！", vos);
    }

    /**
     * 批量新增&更新---分布式事务版
     *
     * @param saveOrUpdateVOs
     * @return
     */
    @PostMapping(value = "/api/settlePool/saveOrUpdateBatchGlobal")
    public CommonResponse<List<SettlePoolVO>> saveOrUpdateBatchGlobal(@RequestBody List<SettlePoolVO> saveOrUpdateVOs) {
       return saveOrUpdateBatch(saveOrUpdateVOs);
    }


    @GetMapping(value = "/api/settlePool/getBySourceId")
    public CommonResponse<SettlePoolVO> getBySourceId(@RequestParam(value = "sourceId") Long sourceId) {
        SettlePoolVO vo = service.getBySourceId(sourceId);
        return CommonResponse.success("查询成功！", vo);
    }

    @PostMapping(value = "/api/settlePool/getBySourceIds")
    public CommonResponse<List<SettlePoolVO>> getBySourceIds(@RequestBody List<Long> sourceIds) {
        List<SettlePoolVO> vos = service.getBySourceIds(sourceIds);
        return CommonResponse.success("查询成功！", vos);
    }

    /**
     * 单条删除
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "/api/settlePool/deleteSettle")
    public CommonResponse<SettlePoolVO> delete(@RequestBody SettlePoolVO vo) {
        logger.info("接收到通知删除结算单数据-{}", JSONObject.toJSONString(vo));

        SettlePoolEntity settle = null;
        if(null != vo.getSourceId()) {
            settle = service.selectById(vo.getSourceId());
        }
        if(null == settle && null != vo.getId()) {
            settle = service.selectById(vo.getId());
        }

        if(null == settle) {
            logger.info("结算单-【{}】删除操作结束，在结算池中无匹配的数据！", JSONObject.toJSONString(vo));
            return CommonResponse.success("删除成功！");
        }

        //合同是否被其他单据引用
        CommonResponse<String> res = billTypeApi.checkQuote(null != settle.getContractId() ? "EJCBT202205000007" : "EJCBT202205000006", settle.getId());
        logger.info("平台返回查询被引用情况：{}", JSONObject.toJSONString(res));
        if(!res.isSuccess()) {
            return CommonResponse.error("结算单["+settle.getBillCode()+"]已被下游业务引用，不能删除！");
        }

        service.delBySourceId(settle.getId());
        return CommonResponse.success("删除成功！");
    }

    /**
     * 根据条件查询结算池列表
     *
     * @param QueryParam
     * @return
     */
    @PostMapping(value = "/api/settlePool/queryList")
    public CommonResponse<List<SettlePoolVO>> queryList(@RequestBody QueryParam QueryParam) {
        List<SettlePoolVO> resp = new ArrayList<>();
        List<SettlePoolEntity> entityList =  service.queryList(QueryParam);
        if(CollectionUtils.isNotEmpty(entityList)) {
            resp = BeanMapper.mapList(entityList, SettlePoolVO.class);
        }

        return CommonResponse.success(resp);
    }


    /**
     * 根据项目id查询结算池中结算类型为产值报量的累计产值金额
     *
     * @param projectId 项目id
     *
     * @return 累计产值金额 = “本期结算金额(含税)”之和
     */
    @GetMapping(value = "/api/settlePool/fetchTotalProductValueTaxMny")
    public CommonResponse<BigDecimal> fetchTotalProductValueTaxMny(@RequestParam(value = "projectId") Long projectId) {
        return CommonResponse.success("查询成功！", service.fetchTotalProductValueTaxMny(projectId));
    }

    /**
     * 根据合同id查询该合同的所有【结算池】数据，汇总“本期结算金额”之和
     *
     * @param contractIds 合同id集合
     *
     * @return 累计结算金额, Map< String, Object>
     */
    @PostMapping(value = "/api/settlePool/fetchTotalSettleTaxMny")
    public CommonResponse<Map<String, Object>> fetchTotalSettleTaxMny(@RequestBody Collection<Long> contractIds) {
        return CommonResponse.success("查询成功！", service.fetchTotalSettleTaxMny(contractIds));
    }

}
