package com.ejianc.framework.core.response;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.ejianc.framework.core.exception.IErrorEnum;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;

/**
 *
 * 统一封装返回给浏览器的数据结构
 *
 **/
public class CommonResponse<T> implements Serializable {

	private static final long serialVersionUID = 2045521669273048261L;
	public static final int SUCCESS = 0;
	public static final int FAILED = 1;
	private int code = 0; // 错误码
	private String msg = ""; // 错误消息
	
	@JsonInclude(JsonInclude.Include.NON_EMPTY)
	private Throwable cause; // 异常类
	private T data = null; // 请求获取到的数据
	// 返回给浏览器的数据，但是逻辑上不是对request的响应（对request的响应放到data里）
	// 使用场景：及时刷新token、返回当前最新系统状态、返回当前实时行情
	@JsonInclude(JsonInclude.Include.NON_EMPTY)
	private Map<String, Object> attachments = new HashMap<String, Object>();
	// 查询是否已经完成
	@JsonInclude(JsonInclude.Include.NON_EMPTY)
	private boolean thisTimeDone = true;
	
	@JsonIgnore
	public boolean isSuccess() {
		return this.getCode() == CommonResponse.SUCCESS;
	}

	/**
	 * 创建response返回结果，默认是否成功为true
	 */
	public CommonResponse() {
		this.code = CommonResponse.SUCCESS;
	}
	
	public static <T> CommonResponse<T> success(T data) {
		return success("success", data);
	}
	
	// 快速返回正常结果
	public static <T> CommonResponse<T> success() {
		CommonResponse<T> result = new CommonResponse<T>();
		result.setCode(SUCCESS);
		return result;
	}

	public static <T> CommonResponse<T> success(String msg) {
		CommonResponse<T> result = new CommonResponse<T>();
		result.setCode(SUCCESS);
		result.setMsg(msg);
		return result;
	}

	public static <T> CommonResponse<T> success(String msg, T data) {
		CommonResponse<T> result = new CommonResponse<T>();
		if (StringUtils.isNotBlank(msg))
			result.setMsg(msg);
		result.setCode(SUCCESS);
		result.setData(data);
		return result;
	}

	public static <T> CommonResponse<T> error(String msg) {
		return CommonResponse.error(msg, null);
	}

	public static <T> CommonResponse<T> error(int code, String msg) {
		return CommonResponse.error(code, msg, null);
	}

	public static <T> CommonResponse<T> error(String msg, Throwable cause) {
		return CommonResponse.error(FAILED, msg, cause);
	}

	public static <T> CommonResponse<T> error(int code, String msg, Throwable cause) {
		CommonResponse<T> result = new CommonResponse<T>();
		result.setCode(code);
		result.setMsg(msg);
		result.setCause(cause);
		return result;
	}

	public static <T> CommonResponse<T> error(IErrorEnum errorEnum) {
		return CommonResponse.error(errorEnum.getErrorCode(), errorEnum.getErrorMsg());
	}

	public int getCode() {
		return code;
	}

	public void setCode(int code) {
		this.code = code;
	}

	public String getMsg() {
		return msg;
	}

	public void setMsg(String msg) {
		this.msg = msg;
	}

	public Throwable getCause() {
		return cause;
	}

	public void setCause(Throwable cause) {
		this.cause = cause;
	}

	public T getData() {
		return data;
	}

	public void setData(T data) {
		this.data = data;
	}

	public Map<String, Object> getAttachments() {
		return attachments;
	}

	public void setAttachments(Map<String, Object> map) {
		if (map != null && map.size() > 0) {
			this.attachments.putAll(map);
		}
	}

	public void addAttachment(String key, Object att) {
		this.attachments.put(key, att);
	}

	public Object getAttachment(String key) {
		return attachments.get(key);
	}

	public boolean isThisTimeDone() {
		return thisTimeDone;
	}

	public void setThisTimeDone(boolean thisTimeDone) {
		this.thisTimeDone = thisTimeDone;
	}

	/**
	 * 自我复制，并重新设置 Data
	 * 
	 * @param data
	 * @return ServiceResult<D>
	 */
	public <D> CommonResponse<D> duplicate(D data) {
		CommonResponse<D> serviceResult = new CommonResponse<D>();
		serviceResult.setData(data);
		serviceResult.setCode(this.getCode());
		serviceResult.setMsg(this.getMsg());
		serviceResult.setCause(this.getCause());
		serviceResult.setAttachments(this.getAttachments());
		return serviceResult;
	}
}
