package com.ejianc.business.budget.utils;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;

import java.util.jar.JarEntry;
import java.util.jar.JarFile;


/**
 * 扫描包下的所有类
 * @author zengke
 * @version 1.0
 * @see ClassUtil
 * @since 2024/12/18 17:17
 */

public class ClassUtil {
    /**
     * 从包package中获取所有的Class
     * 
     * @param pack
     * @return
     */
    public static Set<Class<?>> getClzList(String pack) {

        // 第一个class类的集合
        Set<Class<?>> classes = new LinkedHashSet<Class<?>>();
        // 是否循环迭代
        boolean recursive = true;
        // 获取包的名字 并进行替换
        String packageName = pack;
        String packageDirName = packageName.replace('.', '/');
        // 定义一个枚举的集合 并进行循环来处理这个目录下的things
        Enumeration<URL> dirs;
        try {
            dirs = Thread.currentThread().getContextClassLoader().getResources(packageDirName);
            // 循环迭代下去
            while (dirs.hasMoreElements()) {
                // 获取下一个元素
                URL url = dirs.nextElement();
                // 得到协议的名称
                String protocol = url.getProtocol();
                // 如果是以文件的形式保存在服务器上
                if ("file".equals(protocol)) {
                    // System.err.println("file类型的扫描");
                    // 获取包的物理路径
                    String filePath = URLDecoder.decode(url.getFile(), "UTF-8");
                    // 以文件的方式扫描整个包下的文件 并添加到集合中
                    findAndAddClassesInPackageByFile(packageName, filePath, recursive, classes);
                } else if ("jar".equals(protocol)) {
                    // 如果是jar包文件
                    // 定义一个JarFile
                    // System.err.println("jar类型的扫描");
                    JarFile jar;
                    try {
                        // 获取jar
                        jar = ((JarURLConnection) url.openConnection()).getJarFile();
                        // 从此jar包 得到一个枚举类
                        Enumeration<JarEntry> entries = jar.entries();
                        // 同样的进行循环迭代
                        while (entries.hasMoreElements()) {
                            // 获取jar里的一个实体 可以是目录 和一些jar包里的其他文件 如META-INF等文件
                            JarEntry entry = entries.nextElement();
                            String name = entry.getName();
                            // 如果是以/开头的
                            if (name.charAt(0) == '/') {
                                // 获取后面的字符串
                                name = name.substring(1);
                            }
                            // 如果前半部分和定义的包名相同
                            if (name.startsWith(packageDirName)) {
                                int idx = name.lastIndexOf('/');
                                // 如果以"/"结尾 是一个包
                                if (idx != -1) {
                                    // 获取包名 把"/"替换成"."
                                    packageName = name.substring(0, idx).replace('/', '.');
                                }
                                // 如果可以迭代下去 并且是一个包
                                if ((idx != -1) || recursive) {
                                    // 如果是一个.class文件 而且不是目录
                                    if (name.endsWith(".class") && !entry.isDirectory()) {
                                        // 去掉后面的".class" 获取真正的类名
                                        String className = name.substring(packageName.length() + 1, name.length() - 6);
                                        try {
                                            // 添加到classes
                                            classes.add(Class.forName(packageName + '.' + className));
                                        } catch (ClassNotFoundException e) {
                                            // log
                                            // .error("添加用户自定义视图类错误
                                            // 找不到此类的.class文件");
                                            e.printStackTrace();
                                        }
                                    }
                                }
                            }
                        }
                    } catch (IOException e) {
                        // log.error("在扫描用户定义视图时从jar包获取文件出错");
                        e.printStackTrace();
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return classes;
    }

    /**
     * 以文件的形式来获取包下的所有Class
     * 
     * @param packageName
     * @param packagePath
     * @param recursive
     * @param classes
     */
    public static void findAndAddClassesInPackageByFile(String packageName, String packagePath, final boolean recursive,
            Set<Class<?>> classes) {
        // 获取此包的目录 建立一个File
        File dir = new File(packagePath);
        // 如果不存在或者 也不是目录就直接返回
        if (!dir.exists() || !dir.isDirectory()) {
            // log.warn("用户定义包名 " + packageName + " 下没有任何文件");
            return;
        }
        // 如果存在 就获取包下的所有文件 包括目录
        File[] dirfiles = dir.listFiles(new FileFilter() {
            // 自定义过滤规则 如果可以循环(包含子目录) 或则是以.class结尾的文件(编译好的java类文件)
            public boolean accept(File file) {
                return (recursive && file.isDirectory()) || (file.getName().endsWith(".class"));
            }
        });
        // 循环所有文件
        for (File file : dirfiles) {
            // 如果是目录 则继续扫描
            if (file.isDirectory()) {
                findAndAddClassesInPackageByFile(packageName + "." + file.getName(), file.getAbsolutePath(), recursive,
                        classes);
            } else {
                // 如果是java类文件 去掉后面的.class 只留下类名
                String className = file.getName().substring(0, file.getName().length() - 6);
                try {
                    // 添加到集合中去
                    // classes.add(Class.forName(packageName + '.' +
                    // className));
                    // 这里用forName有一些不好，会触发static方法，没有使用classLoader的load干净
                    classes.add(
                            Thread.currentThread().getContextClassLoader().loadClass(packageName + '.' + className));
                } catch (ClassNotFoundException e) {
                    // log.error("添加用户自定义视图类错误 找不到此类的.class文件");
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * 查询对应包下的枚举项集合
     * @param pack 包路径
     * @return 枚举集合
     */
    public static Map<String, List<Map<String,Object>>> getEnumMap(String pack){

        //查询该包下的所有类列表
        Set<Class<?>> clzList = ClassUtil.getClzList(pack);

        //创建一个Map集合 存储所有的枚举类中的 枚举项集合  key:枚举类的名称  value:枚举值列表
        Map<String, List<Map<String,Object>>> enumConstantMap = new HashMap<>();

        //遍历类列表
        clzList.stream()
                .filter(Class::isEnum) //过滤出所有的枚举类
                .forEach(enumClz -> {

                    //定义一个list集合用于存储枚举值、Map中存的是该枚举项中的属性及属性对应的值(例如 code = P1A,name=P1-A)
                    List<Map<String, Object>> enumValueList = new ArrayList<>();

                    //获取当前类的枚举实例对象列表
                    Object[] enumConstantList = enumClz.getEnumConstants();

                    //遍历枚举实例列表
                    for (Object enumConstant : enumConstantList) {

                        //定义一个map集合用于存储该枚举实例包含的值
                        Map<String, Object> enumFieldMap = new HashMap<>();

                        // 获取所有非静态字段（即实例字段 如code、name等并没有被static修饰、而枚举实例、$VALUES都有被static修饰故需要去掉）
                        Field[] fields = enumConstant.getClass().getDeclaredFields();
                        for (Field field : fields) {
                            if (!field.getName().equals("$VALUES") && !Modifier.isStatic(field.getModifiers())) {
                                field.setAccessible(true);
                                try {

                                    //当前字段名为定义的名称、而值是当前枚举实例对应这个名称的值
                                    enumFieldMap.put(field.getName(), field.get(enumConstant));
                                } catch (IllegalAccessException e) {

                                }
                            }
                        }

                        //将枚举字段Map集合 添加到枚举值列表
                        enumValueList.add(enumFieldMap);
                    }

                    //将枚举值列表添加到枚举Map集合中
                    enumConstantMap.put(enumClz.getSimpleName(),enumValueList);
                });

        //返回枚举Map集合
        return enumConstantMap;
    }


//    public static void main(String[] args) {
//
//        Set<Class<?>> classList = getClzList("com.ejianc.business.hk.enums");
//
//        System.out.println(classList);
//        Object[] ts =   classList.toArray();
//        for(Object t:ts){
//            Class<?> tt = (Class<?>) t;
//            System.out.println(tt.getName());
//        }
//    }
}