package com.ejianc.business.target.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.cost.utils.TreeNodeBUtil;
import com.ejianc.business.target.bean.TargetCostFinishDetailEntity;
import com.ejianc.business.target.bean.TargetCostFinishEntity;
import com.ejianc.business.target.service.ITargetCostFinishDetailService;
import com.ejianc.business.target.service.ITargetCostFinishService;
import com.ejianc.business.target.vo.TargetCostExecutionReportVO;
import com.ejianc.business.target.vo.TargetCostFinishDetailComparatorVO;
import com.ejianc.business.target.vo.TargetCostFinishDetailVO;
import com.ejianc.business.target.vo.TargetCostFinishVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 目标成本完工统计
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("targetCostFinish")
public class TargetCostFinishController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;

    private static final String BILL_CODE = "TARGET_COST_FINISH_CODE";

    @Autowired
    private ITargetCostFinishService service;

    @Autowired
    private ITargetCostFinishDetailService targetCostFinishDetailService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired(required = false)
    private RestHighLevelClient client;

    private final static Integer QUERY_TIMEOUT = 60;

    private final static String INDEX_NAME = "ejc_target_cost_execution";


    /**
     * @description: 目标成本执行情况
     *
     * @param projectId 
     * @return {@link CommonResponse<List<TargetCostExecutionReportVO>>}
     * @author songlx
     * @date: 2024/2/19
     */
    @RequestMapping(value = "/executionReport", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<TargetCostExecutionReportVO>> executionReport(@RequestParam Long projectId) {
        if (projectId == null) {
            throw new BusinessException("项目不能为空！");
        }
        List<TargetCostExecutionReportVO> list = this.getExecutionReport(projectId);
        return CommonResponse.success("查询目标成本执行情况成功！", list);
    }

    /**
     * @description: 目标成本执行情况--导出
     *
     * @param projectId
     * @return {@link CommonResponse<List<TargetCostExecutionReportVO>>}
     * @author songlx
     * @date: 2024/2/19
     */
    @RequestMapping(value = "/executionExport", method = RequestMethod.POST)
    @ResponseBody
    public void executionExport(@RequestParam Long projectId, HttpServletResponse response) {
        if (projectId == null) {
            throw new BusinessException("项目不能为空！");
        }
        List<TargetCostExecutionReportVO> list = this.getExecutionReport(projectId);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().exportWithTrans("execution-export.xlsx", beans, response);
    }

    private List<TargetCostExecutionReportVO> getExecutionReport(Long projectId) {
        List<TargetCostExecutionReportVO> list = new ArrayList<>();
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("projectId", projectId));
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));
        SearchRequest searchRequest = new SearchRequest(INDEX_NAME);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.query(boolQuery);
        sourceBuilder.sort("code", SortOrder.ASC);
        sourceBuilder.size(10000);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, TargetCostExecutionReportVO.class));
            }
        } catch (IOException e) {
            try {
                response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    list.add(JSON.parseObject(sourceAsString, TargetCostExecutionReportVO.class));
                }
            } catch (IOException e1) {
                logger.error(e1.getMessage());
            }
        }
        return list;
    }


    /**
     * @description: 获取项目累计完工总金额
     *
     * @param projectId
     * @param id
     * @return {@link CommonResponse< JSONObject>}
     * @author songlx
     * @date: 2024/2/5
     */
    @GetMapping("/getSumFinishTaxMny")
    @ResponseBody
    public CommonResponse<JSONObject> getSumFinishTaxMny(@RequestParam Long projectId, @RequestParam(required = false) Long id) {
        LambdaQueryWrapper<TargetCostFinishEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(TargetCostFinishEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()
        ));
        queryWrapper.eq(TargetCostFinishEntity::getProjectId, projectId);
        queryWrapper.ne(id != null, TargetCostFinishEntity::getId, id);
        List<TargetCostFinishEntity> list = service.list(queryWrapper);
        BigDecimal sumFinishTaxMny = BigDecimal.ZERO;
        if (ListUtil.isNotEmpty(list)) {
            sumFinishTaxMny = list.stream().map(TargetCostFinishEntity::getFinishTaxMny).reduce(ComputeUtil::safeAdd).orElse(BigDecimal.ZERO);
        }
        JSONObject data = new JSONObject();
        data.put("sumFinishTaxMny", sumFinishTaxMny);
        return CommonResponse.success("获取项目累计完工总金额成功！", data);
    }

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TargetCostFinishVO> saveOrUpdate(@RequestBody TargetCostFinishVO saveOrUpdateVO) {
        TargetCostFinishEntity entity = BeanMapper.map(saveOrUpdateVO, TargetCostFinishEntity.class);
        Long id = entity.getId();
        CommonResponse<String> checkResponse = this.checkSave(id, entity.getProjectId());
        if (!checkResponse.isSuccess()) {
            return CommonResponse.error(checkResponse.getMsg());
        }

        if (id == null || id == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        service.saveOrUpdate(entity, false);
        List<TargetCostFinishDetailEntity> targetCostFinishDetailList = entity.getTargetCostFinishDetailList();
        if (CollectionUtils.isNotEmpty(targetCostFinishDetailList)) {
            //维护父子级关系
            Map<String, Long> idMap = new HashMap<>();
            for (TargetCostFinishDetailEntity cdEntity : targetCostFinishDetailList) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (TargetCostFinishDetailEntity cdEntity : targetCostFinishDetailList) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            targetCostFinishDetailService.saveOrUpdateBatch(targetCostFinishDetailList, targetCostFinishDetailList.size());
        }
        TargetCostFinishVO vo = BeanMapper.map(entity, TargetCostFinishVO.class);
        List<TargetCostFinishDetailVO> list = BeanMapper.mapList(targetCostFinishDetailList, TargetCostFinishDetailVO.class);
        Collections.sort(list, new TargetCostFinishDetailComparatorVO());
        vo.setTargetCostFinishDetailList(TreeNodeBUtil.buildTree(list));
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @description: 保存校验
     *
     * @param id
     * @param projectId
     * @return {@link CommonResponse< String>}
     * @author songlx
     * @date: 2024/2/19
     */
    @RequestMapping(value = "/checkSave", method = RequestMethod.GET)
    @ResponseBody
    private CommonResponse<String> checkSave(Long id, Long projectId) {
        LambdaQueryWrapper<TargetCostFinishEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.notIn(TargetCostFinishEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()
        ));
        queryWrapper.eq(TargetCostFinishEntity::getProjectId, projectId);
        queryWrapper.ne(id != null, TargetCostFinishEntity::getId, id);
        int count = service.count(queryWrapper);
        if (count > 0) {
            return CommonResponse.error("该项目已存在非生效态的单据，不允许新增！");
        }
        return CommonResponse.success();
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<TargetCostFinishVO> queryDetail(Long id) {
        TargetCostFinishEntity entity = service.selectById(id);
        TargetCostFinishVO vo = BeanMapper.map(entity, TargetCostFinishVO.class);
        List<TargetCostFinishDetailVO> targetCostFinishDetailList = vo.getTargetCostFinishDetailList();
        if (CollectionUtils.isNotEmpty(targetCostFinishDetailList)) {
            for (TargetCostFinishDetailVO cdEntity : targetCostFinishDetailList) {
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                cdEntity.setRowState("edit");
            }
            //实现排序
            Collections.sort(targetCostFinishDetailList, new TargetCostFinishDetailComparatorVO());
            vo.setTargetCostFinishDetailList(TreeNodeBUtil.buildTree(targetCostFinishDetailList));
        }
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<TargetCostFinishVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (TargetCostFinishVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
            }
        }
        service.removeByIds(vos.stream().map(TargetCostFinishVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<TargetCostFinishVO>> queryList(@RequestBody QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("billCode", "projectName", "employeeName", "departmentName"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        /** 数据隔离 本下 没有组织orgId的删除下面代码-------------开始 */
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }

        IPage<TargetCostFinishEntity> page = service.queryPage(param, false);
        IPage<TargetCostFinishVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), TargetCostFinishVO.class));

        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("billCode", "projectName", "employeeName", "departmentName"));
        param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<TargetCostFinishEntity> list = service.queryList(param);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().exportWithTrans("TargetCostFinish-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refTargetCostFinishData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<TargetCostFinishVO>> refTargetCostFinishData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                                             String condition,
                                                                             String searchObject,
                                                                             String searchText) {
        QueryParam param = new QueryParam();
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
        }

        IPage<TargetCostFinishEntity> page = service.queryPage(param, false);
        IPage<TargetCostFinishVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), TargetCostFinishVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }
}
