package com.ejianc.certify.utils;

import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

/**
 * http工具类
 */
public class HttpClientTool {

    private static final Logger log = LoggerFactory.getLogger(HttpClientTool.class);

    /**
     * 发送http+get请求
     * @param url
     * @return 返回结果
     * @throws Exception
     */
    public static String sendHttpGet(String url) throws Exception {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        return doGet(url,httpClient);
    }

    /**
     * 发送https+get请求，绕过证书
     * @param url 请求地址
     * @return 返回结果
     * @throws Exception
     */
    public static String sendHttpsGet(String url) throws Exception {
        CloseableHttpClient httpClient = createIgnoreVerifyHttpClient();
        //CloseableHttpClient httpClient = HttpClients.createDefault();
        return doGet(url,httpClient);
    }

    /**
     * 发送http+post请求
     * @param url 请求地址
     * @param params 请求参数 json字符串
     * @return 返回结果
     * @throws Exception
     */
    public static String sendHttpPost(String url, String params) throws Exception {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        return doPost(httpClient, url, params);
    }

    /**
     * 发送https+post请求
     * @param url 请求地址
     * @param params 请求参数 json字符串
     * @return 返回结果
     * @throws Exception
     */
    public static String sendHttpsPost(String url, String params) throws Exception {
        CloseableHttpClient httpClient = createIgnoreVerifyHttpClient();
        //CloseableHttpClient httpClient = HttpClients.createDefault();
        return doPost(httpClient, url, params);

    }

    /**
     * 封装get请求方式的处理
     */
    private static String doGet(String url,CloseableHttpClient httpClient) throws Exception {
        log.info("HGet请求url={}", url);
        HttpGet httpGet = new HttpGet(url);
        return execute(httpClient,httpGet);
    }

    /**
     * 封装post请求方式的处理
     */
    private static String doPost(CloseableHttpClient httpClient, String url, String params) throws Exception {
        log.info("Post请求url：{}", url);
        log.info("Post请求params：{}", params);
        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type", "application/json; charset=utf-8");
        httpPost.setEntity(new StringEntity(params, Charset.forName("UTF-8")));//发送端进行设置编码
        return execute(httpClient,httpPost);
    }

    /**
     * 执行发送
     */
    private static String execute(CloseableHttpClient httpClient, HttpRequestBase requestBase) throws Exception {
        String result = null;
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();
            log.info("HttpClient响应码={}", statusCode);
            if (statusCode == 200) {
                result = EntityUtils.toString(response.getEntity(),"utf-8");
            } else {
                result = String.valueOf(statusCode);
                log.error("HttpClient请求失败，错误码={}", statusCode);
            }
        } catch (Exception e) {
            log.error("HttpClient请求异常：", e);
        } finally {
            if (null != httpClient) {
                httpClient.close();
            }
            if (null != response) {
                response.close();
            }
        }
        log.info("HttpClient请求结果：{}", result);
        return result;
    }

    /**
     * 绕过验证
     *
     * @return
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static CloseableHttpClient createIgnoreVerifyHttpClient() throws Exception {
        SSLContext sslContext = SSLContext.getInstance("TLS");
        // 实现一个X509TrustManager接口
        X509TrustManager trustManager = new X509TrustManager() {
            @Override
            public void checkClientTrusted(
                    X509Certificate[] paramArrayOfX509Certificate,
                    String paramString) throws CertificateException {
            }

            @Override
            public void checkServerTrusted(
                    X509Certificate[] paramArrayOfX509Certificate,
                    String paramString) throws CertificateException {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return null;
            }
        };
        sslContext.init(null, new TrustManager[]{trustManager}, null);
        Registry<ConnectionSocketFactory> socketFactoryRegistry =
                RegistryBuilder.<ConnectionSocketFactory>create()
                        .register("http", PlainConnectionSocketFactory.INSTANCE)
                        .register("https", new SSLConnectionSocketFactory(sslContext)).build();
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
        CloseableHttpClient httpClient = HttpClients.custom().setConnectionManager(connManager).build();
        return httpClient;
    }
    /**
     * 绕过验证，带basic认证
     *
     * @return
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static CloseableHttpClient createIgnoreVerifyHttpClient(String user,String password) throws Exception {
        SSLContext sslContext = SSLContext.getInstance("TLS");
        // 实现一个X509TrustManager接口
        X509TrustManager trustManager = new X509TrustManager() {
            @Override
            public void checkClientTrusted(
                    X509Certificate[] paramArrayOfX509Certificate,
                    String paramString) throws CertificateException {
            }

            @Override
            public void checkServerTrusted(
                    X509Certificate[] paramArrayOfX509Certificate,
                    String paramString) throws CertificateException {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return null;
            }
        };
        sslContext.init(null, new TrustManager[]{trustManager}, null);
        Registry<ConnectionSocketFactory> socketFactoryRegistry =
                RegistryBuilder.<ConnectionSocketFactory>create()
                        .register("http", PlainConnectionSocketFactory.INSTANCE)
                        .register("https", new SSLConnectionSocketFactory(sslContext)).build();
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);

        CredentialsProvider provider = new BasicCredentialsProvider();
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(user, password);
        provider.setCredentials(AuthScope.ANY, credentials);

        CloseableHttpClient httpClient = HttpClients.custom().setDefaultCredentialsProvider(provider).setConnectionManager(connManager).build();
        return httpClient;
    }
    /**
     * 发送https+post请求，绕过证书，带basic认证
     * @param url
     * @param params
     * @return
     * @throws Exception
     */
    public static String sendHttpsPost(String url, String params,String user,String password) throws Exception {
        CloseableHttpClient httpClient = createIgnoreVerifyHttpClient(user,password);
        //CloseableHttpClient httpClient = HttpClients.createDefault();
        return doPost(httpClient, url, params);

    }

    /**
     * 向指定 URL 发送POST方法的请求（发送值）
     * @param url 发送请求的 URL
     * @param paramJson json串格式的报文
     * @param username 用户名
     * @param password 密码
     * @return 所代表远程资源的响应结果(值类型为Json对象)
     */
    public static String sendJsonPost(String url, String paramJson, String username,String password) throws Exception{
        OutputStreamWriter out = null;
        BufferedReader in = null;
        StringBuffer result = new StringBuffer();
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            URLConnection conn = realUrl.openConnection();
            // 设置通用的请求属性
            conn.setRequestProperty("accept", "*/*");
            conn.setRequestProperty("connection", "Keep-Alive");
            conn.setRequestProperty("user-agent",  "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1;SV1)");
            //设置请求字符集  防止请求参数因字符集原因（非UTF-8）导致调用接口不一致报错（数据类型不一致）
            conn.setRequestProperty("Content-Type", "application/json;charset=UTF-8");
            addHeaderBasic(username,password,conn);//Http添加Basic认证
            //addHeaderApiKey(esbcode,conn);//Http添加apikey认证
            conn.setConnectTimeout(15000);/** 设置连接主机服务器超时时间：15000毫秒 */
            conn.setReadTimeout(60000);/** 设置读取远程返回的数据时间：60000毫秒 */
            //发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);
            // 获取URLConnection对象对应的输出流
            out = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");
            // 发送请求参数
            out.write(paramJson);
            // flush输出流的缓冲
            out.flush();
            // 定义BufferedReader输入流来读取URL的响应
            in = new BufferedReader(new InputStreamReader(conn.getInputStream(), "UTF-8"));
            String line;
            while ((line = in.readLine()) != null) {
                result.append(line);
            }
        } catch (Exception e) {
            throw e;
        }
        //使用finally块来关闭输出流、输入流
        finally{
            try{
                if(out!=null){
                    out.close();
                }
                if(in!=null){
                    in.close();
                }
            }
            catch(IOException ex){
                ex.printStackTrace();
            }
        }
        if(result!=null && result.length()>0)//转换为Json对象输出
            return result.toString();
            //return JSONObject.parseObject(result.toString());
        else
            return null;
    }
    /**
     * 向指定 URL 发送POST方法的请求（发送值）
     * @param url 发送请求的 URL
     * @param paramJson json串格式的报文
     * @return 所代表远程资源的响应结果(值类型为Json对象)
     */
    public static String sendJsonPostApikey(String url, String paramJson, String apikey) throws Exception{
        OutputStreamWriter out = null;
        BufferedReader in = null;
        StringBuffer result = new StringBuffer();
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            URLConnection conn = realUrl.openConnection();
            // 设置通用的请求属性
            conn.setRequestProperty("accept", "*/*");
            conn.setRequestProperty("connection", "Keep-Alive");
            conn.setRequestProperty("user-agent",  "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1;SV1)");
            //设置请求字符集  防止请求参数因字符集原因（非UTF-8）导致调用接口不一致报错（数据类型不一致）
            conn.setRequestProperty("Content-Type", "application/json;charset=UTF-8");
            //addHeaderBasic(username,password,conn);//Http添加Basic认证
            addHeaderApiKey(apikey,conn);//Http添加apikey认证
            conn.setConnectTimeout(15000);/** 设置连接主机服务器超时时间：15000毫秒 */
            conn.setReadTimeout(60000);/** 设置读取远程返回的数据时间：60000毫秒 */
            //发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);
            // 获取URLConnection对象对应的输出流
            out = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");
            // 发送请求参数
            out.write(paramJson);
            // flush输出流的缓冲
            out.flush();
            // 定义BufferedReader输入流来读取URL的响应
            in = new BufferedReader(new InputStreamReader(conn.getInputStream(), "UTF-8"));
            String line;
            while ((line = in.readLine()) != null) {
                result.append(line);
            }
        } catch (Exception e) {
            throw e;
        }
        //使用finally块来关闭输出流、输入流
        finally{
            try{
                if(out!=null){
                    out.close();
                }
                if(in!=null){
                    in.close();
                }
            }
            catch(IOException ex){
                ex.printStackTrace();
            }
        }
        if(result!=null && result.length()>0)//转换为Json对象输出
            return result.toString();
            //return JSONObject.parseObject(result.toString());
        else
            return null;
    }
    /**
     * Http添加Basic认证
     * @param username
     * @param password
     * @param urlConn
     */
    public static void addHeaderBasic(String username,String password,final URLConnection urlConn){
        String basic=username+":"+password;
        urlConn.addRequestProperty("Authorization", "Basic "+Base64Util.getBase64Encoder().encode(basic.getBytes()));
    }

    public static void addHeaderApiKey(String appkey,final URLConnection urlConn){
        urlConn.addRequestProperty("APPKEY", appkey);
    }
}