package com.ejianc.business.doc.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.doc.bean.DocCategoryEntity;
import com.ejianc.business.doc.bean.ProjectDocEntity;
import com.ejianc.business.doc.service.IDocCategoryService;
import com.ejianc.business.doc.service.IProjectDocService;
import com.ejianc.business.doc.vo.ProjectDocVO;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.metadata.api.IMdProjectApi;
import com.ejianc.foundation.metadata.vo.MdReferVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.google.gson.Gson;
import net.bytebuddy.asm.Advice;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2020/11/28 14:00
 */
@RestController
@RequestMapping(value = "/docManage/")
public class DocManageController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IProjectDocService projectDocService;

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private IMdProjectApi mdProjectApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectApi projectApi;

    private final String COMMON_REQ_URL = "/commonstate/queryBillDetail";

    @Autowired
    private IDocCategoryService docCategoryService;

  @PostMapping(value = "pageList")
    public CommonResponse<IPage<ProjectDocVO>> pageList(@RequestBody Map<String, String> params) {
        QueryParam queryParam = new QueryParam();
        Integer pageSize = Integer.valueOf(params.get("pageSize"));
        Integer pageIndex = Integer.valueOf(params.get("pageIndex"));
        Long categoryId = Long.valueOf(params.get("categoryId"));
        String filterType = params.get("filterType");
        String searchText = null != params.get("searchText") ? params.get("searchText") : null;

        List<Long> categoryIds = new ArrayList<>();
        Map<Long, String> categoryIdNameMap = new HashMap<>();

        if("1".equals(filterType)) {
            List<DocCategoryEntity> categoryList = docCategoryService.findChildrenByPid(categoryId);
            categoryList.forEach(category -> {
                categoryIdNameMap.put(category.getId(), category.getCategoryName());
                categoryIds.add(category.getId());
            });
        } else {
            categoryIds.add(categoryId);
            DocCategoryEntity categoryEntity = docCategoryService.getById(categoryId);
            categoryIdNameMap.put(categoryEntity.getId(), categoryEntity.getCategoryName());
        }

        IPage<ProjectDocVO> respPage = new Page<>();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getFuzzyFields().add("fileName");
        queryParam.getFuzzyFields().add("superviseUserName");
        queryParam.getOrderMap().put("superviseTime", QueryParam.DESC); //按归档时间倒序排列
        queryParam.getParams().put("docCategoryId", new Parameter(QueryParam.IN, categoryIds));

        CommonResponse<List<OrgVO>> orgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!orgResp.isSuccess()) {
            return CommonResponse.error("查询失败，获取组织信息失败！");
        }
        List<OrgVO> orgVOList = orgResp.getData();
        if(CollectionUtils.isEmpty(orgVOList)) {
            respPage.setRecords(new ArrayList<>());
            respPage.setCurrent(queryParam.getPageIndex());
            respPage.setSize(queryParam.getPageSize());
            respPage.setTotal(0);
            respPage.setPages(0);
            return CommonResponse.success(respPage);
        }

        //根据项目组织权限过滤
        IPage<ProjectDocEntity> pageData = projectDocService.queryPageList(pageSize, pageIndex, InvocationInfoProxy.getTenantid(), categoryIds, StringUtils.isBlank(searchText) ? null : searchText, orgVOList.stream().map(org -> org.getId()).collect(Collectors.toList()));

        List<ProjectDocVO> records = BeanMapper.mapList(pageData.getRecords(), ProjectDocVO.class);
        if(CollectionUtils.isNotEmpty(records)) {
            records.forEach(doc -> {
                doc.setDocCategoryName(categoryIdNameMap.get(doc.getDocCategoryId()));
            });
        }
        respPage.setRecords(records);
        respPage.setCurrent(pageData.getCurrent());
        respPage.setSize(pageData.getSize());
        respPage.setTotal(pageData.getTotal());
        respPage.setPages(pageData.getPages());

        return CommonResponse.success(respPage);
    }

    @PostMapping(value = "saveProjectDoc")
    public CommonResponse<String> saveOrUpdate(@RequestBody List<ProjectDocVO> projectDocVOs) {
        List<ProjectDocEntity> saveEntitys = BeanMapper.mapList(projectDocVOs, ProjectDocEntity.class);
        projectDocService.saveOrUpdateBatch(saveEntitys, saveEntitys.size());

        return CommonResponse.success("操作成功！");
    }

    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<Long> ids) {
      projectDocService.delArchiveDocs(ids, true);
        return CommonResponse.success("删除成功！");
    }

    @GetMapping(value = "queryDetail")
    public CommonResponse<ProjectDocVO> queryDetail(@RequestParam(value = "id") Long id) {
        ProjectDocEntity projectDocEntity = projectDocService.selectById(id);

        return CommonResponse.success(BeanMapper.map(projectDocEntity, ProjectDocVO.class));
    }

    /**
     * 附件归档
     * 1、根据billType找到单据对应工程应用服务
     * 2、根据sourceId找到单据详情信息
     * 3、获取到组织、项目信息并设置到归档信息中
     * 4、保存归档信息并将对应附件信息设置为已归档
     *
     * @param projectDocVOs
     * @return
     */
    @PostMapping(value = "archiveFiles")
    public CommonResponse<String> archiveFiles(@RequestBody List<ProjectDocVO> projectDocVOs) {
        ProjectDocVO tmpVo = projectDocVOs.get(0);
        String billType = tmpVo.getBillType();
        String orgIdField = tmpVo.getOrgIdField();
        String projectField = tmpVo.getProjectIdField();
        String billCodeField = tmpVo.getBillCodeField();
        Long sourceId = tmpVo.getSourceId();
        Long orgId = null, projectId = null;
        String orgName = null, projectName = null;
        String billCode = null, billTypeName = null;

        //1、获取对应的单据类型
        CommonResponse<BillTypeVO> billTypeResp = billTypeApi.getByCode(billType);
        if(!billTypeResp.isSuccess()) {
            return CommonResponse.error("归档失败，查询对应单据类型信息失败：" + billTypeResp.getMsg());
        }
        billTypeName = billTypeResp.getData().getBillName();

        //2、根据billType找到单据对应工程应用服务
        CommonResponse<MdReferVO> mdReferResp = billTypeApi.queryMetadataByBillType(billType);
        if(!mdReferResp.isSuccess()) {
            return CommonResponse.error("归档失败，查询元数据信息失败！");
        }
        MdReferVO mdReferVO = mdReferResp.getData();
        // 3、 获取单据详情
        JSONObject detail = reqResult(mdReferVO.getProjectName(), mdReferVO.getTableName(), sourceId, mdReferVO.getMetadataId());
        //4、获取到组织、项目信息并设置到归档信息中
        orgId = null != detail.get(orgIdField) ? Long.valueOf(detail.get(orgIdField).toString()) : null;
        projectId = null != detail.get(projectField) ? Long.valueOf(detail.get(projectField).toString()) : null;
        billCode = null != detail.get(billCodeField) ? detail.get(billCodeField).toString() : "";
        if(null == orgId && null == projectId) {
            return CommonResponse.error("归档失败，从业务单据中获取组织、项目信息失败！");
        }
        if(null != projectId) {
            List<Long> proIds = new ArrayList<>();
            proIds.add(projectId);
            CommonResponse<List<ProjectRegisterVO>> projectResp = projectApi.queryProjectByIds(proIds);
            if(!projectResp.isSuccess()) {
                logger.error("【附件归档】根据项目Id{}获取项目信息失败：{}", projectId, projectResp.getMsg());
                return CommonResponse.error("归档失败，获取项目信息失败！");
            }
            List<ProjectRegisterVO> projects = projectResp.getData();
            if(CollectionUtils.isEmpty(projects)) {
                logger.error("【附件归档】：根据项目Id{}查询不到匹配的项目信息", projectId);
                return CommonResponse.error("归档失败，系统中没有匹配的项目信息！");
            }
            ProjectRegisterVO project = projects.get(0);
            orgId = project.getProjectDepartmentId();
            orgName = project.getName();
            projectId = project.getId();
            projectName = project.getName();
        } else {
            CommonResponse<OrgVO> orgResp = orgApi.getOneById(orgId);
            if(!orgResp.isSuccess()) {
                logger.error("【附件归档】根据组织Id{}获取组织信息失败：{}", orgId, orgResp.getMsg());
                return CommonResponse.error("归档失败，获取组织信息失败！");
            }
            OrgVO orgvo = orgResp.getData();
            orgId = orgvo.getId();
            orgName = orgvo.getName();
        }
        List<Long> attachIds = new ArrayList<>();
        for(ProjectDocVO doc : projectDocVOs) {
            doc.setOrgId(orgId);
            doc.setOrgName(orgName);
            doc.setProjectId(projectId);
            doc.setProjectName(projectName);
            doc.setBillSymbol(billTypeName + (StringUtils.isNotBlank(billCode) ? ("-"+billCode) : ""));
            attachIds.add(doc.getFileId());
        }

        projectDocService.archiveFiles(projectDocVOs, attachIds);

        return CommonResponse.success("归档成功！");
    }

    private JSONObject reqResult(String projectName, String tableName, Long dataId, Long metaDataId) {
        Gson gson = new Gson();
        Map<String, Object> paramterMap = new HashMap<>();
        paramterMap.put("billId", dataId);
        paramterMap.put("tableName", tableName);
        paramterMap.put("metadataId", metaDataId);

        String paramterStr = gson.toJson(paramterMap);
        String url = BASE_HOST + projectName + COMMON_REQ_URL;
        logger.info("【附件归档】查询单据详情数据：【url-{}】, 【param-{}】", url, paramterStr);

        try {
            String result = ReferHttpClientUtils.postByJson(url, paramterStr);
            logger.info("【附件归档】查询单据详情数据返回的结果：---------------"+result);
            if(null == JSONObject.parseObject(result, Map.class).get("code")) {
                //返回结果非CommonResponse时
                throw new BusinessException("查询单据详情数据异常：" + result);
            }

            CommonResponse resp  = JSONObject.parseObject(result, CommonResponse.class);
            if(!resp.isSuccess()) {
                throw new BusinessException("查询单据详情数据异常：" + resp.getMsg());
            }
            return (JSONObject) resp.getData();
        } catch (Exception e) {
            logger.error("【附件归档】查询单据详情数据异常：", e);
            throw new BusinessException("查询单据详情数据异常：", e);
        }
    }

    /**
     * 检查文档对应单据是否存在
     *
     * @param id
     * @return
     */
    @GetMapping(value = "checkBill")
    public CommonResponse<String> checkBill(@RequestParam(value = "fileId") Long fileId) {

        //1、查询到对应的附件信息
        CommonResponse<AttachmentVO> fileResp = attachmentApi.queryAttachmentDetail(fileId);
        if(!fileResp.isSuccess()) {
            return CommonResponse.error(fileResp.getMsg());
        }
        AttachmentVO attach = fileResp.getData();

        //2、根据billType找到单据对应工程应用服务
        CommonResponse<MdReferVO> mdReferResp = billTypeApi.queryMetadataByBillType(attach.getBillType());
        if(!mdReferResp.isSuccess()) {
            return CommonResponse.error("获取业务单据元数据信息失败！");
        }

        MdReferVO mdReferVO = mdReferResp.getData();
        // 3、 获取单据详情
        JSONObject detail = reqResult(mdReferVO.getProjectName(), mdReferVO.getTableName(), attach.getSourceId(), mdReferVO.getMetadataId());
        if(null == detail) {
            return CommonResponse.error("业务单据不存在或已被删除！");
        }

        return CommonResponse.success("文档对应单据详情可查看！");
    }
}
