package com.ejianc.business.equipment.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.equipment.bean.PurchaseSettlementEntity;
import com.ejianc.business.equipment.service.IPurchaseSettlementService;
import com.ejianc.business.equipment.vo.PurchaseSettlementRecordVO;
import com.ejianc.business.equipment.vo.PurchaseSettlementVO;
import com.ejianc.business.equipment.vo.RentSettlementVO;
import com.ejianc.business.finance.api.IPayContractApi;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.util.ParamUtil;
import com.ejianc.business.equipment.vo.ParamsCheckVO;
import com.ejianc.business.finance.vo.SumPayMnyVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 设备合同结算表 前端控制器
 * </p>
 *
 * @author sunyj
 * @since 2020-06-04
 */
@RestController
@RequestMapping("purchaseSettlement")
public class PurchaseSettlementController {

    @Autowired
    private IPurchaseSettlementService purchaseSettlementService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IPayContractApi payContractApi;

    /**
     * 新增或者修改
     *
     * @param purchaseSettlementVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> saveOrUpdate(@RequestBody PurchaseSettlementVO purchaseSettlementVo) {
        return purchaseSettlementService.saveOrUpdate(purchaseSettlementVo);
    }

    /**
     * 根据主键ID查询供方详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> queryDetail(@RequestParam Long id) {
        PurchaseSettlementVO purchaseSettlementVo = purchaseSettlementService.queryDetail(id);
        return CommonResponse.success(purchaseSettlementVo);
    }

    /**
     * 根据主键ID,orgId查询累计结算金额，累计冲抵金额,累计付款，累计预付款金额
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/querySumSettlementTaxMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> querySumSettlementTaxMny(@RequestParam Long contractId,@RequestParam Long orgId) {
        /** 剩余冲抵金额 = 累计预付款 - 累计已冲抵 */
        PurchaseSettlementVO vo = new PurchaseSettlementVO();
        CommonResponse<SumPayMnyVO> sumPayMnyVO = payContractApi.getSumPayMny(contractId,orgId);
        if(!sumPayMnyVO.isSuccess()) {
            return CommonResponse.success("获取付款信息失败，请刷新后再试！");
        }
        BigDecimal sumPayMny = sumPayMnyVO.getData().getSumPayMny();
        BigDecimal sumPrePayMny = sumPayMnyVO.getData().getSumPrePayMny();
        vo.setSumPayMny(sumPayMny==null?BigDecimal.ZERO:sumPayMny);
        vo.setSumPrepayMny(sumPrePayMny==null?BigDecimal.ZERO:sumPrePayMny);
        LambdaQueryWrapper<PurchaseSettlementEntity> lambdachange = Wrappers.<PurchaseSettlementEntity>lambdaQuery();
        lambdachange.eq(PurchaseSettlementEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambdachange.eq(PurchaseSettlementEntity::getContractId,contractId);
        lambdachange.eq(PurchaseSettlementEntity::getOrgId,orgId);
        lambdachange.in(PurchaseSettlementEntity::getBillState,BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<PurchaseSettlementEntity> list = purchaseSettlementService.list(lambdachange);
        if(CollectionUtils.isNotEmpty(list)){
            final BigDecimal[] alreadyPurchaseMoney = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP),new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
            list.forEach(e-> {
                alreadyPurchaseMoney[0] = alreadyPurchaseMoney[0].add(e.getSettlementTaxMny()==null? BigDecimal.ZERO:e.getSettlementTaxMny());
                alreadyPurchaseMoney[1] = alreadyPurchaseMoney[1].add(e.getOffsetMny()==null? BigDecimal.ZERO:e.getOffsetMny());
            });
            //累计结算金额
            vo.setSumSettlementTaxMny(alreadyPurchaseMoney[0]);
            //累计冲抵金额
            vo.setSumOffsetMny(alreadyPurchaseMoney[1]);
        }
        BigDecimal sumOffsetMnys = vo.getSumOffsetMny()==null?BigDecimal.ZERO:vo.getSumOffsetMny();
        if(vo.getSumPrepayMny().compareTo(BigDecimal.ZERO)>=0&&sumOffsetMnys.compareTo(BigDecimal.ZERO)>=0){
            vo.setSymny(vo.getSumPrepayMny().subtract(sumOffsetMnys));
        }

        return CommonResponse.success(vo);
    }

    /**
     * 根据合同主键组装变更数据
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetails", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> queryDetails(@RequestParam Long id) {
        PurchaseSettlementVO purchaseSettlementVO = purchaseSettlementService.queryDetails(id);
        return CommonResponse.success(purchaseSettlementVO);
    }

    /**
     * 根据合同主键ID查询结算记录
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetailRecord", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PurchaseSettlementRecordVO> queryDetailRecord(@RequestParam Long id) {
        PurchaseSettlementRecordVO purchaseSettlementRecordVO = purchaseSettlementService.querySettlementRecord(id);
        return CommonResponse.success(purchaseSettlementRecordVO);
    }

    /**
     * 删除
     *
     * @param vos
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<PurchaseSettlementVO> vos) {
        purchaseSettlementService.deletePurchaseSettlement(vos);
        return CommonResponse.success("删除成功");
    }

    /**
     * 查询供方分页列表
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/pageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("memo");
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));

        List<Long> commonOrgIds = new ArrayList<>();
        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
        if(!orgListResp.isSuccess()) {
            return CommonResponse.success("合同分页列表查询失败, 查询组织信息失败！");
        }
        commonOrgIds.addAll(orgListResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        if (CollectionUtils.isNotEmpty(commonOrgIds)) {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
        }
        IPage<PurchaseSettlementEntity> pageData= purchaseSettlementService.queryPage(param,false);
        com.alibaba.fastjson.JSONObject page = new com.alibaba.fastjson.JSONObject();
        page.put("records", BeanMapper.mapList(pageData.getRecords(), PurchaseSettlementVO.class));
        page.put("total", pageData.getTotal());
        page.put("current", pageData.getCurrent());
        page.put("size", pageData.getSize());
        page.put("pages", pageData.getPages());
        return CommonResponse.success("查询列表数据成功！",page);
    }

    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("memo");
        param.setPageIndex(1);
        param.setPageSize(10000);
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));

        List<Long> commonOrgIds = new ArrayList<>();
        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
        if(!orgListResp.isSuccess()) {
            throw new BusinessException("合同分页列表查询失败, 查询组织信息失败！");
        }
        commonOrgIds.addAll(orgListResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        if (CollectionUtils.isNotEmpty(commonOrgIds)) {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
        }
        IPage<PurchaseSettlementEntity> pageData= purchaseSettlementService.queryPage(param,false);
        List<PurchaseSettlementVO> list = new ArrayList<>();
        if(null!=pageData.getRecords()&&CollectionUtils.isNotEmpty(pageData.getRecords())){
            list = BeanMapper.mapList(pageData.getRecords(), PurchaseSettlementVO.class);
        }
        list.forEach(vo -> {
            vo.setPurchaseTypeName("1".equals(vo.getPurchaseType())?"自采":"集采");
            vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().export("purchaseSettlementListExport.xlsx", beans, response);
    }

    /**
     * @Author yqls
     * @Date 2020/7/9 采购结算参照
     * @Description purchaseSettlementRef
     * @Param [pageNumber, pageSize, condition, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<com.ejianc.business.material.vo.PurchaseSettlementVO>>
     */
    @RequestMapping(value = "/purchaseSettlementRef", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PurchaseSettlementVO>> purchaseSettlementRef(
            @RequestParam(defaultValue = "1") Integer pageNumber,
            @RequestParam(defaultValue = "10") Integer pageSize,
            @RequestParam(value = "condition", required = false) String condition,
            @RequestParam(value = "searchText", required = false) String searchText) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        queryParam.getFuzzyFields().add("contractName");
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("supplierName");
        /** 剩余可申请金额不等于0  */ //剩余可申请金额大于0
        ComplexParam c4 = ParamUtil.getOrParam("surplusApplyMny", new Parameter(QueryParam.GT, 0), new Parameter(QueryParam.EQ, null));
        queryParam.getComplexParams().add(c4);
        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        //已生效状态的单据
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        queryParam.getOrderMap().put("create_time",QueryParam.DESC);
        if(StringUtils.isNotBlank(condition)){
            JSONObject con = JSONObject.parseObject(condition);
            if(condition.contains("contractId")){
                queryParam.getParams().put("contractId",new Parameter(QueryParam.EQ,con.getLong("contractId")));
            }
            if(condition.contains("orgId")){
                queryParam.getParams().put("orgId",new Parameter(QueryParam.EQ,con.getLong("orgId")));
            }
            if(condition.contains("supplierId")){
                queryParam.getParams().put("supplierId",new Parameter(QueryParam.EQ,con.getLong("supplierId")));
            }
        }
        IPage<PurchaseSettlementEntity> pageData = purchaseSettlementService.queryPage(queryParam,false);
        IPage<PurchaseSettlementVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        for(PurchaseSettlementEntity entity : pageData.getRecords()){
            entity.setSurplusApplyMny(MathUtil.safeSub(MathUtil.safeSub(entity.getSettlementTaxMny(), entity.getOffsetMny()), entity.getSumApplyMny()));
        }
        List<PurchaseSettlementVO> purchaseSettlementVos = BeanMapper.mapList(pageData.getRecords(), PurchaseSettlementVO.class);
        /** 3 设备采购 4 设备租聘**/
        for (PurchaseSettlementVO purchaseSettlementVo : purchaseSettlementVos) {
            purchaseSettlementVo.setSettleType(3);
        }
        result.setRecords(purchaseSettlementVos);
        return CommonResponse.success("参照查询成功！", result);
    }
    /**
     * 推送接口
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/pushCost", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> pushCost(@RequestBody PurchaseSettlementVO purchaseSettlementVO) {
        return purchaseSettlementService.pushCost(purchaseSettlementVO);
    }
    /**
     * @Description  参数校验
     */
    @RequestMapping(value = "checkParams", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ParamsCheckVO> checkParams(@RequestParam(value="purchaseType") Integer purchaseType,@RequestParam(value="sumPayMny") BigDecimal sumPayMny, @RequestParam(value="payMny") BigDecimal payMny, @RequestParam(value="contractMny") BigDecimal contractMny,
                                                     @RequestParam(value="projectId") Long projectId) {
        ParamsCheckVO paramsCheckVO = purchaseSettlementService.checkParams(purchaseType,sumPayMny, payMny, contractMny, projectId);
        return CommonResponse.success("参数校验成功！", paramsCheckVO);
    }
}
