package com.ejianc.business.equipment.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.equipment.bean.RentContractEntity;
import com.ejianc.business.equipment.service.IRentContractService;
import com.ejianc.business.equipment.service.IRentSettlementService;
import com.ejianc.business.equipment.vo.ParamsCheckVO;
import com.ejianc.business.equipment.vo.RentContractPayVO;
import com.ejianc.business.equipment.vo.RentContractVO;
import com.ejianc.business.equipment.vo.RentSettlementRecordVO;
import com.ejianc.business.finance.api.IPayContractApi;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.SumPayMnyVO;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.business.tax.vo.InvoiceReceiveRecordVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 设备合同表 前端控制器
 * </p>
 *
 * @author yqls
 * @since 2020-06-08
 */
@RestController
@RequestMapping("rentContract")
public class RentContractController {
    @Autowired
    private IRentContractService rentContractService;
    @Autowired
    private IRentSettlementService rentSettlementService;

    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private IInvoiceApi invoiceApi;
    @Autowired
    private IPayContractApi payContractApi;
    @Autowired
    private SessionManager sessionManager;

    /**
     * 新增或者修改
     *
     * @param rentContractVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<RentContractVO> saveOrUpdate(@RequestBody RentContractVO rentContractVo) {
        return rentContractService.saveOrUpdate(rentContractVo);
    }

    /**
     * 根据主键ID查询供方详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<RentContractVO> queryDetail(@RequestParam Long id) {
        RentContractVO rentContractVo = rentContractService.queryDetail(id);
        return CommonResponse.success(rentContractVo);
    }

    /**
     * 删除
     *
     * @param vos
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<RentContractVO> vos) {
        rentContractService.deleteRentContract(vos);
        return CommonResponse.success("删除成功");
    }

    /**
     * 查询供方分页列表
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/pageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("customerName");
        fuzzyFields.add("employeeName");

        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)){
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            param.getParams().put("orgId",new Parameter("in",authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }else{
            param.getParams().put("orgId",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        IPage<RentContractEntity> pageData= rentContractService.queryPage(param,false);
        //页面统计，查询原合同金额，现合同金额
        Map<String, Object> contractAmountMap = rentContractService.countContractAmount(param);
        JSONObject page = new JSONObject();
        page.put("records", BeanMapper.mapList(pageData.getRecords(), RentContractVO.class));
        page.put("total", pageData.getTotal());
        page.put("current", pageData.getCurrent());
        page.put("size", pageData.getSize());
        page.put("pages", pageData.getPages());
        page.put("rentCount", contractAmountMap);
        return CommonResponse.success("查询列表数据成功！",page);
    }

    /**
     * 查询指定组织本下的所有租赁合同信息列表
     *
     * @param pageNumber
     * @param pageSize
     * @param condition
     * @param searchText
     * @return
     */
    @GetMapping("/rentContractRef")
    public CommonResponse<IPage<RentContractVO>> projectListRefe(@RequestParam(defaultValue = "1") Integer pageNumber,
                                                                    @RequestParam(defaultValue = "10") Integer pageSize,
                                                                    @RequestParam(value = "condition", required = false) String condition,
                                                                    @RequestParam(value = "searchText", required = false) String searchText) {

        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("contractName");
        List<Integer> billstate = new ArrayList<>();
        billstate.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
        billstate.add(BillStateEnum.PASSED_STATE.getBillStateCode());
        queryParam.getParams().put("bill_state",new Parameter(QueryParam.IN,billstate));
        // 已封账，不能被参照到
        queryParam.getParams().put("contractStatus", new Parameter(QueryParam.NE, 3));
        Long orgId = null;
        String type = null;
        if(StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if(null != conditionMap.get("projectId")) {
                Long projectId = Long.valueOf(conditionMap.get("projectId").toString());
                queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
            }
            if(null != conditionMap.get("supplierId")) {
                Long projectId = Long.valueOf(conditionMap.get("supplierId").toString());
                queryParam.getParams().put("supplierId", new Parameter(QueryParam.EQ, projectId));
            }
            if (null != conditionMap.get("orgId")) {
                orgId = Long.valueOf(conditionMap.get("orgId").toString());
            }
            if (null != conditionMap.get("type")) {
                type = conditionMap.get("type").toString();
            }
        }
        if ("pay".equals(type)) {
            queryParam.getParams().remove("contractStatus");
        }
        orgId = orgId != null ? orgId : InvocationInfoProxy.getOrgId();
        queryParam.getOrderMap().put("create_time", QueryParam.DESC);
        queryParam.getParams().put("org_id", new Parameter("in", orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        IPage<RentContractEntity> pageData = rentContractService.queryPage(queryParam, false);
        IPage<RentContractVO> result = new com.baomidou.mybatisplus.extension.plugins.pagination.Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        result.setRecords(BeanMapper.mapList(pageData.getRecords(), RentContractVO.class));
        return CommonResponse.success("租赁合同参照查询成功！", result);
    }
    /**
     * 根据主键ID查询收票记录详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryInvoiceDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<InvoiceReceiveRecordVO> queryInvoiceDetail(@RequestParam Long id) {
        RentContractEntity entity = rentContractService.selectById(id);
        CommonResponse<InvoiceReceiveRecordVO>  res = invoiceApi.getInvoiceReceiveRecord(id);
        if(!res.isSuccess() || !res.isSuccess()) {
            throw new BusinessException("获取收票信息失败！");
        }
        InvoiceReceiveRecordVO vo = res.getData();
        if(null!=vo){
            vo.setContractTaxMny(entity.getContractTaxMny()==null?BigDecimal.ZERO:entity.getContractTaxMny());
            //获取累计结算
            RentSettlementRecordVO settlementRecordVO = rentSettlementService.querySettlementRecord(id);
            if(settlementRecordVO!=null){
                BigDecimal sumSettlementTaxMny = settlementRecordVO.getSumSettlementTaxMny()==null?BigDecimal.ZERO:settlementRecordVO.getSumSettlementTaxMny();
                BigDecimal settleNoRecMny = MathUtil.safeSub(sumSettlementTaxMny, vo.getInvoiceTaxMny());
                //已结算未收票金额
                vo.setSettleNoRecMny(settleNoRecMny);
                vo.setSumSettleTaxMny(sumSettlementTaxMny);

            }

        }
        return CommonResponse.success(vo);
    }
    /**
     * 根据主键ID查询付款记录
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryPayDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<RentContractPayVO> queryPayDetail(@RequestParam Long id) {
        RentContractEntity entity = rentContractService.selectById(id);
        RentContractPayVO rentContractPayVO = BeanMapper.map(entity, RentContractPayVO.class);
        CommonResponse<SumPayMnyVO> sumPayMnyVOList = payContractApi.getSumPayMnyVOList(id, InvocationInfoProxy.getOrgId());
        if(!sumPayMnyVOList.isSuccess() || !sumPayMnyVOList.isSuccess()) {
            throw new BusinessException("获取收票信息失败！");
        }

        BigDecimal sumPayMny = sumPayMnyVOList.getData().getSumPayMny()==null?BigDecimal.ZERO:sumPayMnyVOList.getData().getSumPayMny();
        BigDecimal sumDeductionMny = sumPayMnyVOList.getData().getSumDeductionMny();
        rentContractPayVO.setContractId(sumPayMnyVOList.getData().getContractId());
        rentContractPayVO.setSumPayMny(sumPayMny);
        rentContractPayVO.setSumDeductionMny(sumDeductionMny);
        rentContractPayVO.setSumApplyMny(sumPayMnyVOList.getData().getSumApplyMny()==null?BigDecimal.ZERO: sumPayMnyVOList.getData().getSumApplyMny());
        rentContractPayVO.setContractVOList(sumPayMnyVOList.getData().getContractVOList());

        //获取累计结算
        RentSettlementRecordVO settlementRecordVO = rentSettlementService.querySettlementRecord(id);
        BigDecimal sumSettlementTaxMny = settlementRecordVO.getSumSettlementTaxMny()==null?BigDecimal.ZERO:settlementRecordVO.getSumSettlementTaxMny();
        rentContractPayVO.setSumSettlementTaxMny(sumSettlementTaxMny);
        //获取累计收票
        CommonResponse<InvoiceReceiveRecordVO>  res = invoiceApi.getInvoiceReceiveRecord(id);
        if(!res.isSuccess() || !res.isSuccess()) {
            throw new BusinessException("获取收票信息失败！");
        }
        rentContractPayVO.setSumInvoiceTaxMny(res.getData().getInvoiceTaxMny());
        //累计未付金额 = 累计结算金额 - 累计付款金额 - 累计扣款金额
        rentContractPayVO.setSumUnPayMny(ComputeUtil.safeSub(sumSettlementTaxMny, sumPayMny, sumDeductionMny));
        return CommonResponse.success(rentContractPayVO);
    }
    /**
     * 获取合同开票记录接口
     *
     * @param rentContractVO
     * @return
     */
    @PostMapping("changeState")
    public CommonResponse<String> changeState(@RequestBody RentContractVO rentContractVO) {
        LambdaUpdateWrapper<RentContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(RentContractEntity::getId,rentContractVO.getId());
        updateWrapper.set(RentContractEntity::getContractStatus, rentContractVO.getContractStatus());
        rentContractService.update(updateWrapper);
        return CommonResponse.success("修改合同状态成功！");
    }
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.setPageIndex(1);
        queryParam.setPageSize(-1);

        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("contractName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("supplierName");
        queryParam.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        IPage<RentContractEntity> pageData= rentContractService.queryPage(queryParam,false);

        Map<String, Object> beans = new HashMap<>();
        List<RentContractVO> list = new ArrayList<>();
        if(null!=pageData.getRecords()&&CollectionUtils.isNotEmpty(pageData.getRecords())){
            list = BeanMapper.mapList(pageData.getRecords(), RentContractVO.class);
            list.forEach(vo -> {
                if("1".equals(vo.getContractStatus())){
                    vo.setContractStatusName("未签订");
                }else if("2".equals(vo.getContractStatus())){
                    vo.setContractStatusName("履约中");
                }else if("3".equals(vo.getContractStatus())){
                    vo.setContractStatusName("已封账");
                }
                if(vo.getChangeStatus() == 1){
                    vo.setChangeStatusName("未变更");
                }else if(vo.getChangeStatus() == 2){
                    vo.setChangeStatusName("变更中");
                }else if(vo.getChangeStatus() == 3){
                    vo.setChangeStatusName("已变更");
                }
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            });
        }
        beans.put("records", list);
        ExcelExport.getInstance().export("rentContractListExport.xlsx", beans, response);
    }


    public ComplexParam getPageQueryParam(Long orgId) {
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);

        if(null == orgId) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
        CommonResponse<List<OrgVO>> parentOrgResp = orgApi.findParentsByOrgId(orgId);
        if(!childOrgResp.isSuccess() || !parentOrgResp.isSuccess()) {
            throw new BusinessException("合同分页列表查询失败, 查询组织信息失败！");
        }

        List<Long> parentOrgIds = new ArrayList<>();
        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        parentOrgIds.addAll(parentOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        parentOrgIds.remove(orgId);

        ComplexParam c2 = new ComplexParam();
        c2.setLogic(ComplexParam.OR);
        c2.getParams().put("org_id", new Parameter(QueryParam.IN, childIds));
        c1.getComplexParams().add(c2);

        if(CollectionUtils.isNotEmpty(parentOrgIds)) {
            ComplexParam c3 = new ComplexParam();
            c3.setLogic(ComplexParam.OR);
            c3.getParams().put("rent_type", new Parameter(QueryParam.EQ, "2"));
            c3.getParams().put("org_id", new Parameter(QueryParam.IN, parentOrgIds));
            c1.getComplexParams().add(c3);
            c3.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        }
        return c1;
    }

    /**
     * 统计匹配条件的（属于项目的）合同含税现合同金额总值(单位：万元）
     *
     * @param param
     *      projectIds: 指定所属的项目的id列表
     * @return
     */
    @PostMapping(value = "analysisContract")
    public CommonResponse<BigDecimal> analysisContract(@RequestBody Map<String, Object> param) {
        QueryParam queryParam = new QueryParam();
        //租户隔离
        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("dr",new Parameter(QueryParam.EQ, BaseVO.DR_UNDELETE));
        //审批生效的单据
        queryParam.getParams().put("billState",new Parameter(QueryParam.IN,
                Arrays.asList(new Integer[]{BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode()})));
        if(null != param && null != param.get("projectIds") && CollectionUtils.isNotEmpty((List<Long>)param.get("projectIds"))) {
            queryParam.getParams().put("projectId", new Parameter(QueryParam.IN, param.get("projectIds")));
        }
        List<Long> orgIds = new ArrayList<>();
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            orgIds = authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }  else {
            orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }
        if (CollectionUtils.isNotEmpty(orgIds)){
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
        }
        Map<String, Object> result = rentContractService.countContractAmount(queryParam);
        String dataStr =  (null != result && null != result.get("curAmount")) ? result.get("curAmount").toString() : "0";
        BigDecimal total = new BigDecimal(dataStr);

        return CommonResponse.success(total.divide(new BigDecimal(10000), 2, BigDecimal.ROUND_HALF_UP));
    }
    /**
     * 参数控制
     *
     * @param entity 预算机械费总金额  设备租赁总金额
     *
     * @return {@link ParamsCheckVO}
     */
    @PostMapping("checkParams")
    public CommonResponse<ParamsCheckVO> checkParams(@RequestBody RentContractVO entity) {
        return CommonResponse.success("参数校验成功！", rentContractService.checkParams(entity));
    }

}
