package com.ejianc.foundation.file.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import com.alibaba.druid.util.StringUtils;
import com.ejianc.foundation.file.config.MinioConfiguration;
import com.ejianc.foundation.file.model.FileAttribute;
import com.ejianc.foundation.file.model.FileType;
import com.google.common.collect.Lists;

import io.minio.GetObjectArgs;
import io.minio.GetObjectResponse;
import io.minio.GetPresignedObjectUrlArgs;
import io.minio.MinioClient;
import io.minio.errors.ErrorResponseException;
import io.minio.errors.InsufficientDataException;
import io.minio.errors.InternalException;
import io.minio.errors.InvalidResponseException;
import io.minio.errors.ServerException;
import io.minio.errors.XmlParserException;
import io.minio.http.Method;

/**
 * @date 2017/11/13
 */
@Component
public class FileUtils {
    Logger log= LoggerFactory.getLogger(getClass());

	@Autowired
	private MinioConfiguration minioConfiguration;
	@Autowired
	private MinioClient minioClient;
	@Autowired
	private RedisTemplate<String, Object> redisTemplate;
	
    final String REDIS_FILE_PREVIEW_PDF_KEY = "converted-preview-pdf-file";
    final String REDIS_FILE_PREVIEW_IMGS_KEY = "converted-preview-imgs-file";//压缩包内图片文件集合
    String charset = "utf-8";

    /**
     * 查看文件类型(防止参数中存在.点号或者其他特殊字符，所以先抽取文件名，然后再获取文件类型)
     *
     * @param url
     * @return
     */
    public FileType typeFromUrl(String url) {
        String nonPramStr = url.substring(0, url.indexOf("?") != -1 ? url.indexOf("?") : url.length());
        String fileName = nonPramStr.substring(nonPramStr.lastIndexOf("/") + 1);
        String fileType = fileName.substring(fileName.lastIndexOf(".") + 1);
        if (listPictureTypes().contains(fileType.toLowerCase())) {
          return FileType.picture;
        }
//        if (listArchiveTypes().contains(fileType.toLowerCase())) {
//            return FileType.compress;
//        }
        if (listOfficeTypes().contains(fileType.toLowerCase())) {
            return FileType.office;
        }
        if (listSimpleTexts().contains(fileType.toLowerCase())) {
            return FileType.simText;
        }
        if (listMedias().contains(fileType.toLowerCase())) {
            return FileType.media;
        }
        if("pdf".equalsIgnoreCase(fileType)){
            return FileType.pdf;
        }
        if(listExcelTypes().contains(fileType.toLowerCase())){
            return FileType.excel;
        }
        return FileType.other;
    }
    /**
     * 从url中剥离出文件名
     * @param url
     *      格式如：http://keking.ufile.ucloud.com.cn/20171113164107_月度绩效表模板(新).xls?UCloudPublicKey=ucloudtangshd@weifenf.com14355492830001993909323&Expires=&Signature=I D1NOFtAJSPT16E6imv6JWuq0k=
     * @return
     */
    public String getFileNameFromURL(String url) {
        // 因为url的参数中可能会存在/的情况，所以直接url.lastIndexOf("/")会有问题
        // 所以先从？处将url截断，然后运用url.lastIndexOf("/")获取文件名
        String noQueryUrl = url.substring(0, url.indexOf("?") != -1 ? url.indexOf("?"): url.length());
        String fileName = noQueryUrl.substring(noQueryUrl.lastIndexOf("/") + 1);
        return fileName;
    }

    /**
     * 获取文件后缀
     * @param fileName
     * @return
     */
    public String getSuffixFromFileName(String fileName) {
        String suffix = fileName.substring(fileName.lastIndexOf("."));
        return suffix;
    }

    /**
     * 从路径中获取
     * @param path
     *      类似这种：C:\Users\yudian-it\Downloads
     * @return
     */
    public String getFileNameFromPath(String path) {
        return path.substring(path.lastIndexOf("/") + 1);
    }
    
    public List<String> listMedias(){
    	List<String> list = Lists.newArrayList();
    	return list;
    }
    
    public List<String> listSimpleTexts(){
    	List<String> list = Lists.newArrayList();
    	list.add("txt");
    	return list;
    }


    public List<String> listPictureTypes(){
        List<String> list = Lists.newArrayList();
        list.add("jpg");
        list.add("jpeg");
        list.add("png");
        list.add("gif");
        list.add("bmp");
        list.add("ico");
        list.add("RAW");
        return list;
    }

    public List<String> listArchiveTypes(){
        List<String> list = Lists.newArrayList();
//        list.add("rar");
//        list.add("zip");
//        list.add("jar");
//        list.add("7-zip");
//        list.add("tar");
//        list.add("gzip");
//        list.add("7z");
        return list;
    }

    public List<String> listOfficeTypes() {
        List<String> list = Lists.newArrayList();
        list.add("docx");
        list.add("doc");
        list.add("wps");
//        list.add("xls");
//        list.add("xlsx");
        list.add("ppt");
        list.add("pptx");
        return list;
    }

    public List<String> listExcelTypes() {
        List<String> list = Lists.newArrayList();
        list.add("xls");
        list.add("xlsx");
        return list;
    }

    /**
     * 判断文件编码格式
     * @param path
     * @return
     */
    public String getFileEncodeUTFGBK(String path){
        String enc = Charset.forName("GBK").name();
        File file = new File(path);
        InputStream in= null;
        try {
            in = new FileInputStream(file);
            byte[] b = new byte[3];
            in.read(b);
            if (b[0] == -17 && b[1] == -69 && b[2] == -65) {
                enc = Charset.forName("UTF-8").name();
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }finally {
        	if(in != null) {
        		try {
					in.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
        	}
        }
        System.out.println("文件编码格式为:" + enc);
        return enc;
    }

    /**
     * 对转换后的文件进行操作(改变编码方式)
     * @param outFilePath
     */
    public void doActionConvertedFile(String outFilePath) {
        StringBuffer sb = new StringBuffer();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
        	inputStream = new FileInputStream(outFilePath);
            reader = new BufferedReader(new InputStreamReader(inputStream, charset));
            String line;
            while(null != (line = reader.readLine())){
                if (line.contains("charset=gb2312")) {
                    line = line.replace("charset=gb2312", "charset=utf-8");
                }
                sb.append(line);
            }
            // 添加sheet控制头
            sb.append("<script src=\"js/jquery-3.0.0.min.js\" type=\"text/javascript\"></script>");
            sb.append("<script src=\"js/excel.header.js\" type=\"text/javascript\"></script>");
            sb.append("<link rel=\"stylesheet\" href=\"http://cdn.static.runoob.com/libs/bootstrap/3.3.7/css/bootstrap.min.css\">");
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }finally {
        	try {
        		if(inputStream != null) {
        			inputStream.close();
        		}
        		if(reader != null) {
        			reader.close();
        		}
        	}catch(Exception e) {
        		e.printStackTrace();
        	}
        }
        // 重新写入文件
        FileOutputStream fos = null;
        BufferedWriter writer = null;
        try {
        	fos = new FileOutputStream(outFilePath);
            writer = new BufferedWriter(new OutputStreamWriter(fos,charset));
            writer.write(sb.toString());
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }finally {
        	try {
        		if(fos != null) {
        			fos.close();
        		}
        		if(writer != null) {
        			writer.close();
        		}
        	}catch(Exception e) {
        		e.printStackTrace();
        	}
        }
    }
    /**
     * 对转换后的文件进行操作(改变编码方式)
     * @param is
     */
    public ByteArrayOutputStream doActionConvertedFile(InputStream is) {
        StringBuffer sb = new StringBuffer();
        BufferedReader reader = null;
        try {
        	reader = new BufferedReader(new InputStreamReader(is, charset));
            String line;
            while(null != (line = reader.readLine())){
                if (line.contains("charset=gb2312")) {
                    line = line.replace("charset=gb2312", "charset=utf-8");
                }
                sb.append(line);
            }
            // 添加sheet控制头
            sb.append("<script src=\"js/jquery-3.0.0.min.js\" type=\"text/javascript\"></script>");
            sb.append("<script src=\"js/excel.header.js\" type=\"text/javascript\"></script>");
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }finally {
        	try {
        		if(reader != null) {
        			reader.close();
        		}
        	}catch(Exception e) {
        		e.printStackTrace();
        	}
        }

        // 重新写入文件
        ByteArrayOutputStream bos = null;
        BufferedWriter writer = null;
        try {
            bos = new ByteArrayOutputStream();
            int ch;
            while ((ch = is.read()) != -1) {
                bos.write(ch);
            }
            writer = new BufferedWriter(new OutputStreamWriter(bos, charset));
            writer.write(sb.toString());
            writer.close();
            return bos;
        } catch (Exception e1) {
            e1.printStackTrace();
        }finally {
        	try {
        		if(bos != null) {
        			bos.close();
        		}
        		if(writer != null) {
        			writer.close();
        		}
        	}catch(Exception e) {
        		e.printStackTrace();
        	}
        }
        return null;
    }

    /**
     * 获取文件后缀
     * @param url
     * @return
     */
    private String suffixFromUrl(String url) {
        String nonPramStr = url.substring(0, url.indexOf("?") != -1 ? url.indexOf("?") : url.length());
        String fileName = nonPramStr.substring(nonPramStr.lastIndexOf("/") + 1);
        String fileType = fileName.substring(fileName.lastIndexOf(".") + 1);
        return fileType;
    }

    public FileAttribute getFileAttribute(String url) {
        String decodedUrl=null;
        try {
            decodedUrl = URLDecoder.decode(url, "utf-8");
        }catch (UnsupportedEncodingException e){
            log.debug("url解码失败");
        }
        // 路径转码
        FileType type = typeFromUrl(url);
        String suffix = suffixFromUrl(url);
        // 抽取文件并返回文件列表
        String fileName = getFileNameFromURL(decodedUrl);
        return new FileAttribute(type,suffix,fileName,url,decodedUrl);
    }
    
    /**
     * 下载文件
     *  返回文件字节流大小
     * @param fileUrl 文件URL
     * @return 文件字节
     * @throws IOException
     */
    public GetObjectResponse downloadFileByKeyName(String keyName)  {
		try {
			GetObjectResponse getObjectResponse = minioClient.getObject(GetObjectArgs.builder().bucket(minioConfiguration.getBucketName()).object(keyName).build());
			return getObjectResponse;
		} catch (Exception e) {
			e.printStackTrace();
		}
        return null;
    }
    
    
	
	/**
	 * 获取文件后缀格式
	 * @param filename
	 * @return 文件后缀格式 eg:  (.jpg)
	 */
	public static String getFileExt(String filename,Boolean withPoint) {
		if(StringUtils.isEmpty(filename) || filename.lastIndexOf('.') == -1) {
			return "";
		}
		if(withPoint==null){
			withPoint = true;
		}
		int index = -1;
		if(withPoint){
			index = filename.lastIndexOf('.');
		}else{
			index = filename.lastIndexOf('.')+1;
		}
		String extName = filename.substring(index).toLowerCase();
		return extName;
	}
}
