package com.ejianc.business.finance.service.impl;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.enums.PaymentStatusEnum;
import com.ejianc.business.finance.vo.PayReportedVO;
import com.ejianc.business.finance.vo.RequestPaymentReportVO;
import com.ejianc.foundation.bpm.api.IBpmApi;
import com.ejianc.foundation.metadata.vo.MdReferVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.tools.ant.util.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.finance.mapper.PayReportedMapper;
import com.ejianc.business.finance.bean.PayReportedEntity;
import com.ejianc.business.finance.service.IPayReportedService;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 内部付款报备表
 * 
 * @author generator
 * 
 */
@Service("payReportedService")
public class PayReportedServiceImpl extends BaseServiceImpl<PayReportedMapper, PayReportedEntity> implements IPayReportedService{

    /**
     * 线程池
     */
    private static final ExecutorService executor = Executors.newFixedThreadPool(10);

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    private static final String BILL_CODE = "FKBB";//此处需要根据实际修改

    private static final String BILL_TYPE_CODE = "BT251118000000001";

    @Autowired
    private IBpmApi bpmApi;

    @Autowired
    private PayReportedBpmServiceImpl payReportedBpmService;

    @Override
    public List<PayReportedVO> queryRefPayData(Page<PayReportedVO> page, String sql) {
        return baseMapper.queryRefPayData(page, sql);
    }

    /**
     * 批量保存报备数据
     * @param vo 报备数据
     */
    @Override
    public void batchSaveReport(List<RequestPaymentReportVO> vo) {

        //查询单据类型名称
        CommonResponse<BillTypeVO> billTypeVOCommonResponse = billTypeApi.getByCode(BILL_TYPE_CODE);

        //获取单据类型名称
        String billName = billTypeVOCommonResponse.getData().getBillName();

        //批量生成付款报备编号
        CommonResponse<List<String>> codeBatchByRuleCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid(), vo.size());

        //判断是否成功
        if (codeBatchByRuleCode.isSuccess()) {

            //单据编码索引
            AtomicInteger codeIndex = new AtomicInteger(1);

            //实体索引
            AtomicInteger entityIndex = new AtomicInteger(1);

            //获取编号集合、转换为
            Map<Integer, String> codeMap = codeBatchByRuleCode.getData().stream().collect(Collectors.toMap(
                    item -> codeIndex.getAndIncrement(),
                    item -> item
            ));

            //将请款报备列表转换为付款报备实体列表
            List<PayReportedEntity> payReportedEntityList = vo.stream().map(item -> {

                //获取单据编码
                String billCode = codeMap.get(entityIndex.getAndIncrement());

                //转化实体信息
                PayReportedEntity payReportedEntity = ofPayReportedEntity(item);

                //设置单据编码
                payReportedEntity.setBillCode(billCode);

                //返回实体
                return payReportedEntity;
            }).collect(Collectors.toList());

            //批量保存
            super.saveBatch(payReportedEntityList);

            //将创建的单据都走审批
            batchDoSubmit(payReportedEntityList,billName);
        }

    }

    /**
     * 转换请款报备实体
     * @param vo 请款报备数据
     * @return 付款报备实体
     */
    private static PayReportedEntity ofPayReportedEntity(RequestPaymentReportVO vo) {

        PayReportedEntity payReportedEntity = new PayReportedEntity();
        payReportedEntity.setOrgId(vo.getCashOutOrgId() == null ? null : Long.parseLong(vo.getCashOutOrgId()));
        payReportedEntity.setPaymentId(vo.getCashOutPaymentId() == null ? null : Long.parseLong(vo.getCashOutPaymentId()));
        payReportedEntity.setPaymentCode(vo.getCashOutPaymentCode());
        payReportedEntity.setProjectId(vo.getCashOutProjectId() == null ? null : Long.parseLong(vo.getCashOutProjectId()));
        payReportedEntity.setProjectCode(vo.getCashOutProjectCode());
        payReportedEntity.setProjectName(vo.getCashOutProjectName());
        payReportedEntity.setSupplierId(vo.getCashOutSupplierId() == null ? null : Long.parseLong(vo.getCashOutSupplierId()));
        payReportedEntity.setSupplierCode(vo.getCashOutSupplierCode());
        payReportedEntity.setSupplierName(vo.getCashOutSupplierName());
        payReportedEntity.setContractId(vo.getCashOutContractId() == null ? null : Long.parseLong(vo.getCashOutContractId()));
        payReportedEntity.setContractCode(vo.getCashOutContractCode());
        payReportedEntity.setContractName(vo.getCashOutContractName());
        payReportedEntity.setContractMny(vo.getCashOutContractAmount() == null ? null : new BigDecimal(vo.getCashOutContractAmount()));
        payReportedEntity.setProductName(vo.getCashOutLaborServiceName());
        payReportedEntity.setApproveTime(vo.getCashOutApproveTime());
        payReportedEntity.setApplyMny(vo.getCashOutApplyPaymentAmount() == null ? null : new BigDecimal(vo.getCashOutApplyPaymentAmount()));
        payReportedEntity.setFeeKind(vo.getCashOutFeeKind() == null ? null : Long.parseLong(vo.getCashOutFeeKind()));
        payReportedEntity.setFeeKindName(vo.getCashOutFeeKindName());
        payReportedEntity.setReportedPaymentTime(vo.getCashOutReportPaymentTime());
        payReportedEntity.setReason(vo.getCashOutUrgentReason());
        payReportedEntity.setEmployeeId(InvocationInfoProxy.getUserid()); //报备人为当前登录用户
        payReportedEntity.setEmployeeName(vo.getCashOutPayerName());
        payReportedEntity.setPaymentState(PaymentStatusEnum.REPORTING.getCode()); //创建的单据都是报备中
        return payReportedEntity;
    }

    /**
     * 批量处理单据
     * @param payReportedEntityList 单据列表
     */
    private void batchDoSubmit(List<PayReportedEntity> payReportedEntityList,String billName){

        //获取当前操作用户id
        Long userid = InvocationInfoProxy.getUserid();

        //批量处理单据
        payReportedEntityList.forEach(item -> {

            //生成单据实例名称
            String billInstanceName = generateBillInstanceName(item,billName);

            //构建审批参数
            Map<String, Object> params = new HashMap<>();
            params.put("userId", userid); //用户id
            params.put("billtypeId", BILL_TYPE_CODE); //单据类型code
            params.put("businessKey", item.getId());//单据id
            params.put("formurl","/ejc-finance-frontend/#/payReported"); //PC端url,必填
            params.put("billOrgId",item.getOrgId());//单据提交组织
            params.put("procInstName", billInstanceName); //流程实例名称

            //提交流程
            bpmApi.doSubmit(params);

        });

    }

    /**
     * 生成单据实例名称
     * @param payReportedEntity 付款报备实体
     */
    private String generateBillInstanceName(PayReportedEntity payReportedEntity,String billName){

        //获取当前时间 yyyyMMddHHmmss
        String now = DateUtil.now().replaceAll("-", "").replaceAll(":", "");

        return billName + "_" + payReportedEntity.getBillCode() + "_" + now;
    }

    /**
     * 根据付款ID查询请款报备列表
     *
     * @param paymentIds 付款id
     * @return 请款报备列表
     */
    @Override
    public List<PayReportedVO> queryRequestPaymentReportList(List<Long> paymentIds) {

        //将付款id集合转换为字符串、前缀为( 后缀为 ) 以,分割
        StringJoiner sj = new StringJoiner(",","(" ,")");
        for (Long paymentId : paymentIds) {
            if (paymentId != null){
                sj.add(paymentId.toString());
            }
        }

        //构建付款报备查询参数
        String sql = "AND A1.paymentId IN " + sj;

        //查询付款报备列表
        return baseMapper.queryRefPayData(null, sql);
    }

}
