package com.ejianc.business.bank.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.bank.bean.BankAcctEntity;
import com.ejianc.business.bank.bean.BankFlowEntity;
import com.ejianc.business.bank.consts.BankFlowSourceType;
import com.ejianc.business.bank.service.IBankAcctService;
import com.ejianc.business.bank.service.IBankFlowService;
import com.ejianc.business.bank.vo.BankAcctVO;
import com.ejianc.business.bank.vo.BankFlowVO;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 银行账户
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("bankAcct")
public class BankAcctController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IBankAcctService service;
    @Autowired
    private IBillTypeApi billTypeApi;
    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IBankFlowService bankFlowService;


    @RequestMapping(value = "/init", method = RequestMethod.GET)
    @ResponseBody
    public void init() {
        LambdaQueryWrapper<BankAcctEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BankAcctEntity::getTenantId, InvocationInfoProxy.getTenantid());
        List<BankAcctEntity> list = service.list(queryWrapper);
        for (BankAcctEntity entity : list) {
            BankFlowVO flowVO = BankFlowVO.instanceVOBySourceType(BankFlowSourceType.初始余额);
            flowVO.setBillDate(entity.getInitDate());
            flowVO.setSourceId(entity.getId());
            flowVO.setBillId(entity.getId());
            flowVO.setReceiveMoney(entity.getInitSurplusMny());
            flowVO.setAccountId(entity.getId());
            flowVO.setAccountName(entity.getAccountName());
            flowVO.setAccountBank(entity.getAccountBank());
            flowVO.setAccountNum(entity.getAccountNum());
            flowVO.setCreateTime(entity.getCreateTime());
            bankFlowService.saveFlowVO(flowVO);
        }
    }


    /**
     * @description: 停用、启用
     *
     * @return {@link CommonResponse< BankAcctVO>}
     * @author songlx
     * @date: 2024/3/29
     */
    @RequestMapping(value = "/changeState", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<BankAcctVO> changeState(@RequestBody BankAcctVO bankAcctVO) {
        LambdaUpdateWrapper<BankAcctEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(BankAcctEntity::getId, bankAcctVO.getId());
        updateWrapper.set(BankAcctEntity::getState, bankAcctVO.getState());
        service.update(updateWrapper);
        return CommonResponse.success("操作成功！");
    }


    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<BankAcctVO> saveOrUpdate(@RequestBody BankAcctVO saveOrUpdateVO) {
        LambdaQueryWrapper<BankAcctEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BankAcctEntity::getTenantId, InvocationInfoProxy.getTenantid());
        queryWrapper.ne(saveOrUpdateVO.getId() != null, BankAcctEntity::getId, saveOrUpdateVO.getId());
        queryWrapper.eq(BankAcctEntity::getAccountNum, saveOrUpdateVO.getAccountNum());
        int count = service.count(queryWrapper);
        if (count > 0) {
            throw new BusinessException("该账号已存在!");
        }
        boolean updFlag = saveOrUpdateVO.getId() != null;
        //被引用的单据不能修改开户行和银行账户校验
        if (updFlag) {
            BankAcctEntity bankAcctEntity = service.selectById(saveOrUpdateVO.getId());
            CommonResponse<String> resp = billTypeApi.checkQuote("BT202403000030", saveOrUpdateVO.getId());
            if (!resp.isSuccess()
                    &&(!saveOrUpdateVO.getAccountBank().equals(bankAcctEntity.getAccountBank())
                    || !saveOrUpdateVO.getAccountNum().equals(bankAcctEntity.getAccountNum()) )){
                throw new BusinessException("该账户已使用，开户行与账户信息不可更改！");
            }
            // 修改并且初始余额发生变化
            if (!ComputeUtil.equals(saveOrUpdateVO.getInitSurplusMny(), bankAcctEntity.getInitSurplusMny())) {
                LambdaUpdateWrapper<BankFlowEntity> wrapper = new LambdaUpdateWrapper<>();
                wrapper.eq(BankFlowEntity::getBillId, saveOrUpdateVO.getId());
                wrapper.eq(BankFlowEntity::getSourceFlag, BankFlowSourceType.初始余额.getCode());
                wrapper.set(BankFlowEntity::getReceiveMoney, saveOrUpdateVO.getInitSurplusMny());
                bankFlowService.update(wrapper);
            }
        }
        BankAcctEntity entity = BeanMapper.map(saveOrUpdateVO, BankAcctEntity.class);
        boolean b = service.saveOrUpdate(entity, false);
        // 新增保存初始余额流水
        if (b && !updFlag) {
            BankFlowVO flowVO = BankFlowVO.instanceVOBySourceType(BankFlowSourceType.初始余额);
            flowVO.setBillDate(new Date());
            flowVO.setSourceId(entity.getId());
            flowVO.setBillId(entity.getId());
            flowVO.setReceiveMoney(entity.getInitSurplusMny());
            flowVO.setAccountId(entity.getId());
            flowVO.setAccountName(entity.getAccountName());
            flowVO.setAccountBank(entity.getAccountBank());
            flowVO.setAccountNum(entity.getAccountNum());
            bankFlowService.saveFlowVO(flowVO);
        }

        BankAcctVO vo = BeanMapper.map(entity, BankAcctVO.class);
        bankFlowService.getSurplusMny(Arrays.asList(vo));
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<BankAcctVO> queryDetail(Long id) {
        BankAcctEntity entity = service.selectById(id);
        BankAcctVO vo = BeanMapper.map(entity, BankAcctVO.class);
        bankFlowService.getSurplusMny(Arrays.asList(vo));
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<BankAcctVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            List<Long> ids = vos.stream().map(BankAcctVO::getId).collect(Collectors.toList());
            for (BankAcctVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                CommonResponse<String> resp = billTypeApi.checkQuote("BT202403000030", vo.getId());
                if (!resp.isSuccess()) {
                    return CommonResponse.error(resp.getMsg());
                }
            }
            LambdaQueryWrapper<BankFlowEntity> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(BankFlowEntity::getBillId, ids);
            queryWrapper.ne(BankFlowEntity::getSourceFlag, BankFlowSourceType.初始余额.getCode());
            int count = bankFlowService.count(queryWrapper);
            if (count > 0) {
                return CommonResponse.error("该账户已使用，无法删除！");
            }
            boolean b = service.removeByIds(ids, true);
            if (b) {
                LambdaQueryWrapper<BankFlowEntity> wrapper = new LambdaQueryWrapper<>();
                wrapper.in(BankFlowEntity::getBillId, ids);
                wrapper.eq(BankFlowEntity::getSourceFlag, BankFlowSourceType.初始余额.getCode());
                bankFlowService.remove(wrapper);
            }
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<BankAcctVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("orgName", "accountName", "accountNum", "accountBank", "employeeName"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        IPage<BankAcctEntity> page = service.queryPage(param, false);
        IPage<BankAcctVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<BankAcctVO> bankAcctVOS = BeanMapper.mapList(page.getRecords(), BankAcctVO.class);
        bankFlowService.getSurplusMny(bankAcctVOS);
        pageData.setRecords(bankAcctVOS);

        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    @RequestMapping(value = "/totalSurplusMny", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<BigDecimal> totalSurplusMny(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("orgName", "accountName", "accountNum", "accountBank", "employeeName"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getParams().put("dr", new Parameter(QueryParam.EQ, 0));
        // 余额统计
        BigDecimal surplusMny = service.totalSurplusMny(param);

        return CommonResponse.success("查询列表数据成功！", surplusMny);
    }



    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("orgName", "accountName", "accountNum", "accountBank", "employeeName"));
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<BankAcctEntity> list = service.queryList(param);
        List<BankAcctVO> bankAcctVOS = BeanMapper.mapList(list, BankAcctVO.class);
        bankFlowService.getSurplusMny(bankAcctVOS);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", bankAcctVOS);
        ExcelExport.getInstance().export("BankAcct-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refBankAcctData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<BankAcctVO>> refBankAcctData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                             String condition,
                                                             String searchObject,
                                                             String searchText) {
        QueryParam param = new QueryParam();
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("orgName", "accountName", "accountNum", "accountBank", "employeeName"));
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("state", new Parameter(QueryParam.EQ, 1));
        Long orgId = InvocationInfoProxy.getOrgId();
        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            if (_con.containsKey("orgId")) {
                orgId = _con.getLong("orgId");
            }
            // 流水需要参照到所有的
            if (_con.containsKey("state") && "all".equals(_con.getString("state"))) {
                param.getParams().remove("state");
            }
        }
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        LinkedHashMap<String, String> orderMap = param.getOrderMap();
        if (MapUtils.isEmpty(orderMap)) {
            orderMap.put("createTime", "desc");
        }
        IPage<BankAcctEntity> page = service.queryPage(param, false);
        IPage<BankAcctVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), BankAcctVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }
}
