package com.ejianc.business.finance.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.finance.bean.PayReimburseEntity;
import com.ejianc.business.finance.bean.PaySporadicEntity;
import com.ejianc.business.finance.service.ILoadReimburseService;
import com.ejianc.business.finance.service.IPayReimburseService;
import com.ejianc.business.finance.service.IloadInvoiceService;
import com.ejianc.business.finance.vo.*;
import com.ejianc.business.income.consts.EJCDateUtil;
import com.ejianc.business.material.vo.ParamsCheckVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.permission.api.IRoleApi;
import com.ejianc.foundation.permission.vo.RoleVO;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.ParamRegisterSetVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;

/**
 * <p>
 *  前端控制器
 * </p>
 *
 * @author yqls
 * @since 2020-05-28
 */
@Controller
@RequestMapping("/payReimburse")
public class PayReimburseController {

    private static final long serialVersionUID = 1L;

    // 费用报销数据权限白名单角色配置
    private static final String PARAM_VIEW_CODE = "P-46055G32";

    @Autowired
    private IRoleApi roleApi;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private IPayReimburseService payReimburseService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 新增或者修改
     * @param payReimburseVO
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PayReimburseVO> saveOrUpdate(@RequestBody PayReimburseVO payReimburseVO) {
        PayReimburseVO backVO = payReimburseService.insertOrUpdate(payReimburseVO);
        return CommonResponse.success("保存或修改单据成功！", backVO);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询详情
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayReimburseVO> queryDetail(Long id) {
        PayReimburseVO backVO = payReimburseService.queryDetail(id);
        return CommonResponse.success("查询详情数据成功！", backVO);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description queryPrint 查询打印详情
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryPrint", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryPrint(Long id) {
        JSONObject json = new JSONObject();
        json.put("PayReimburse", payReimburseService.queryDetail(id));
        return CommonResponse.success("查询打印数据成功！", json);
    }

    /**
     * @description: 根据当前用户角色判断是都可以查看全部报销单
     * @return: java.lang.Boolean
     * @author songlx
     * @date: 2022/7/26
     */
    public Boolean canViewAll() {
        CommonResponse<ParamRegisterSetVO> byCode = paramConfigApi.getByCode(PARAM_VIEW_CODE);
        if (byCode.isSuccess() && null != byCode.getData()) {
            ParamRegisterSetVO paramRegisterSetVO = byCode.getData();
            String valueData = paramRegisterSetVO.getValueData();
            if (StringUtils.isNotEmpty(valueData)) {
                Long userid = InvocationInfoProxy.getUserid();
                CommonResponse<List<RoleVO>> roleRes = roleApi.queryRoleByUserId(userid);
                if (roleRes.isSuccess() && CollectionUtils.isNotEmpty(roleRes.getData())) {
                    for (RoleVO roleVO : roleRes.getData()) {
                        if (valueData.contains(roleVO.getCode()))
                            return Boolean.TRUE;
                    }
                }
            }
        }
        return Boolean.FALSE;
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
        if(param.getOrderMap().size() == 0){
            param.getOrderMap().put("createTime", QueryParam.DESC);
        }
        // 判断当前用户是否可以查看所有的报销单
        if (!canViewAll()) {
            param.getParams().put("createUserCode", new Parameter(QueryParam.EQ, InvocationInfoProxy.getUsercode()));
        }
        JSONObject page = payReimburseService.queryPageJson(param, false);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryApproveList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryApproveList(@RequestBody QueryParam param) {
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("approveTime", QueryParam.DESC);
        JSONObject page = payReimburseService.queryPageJson(param, false);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description delete 批量删除单据
     * @Param [ids]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<PayReimburseVO> vos) {
        List<Long> ids = vos.stream().map(PayReimburseVO::getId).collect(Collectors.toList());
        String msg = payReimburseService.delete(ids);
        return CommonResponse.success(msg);
    }


    /**
     * excel导出
     *
     * @param response
     * @return
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        //根据查看权限导出
        if (!canViewAll()) {
            queryParam.getParams().put("createUserCode", new Parameter(QueryParam.EQ, InvocationInfoProxy.getUsercode()));
        }

        List<PayReimburseVO> supplierVos = payReimburseService.queryExportList(queryParam);
        supplierVos.forEach(vo -> {
            if (null != vo.getApplyMny()) {
                vo.setUnpaidMny(vo.getApplyMny().subtract(vo.getPayMny() == null ? BigDecimal.ZERO : vo.getPayMny()));
                if(vo.getIsShareFlag() != null){
                    vo.setIsShareFlagName(vo.getIsShareFlag() == 1 ? "是" : "否");
                }
            }
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", supplierVos);
        ExcelExport.getInstance().export("reim-payapply-export.xlsx", beans, response);
    }

    /**
     * 支出管理excel导出
     *
     * @param response
     * @return
     */
    @RequestMapping(value = "/excelExportApprove", method = RequestMethod.POST)
    @ResponseBody
    public void excelExportApprove(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        //已生效状态的单据
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        List<PayReimburseVO> supplierVos = payReimburseService.queryExportList(queryParam);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", supplierVos);
        ExcelExport.getInstance().export("reim-payapply-export.xlsx", beans, response);
    }


    @RequestMapping(value = "/referData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PayReimburseVO>> referData(
            @RequestParam(defaultValue = "1") Integer pageNumber,
            @RequestParam(defaultValue = "10") Integer pageSize,
            @RequestParam(value = "condition", required = false) String condition,
            @RequestParam(value = "searchObject", required = false) String searchObject,
            @RequestParam(value = "searchText", required = false) String searchText) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchObject(searchObject);
        queryParam.setSearchText(searchText);
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("feeType");
        fuzzyFields.add("payReason");
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("payType", new Parameter(QueryParam.EQ, 1)); //付款类型为组织付款
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
        queryParam.getParams().put("payStatus", new Parameter(QueryParam.EQ, 2)); //支付状态：1-未支付，2-已支付

        //移动端和pc端权限处理
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();

        if (org.apache.commons.lang.StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<PayReimburseVO> page = new com.baomidou.mybatisplus.extension.plugins.pagination.Page<>(queryParam.getPageIndex(), queryParam.getPageSize());
        QueryWrapper<PayReimburseEntity> wrapper = changeToQueryWrapper(queryParam);
        List<PayReimburseVO> list = payReimburseService.queryReferData(page, wrapper);
        IPage<PayReimburseVO> result = new com.baomidou.mybatisplus.extension.plugins.pagination.Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        if (CollectionUtils.isNotEmpty(list)) {
            result.setRecords(list);
        }
        return CommonResponse.success("参照查询成功！", result);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description refer 商机登记列表参照
     *
     * @Param [pageNumber, pageSize, searchText]
     * @Return com.alibaba.fastjson.JSONObject
     */
    @RequestMapping(value = "/refer")
    @ResponseBody
    public JSONObject refer(@RequestParam(defaultValue = "1") Integer pageNumber,
                            @RequestParam(defaultValue = "10") Integer pageSize,
                            String searchText) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("address");
        fuzzyFields.add("area");
        fuzzyFields.add("engineeringName");
        fuzzyFields.add("engineeringCode");
        fuzzyFields.add("engineeringTypeName");
        fuzzyFields.add("infoSourceName");
        fuzzyFields.add("followPersonName");
        fuzzyFields.add("constructionUnitName");
        IPage<PayReimburseEntity> pageData = payReimburseService.queryPage(queryParam, false);
        Page<PayReimburseEntity> page = new PageImpl<>(pageData.getRecords(), new PageRequest(pageNumber, pageSize), pageData.getTotal());
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("data", page);
        jsonObject.put("code", "success");
        jsonObject.put("msg", "查询数表参照成功");
        return jsonObject;
    }

    /**
     * 查询收款账户信息
     * @param queryParam
     * @return
     */
    @RequestMapping(value = "/queryReceiveInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayReimburseVO> queryReceiveInfo(QueryParam queryParam) {
        queryParam.getParams().put("applyUserId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getEmployeeId()));
        //已生效状态的单据
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<PayReimburseEntity> entityList = payReimburseService.queryList(queryParam, false);
        PayReimburseVO backVO = !entityList.isEmpty() ? BeanMapper.map(entityList.get(0), PayReimburseVO.class) : new PayReimburseVO();
        return CommonResponse.success("查询数据成功！", backVO);
    }

    /**成本改造
     * 关联保存接口
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/pushCost", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PayReimburseVO> pushCost(@RequestBody PayReimburseVO paySporadicVO) {
        return payReimburseService.pushCost(paySporadicVO);
    }

    /**
     * @Author 曹鹏辉
     * @Date 2021/9/17
     * @Description 获取合计栏信息(累计申请金额总计 ， 累计支付金额总计)
     * @Param
     * @Return com.ejianc.framework.core.response.CommonResponse<TotalColumnVO>
     */
    @RequestMapping(value = "getTotalColumnInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TotalColumnVO> getTotalColumnInfo(@RequestBody QueryParam param) {
        param.getOrderMap().put("createTime", QueryParam.DESC);
        // 判断当前用户是否可以查看所有的报销单
        if (!canViewAll()) {
            param.getParams().put("createUserCode", new Parameter(QueryParam.EQ, InvocationInfoProxy.getUsercode()));
        }
        return CommonResponse.success("查询成功！", payReimburseService.getTotalColumnInfo(param, false));
    }

    /**
     * @Author 曹鹏辉
     * @Date 2021/9/17
     * @Description 获取支出管理 合计栏信息(累计申请金额总计，累计支付金额总计)
     * @Param
     * @Return com.ejianc.framework.core.response.CommonResponse<TotalColumnVO>
     */
    @RequestMapping(value = "getApproveTotalColumnInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TotalColumnVO> getApproveTotalColumnInfo(@RequestBody QueryParam param) {
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("approveTime", QueryParam.DESC);
        return CommonResponse.success("查询成功！", payReimburseService.getApproveTotalColumnInfo(param, false));
    }

    /**
     * @param vo
     * @description: 更新是否含有发票
     * @author songlx
     * @date: 2021/7/22
     */
    @RequestMapping(value = "/updateInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PayReimburseVO> updateInfo(@RequestBody PayReimburseVO vo) {
        PayReimburseEntity entity = BeanMapper.map(vo, PayReimburseEntity.class);
        if (vo.getId() != null && vo.getInvoiceFlag() != null) {
            //更新是否含有发票
            LambdaUpdateWrapper<PayReimburseEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.eq(PayReimburseEntity::getId, entity.getId());
            updateWrapper.set(PayReimburseEntity::getInvoiceFlag, entity.getInvoiceFlag());
            payReimburseService.update(updateWrapper);
        }

        return CommonResponse.success("修改成功！", vo);
    }

    /**
     * @Author xuannl
     * @Date 2021/11/08
     * @Description checkParams 校验是否超额
     * @param projectId
     * @param money
     */
    @RequestMapping(value = "/checkParams", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ParamsCheckSpreadVO> checkParams(Long projectId, BigDecimal money) {
        ParamsCheckSpreadVO paramsCheckVO = payReimburseService.checkParams(projectId, money);
        return CommonResponse.success("查询打印数据成功！", paramsCheckVO);
    }
    /**
     * 获取报销费用统计信息合计
     */
    @RequestMapping(value = "/getPayReimburseInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayReimburseStatisticsVO> getPayReimburseInfo(Integer dependOnProject) {
        CommonResponse<PayReimburseStatisticsVO> response= payReimburseService.getPayReimburseInfo(dependOnProject);
        return response;
    }
}
