package com.ejianc.business.finance.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.finance.service.IPayContractService;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.PayAndReceiveMnyWarnVO;
import com.ejianc.business.income.api.IIncomeContractApi;
import com.ejianc.business.income.dto.ContractDto;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @Description: 预警控制类
 * @date 2021/03/18 17:46
 */
@RestController
@RequestMapping(value = "/warn/")
public class WarnController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String PAY_WARN_NAME = "项目付款金额大于收款金额预警";

    private static final String REIMBURSE_WARN_NAME = "报销费用大于收入合同金额预警";

    private static final String BOND_EXPIRE_WARN_NAME = "保证金到期预警";

    private static final String PERCENT_UNIT = "bfh";

    private static final String MNY_UNIT = "yuan";

    private static final String PC_URL = "/ejc-report-frontend/#/proPeportDetail?id=";// 项目统计详情页

    private static final String BOND_UP_PC_URL = "/ejc-finance-frontend/#/bondManageList/bondUpDetailManage?id=";// 对上保证金详情页

    private static final String BOND_DOWN_PC_URL = "/ejc-finance-frontend/#/bondManageList/bondDownDetailManage?id=";// 对下保证金详情页

    @Autowired
    private IWarnCenterApi warnCenterApi;

    @Autowired
    private ITenantApi tenantApi;

    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private IIncomeContractApi incomeApi;

    @Autowired
    private IPayContractService contractService;

    @PostMapping(value = "payMnyWarn")
    public CommonResponse<String> payMnyWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if(StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        logger.info("忽略租户列表：{}", ignoreTenantIds);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if(!tenantResp.isSuccess()) {
            logger.error("执行" + PAY_WARN_NAME + "失败，获取有效的租户Id列表失败：{}", tenantResp.getMsg());
            return CommonResponse.error("执行" + PAY_WARN_NAME + "失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if(CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + PAY_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //1.去掉停用的预警对应的租户Id（不用任何处理）
        tenantIds.removeAll(ignoreTenantIds);
        if(CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + PAY_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.再去掉自定义租户ID，走自定义的预警条件
        List<SqlParam> sqlParamList = new ArrayList<>();
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for(JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }

        //去除自定义预警设置的对应的租户Id
        tenantIds.removeAll(customTenantIds);

        //3.除了忽略和自定义，其他都走默认预警条件
        for(Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
        }

        // 查询
        Map<String, SqlParam> paramMap = new HashMap<>();
        List<Long> tenantidList = new ArrayList<>();
        for (SqlParam param : sqlParamList) {
            paramMap.put(param.getTenantId() +  param.getUnit(), param);
            tenantidList.add(param.getTenantId());
        }

        // 付款信息 + 收款信息
        Map<Long, PayAndReceiveMnyWarnVO> payMap = contractService.getSumPayMnyMap(tenantidList);
        Map<Long, PayAndReceiveMnyWarnVO> receiveMap = contractService.getSumReceiveMnyMap(tenantidList);
        if(CollectionUtils.isEmpty(payMap.keySet())) {
            logger.error("执行" + PAY_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        // 项目信息
        List<Long> projectIds = new ArrayList<>(payMap.keySet());
        projectIds.addAll(receiveMap.keySet());
        CommonResponse<List<ProjectRegisterVO>> res = projectApi.queryProjectByIds(projectIds);
        Map<Long, ProjectRegisterVO> projectMap = new HashMap<>();
        if(res.isSuccess() && !res.getData().isEmpty()){
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectRegisterVO::getId, Function.identity()));
        }

        // 收入合同
        CommonResponse<List<ContractVo>> response = incomeApi.searchContractByTenantIds(tenantidList);
        Map<Long, BigDecimal> incomeMap = new HashMap<>();
        if(response.isSuccess() && !response.getData().isEmpty()){
            List<ContractVo> list = response.getData();
            for(ContractVo vo : list){
                if(incomeMap.containsKey(vo.getProjectId())){
                    incomeMap.put(vo.getProjectId(), MathUtil.safeAdd(incomeMap.get(vo.getProjectId()), vo.getContractTaxMny()));
                } else {
                    incomeMap.put(vo.getProjectId(), vo.getContractTaxMny());
                }
            }
        }

        // 构造转换VO
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for(Long key : payMap.keySet()){
            PayAndReceiveMnyWarnVO payVO = payMap.get(key);
            ProjectRegisterVO projectVO = projectMap.get(payVO.getProjectId());
            BigDecimal sumPayMny = payVO.getSumPayMny();
            BigDecimal sumReceiveMny = BigDecimal.ZERO;
            if(receiveMap != null && receiveMap.containsKey(key)){
                sumReceiveMny = receiveMap.get(key).getSumReceiveMny();
            }
            // 可能有两种预警条件
            String warnLevel = null;
            StringBuffer content = new StringBuffer();
            boolean flag = false;
            // 付款金额 - 收款金额 >= 收入合同金额 * X%
            if(paramMap.containsKey(payVO.getTenantId() + PERCENT_UNIT)){
                SqlParam param = paramMap.get(payVO.getTenantId() + PERCENT_UNIT);
                warnLevel = param.getWarnLevel();
                BigDecimal warnMny = MathUtil.safeMultiply(incomeMap.get(key), new BigDecimal(param.getValue()));
                if(MathUtil.safeSub(sumPayMny, sumReceiveMny).compareTo(warnMny) > 0){
                    flag = true;
                    content.append("收入合同金额*");
                    content.append(String.valueOf((param.getValue() * 100d)));
                    content.append("%：");
                    content.append(df.format(warnMny != null ? warnMny : BigDecimal.ZERO));
                    content.append("元");
                }
            }
            // 付款金额 >= 收款金额
            if(paramMap.containsKey(payVO.getTenantId() + MNY_UNIT)){
                String dou = flag ? "，" : "";
                SqlParam param = paramMap.get(payVO.getTenantId() + MNY_UNIT);
                warnLevel = param.getWarnLevel();
                BigDecimal setMny = new BigDecimal(param.getValue());
                if(MathUtil.safeSub(sumPayMny, sumReceiveMny).compareTo(setMny) > 0){
                    flag = true;
                    content.append(dou);
                    content.append(setMny + "元");
                }
            }
            if(flag){
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "")  + projectVO.getCode());
                vo.setPcTitle("项目统计");
                vo.setPcUrl(PC_URL + payVO.getProjectId().toString() + "&orgId=" + payVO.getOrgId());
                vo.setOrgId(payVO.getOrgId());
                vo.setOrgName(payVO.getOrgName());
                vo.setSourceId(payVO.getProjectId().toString());
                vo.setTenantId(payVO.getTenantId());
                vo.setWarnLevel(warnLevel);
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", payVO.getProjectName())
                                .replaceAll("#sumPayMny#", df.format(this.getBigDecimal(sumPayMny)))
                                .replaceAll("#sumReceiveMny#", df.format(this.getBigDecimal(sumReceiveMny)))
                                .replaceAll("#content#", String.valueOf(content))
                );
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    @PostMapping(value = "reimburseMnyWarn")
    public CommonResponse<String> reimburseMnyWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if(StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        logger.info("忽略租户列表：{}", ignoreTenantIds);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if(!tenantResp.isSuccess()) {
            logger.error("执行" + REIMBURSE_WARN_NAME + "失败，获取有效的租户Id列表失败：{}", tenantResp.getMsg());
            return CommonResponse.error("执行" + REIMBURSE_WARN_NAME + "失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if(CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + REIMBURSE_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //1.去掉停用的预警对应的租户Id（不用任何处理）
        tenantIds.removeAll(ignoreTenantIds);
        if(CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + REIMBURSE_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.再去掉自定义租户ID，走自定义的预警条件
        List<SqlParam> sqlParamList = new ArrayList<>();
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for(JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }

        //去除自定义预警设置的对应的租户Id
//        tenantIds.removeAll(customTenantIds);

        //3.除了忽略和自定义，其他都走默认预警条件
        for(Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
        }

        // 查询
        Map<Long, SqlParam> paramMap = new HashMap<>();
        List<Long> tenantidList = new ArrayList<>();
        for (SqlParam param : sqlParamList) {
            paramMap.put(param.getTenantId(), param);
            tenantidList.add(param.getTenantId());
        }

        // 报销信息 累计报销金额，放入sumPayMny，为了复用代码
        Map<Long, PayAndReceiveMnyWarnVO> payMap = contractService.getSumReimburseMnyMap(tenantidList);
        if(CollectionUtils.isEmpty(payMap.keySet())) {
            logger.error("执行" + REIMBURSE_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        // 项目信息
        CommonResponse<List<ProjectRegisterVO>> res = projectApi.queryProjectByIds(new ArrayList<>(payMap.keySet()));
        Map<Long, ProjectRegisterVO> projectMap = new HashMap<>();
        if(res.isSuccess() && !res.getData().isEmpty()){
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectRegisterVO::getId, Function.identity()));
        }

        // 收入合同
        CommonResponse<List<ContractDto>> response = incomeApi.postSearchContractByTenantIds(tenantidList);
        Map<Long, BigDecimal> incomeMap = new HashMap<>();
        logger.info("合同查询结果--------------------------------------------------:"+response.isSuccess());
        if(response.isSuccess() && !response.getData().isEmpty()){
            List<ContractDto> list = response.getData();
            for(ContractDto vo : list){
                if(incomeMap.containsKey(vo.getProjectId())){
                    incomeMap.put(vo.getProjectId(), MathUtil.safeAdd(incomeMap.get(vo.getProjectId()), vo.getContractTaxMny()));
                } else {
                    incomeMap.put(vo.getProjectId(), vo.getContractTaxMny());
                }
            }
        }

        // 构造转换VO
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for(Long key : payMap.keySet()){
            PayAndReceiveMnyWarnVO payVO = payMap.get(key);
            ProjectRegisterVO projectVO = projectMap.get(payVO.getProjectId());
            BigDecimal sumReimburseMny = payVO.getSumPayMny();
            BigDecimal sumIncomeMny = BigDecimal.ZERO;
            if(incomeMap != null && incomeMap.containsKey(key)){
                sumIncomeMny = incomeMap.get(key);
            }
            // 报销费用 >= 收入合同金额 * X%
            SqlParam param = paramMap.get(payVO.getTenantId());
            BigDecimal warnMny = MathUtil.safeMultiply(incomeMap.get(key), new BigDecimal(param.getValue()));
            if(sumReimburseMny.compareTo(warnMny) >= 0){
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "")  + projectVO.getCode());
                vo.setPcTitle("项目统计");
                vo.setPcUrl(PC_URL + payVO.getProjectId().toString() + "&orgId=" + payVO.getOrgId());
                vo.setOrgId(payVO.getOrgId());
                vo.setOrgName(payVO.getOrgName());
                vo.setSourceId(payVO.getProjectId().toString());
                vo.setTenantId(payVO.getTenantId());
                vo.setWarnLevel(param.getWarnLevel());
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", payVO.getProjectName())
                                .replaceAll("#sumReimburseMny#", df.format(this.getBigDecimal(sumReimburseMny)))
                                .replaceAll("#sumIncomeMny#", df.format(this.getBigDecimal(sumIncomeMny)))
                                .replaceAll("#percent#", String.valueOf((param.getValue() * 100d)))
                                .replaceAll("#warnMny#", df.format(this.getBigDecimal(warnMny)))
                );
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    @PostMapping(value = "bondExpireWarn")
    public CommonResponse<String> bondExpireWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if(StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        logger.info("忽略租户列表：{}", ignoreTenantIds);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);


        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if(!tenantResp.isSuccess()) {
            logger.error("执行" + BOND_EXPIRE_WARN_NAME + "失败，获取有效的租户Id列表失败：{}", tenantResp.getMsg());
            return CommonResponse.error("执行设备租赁合同预警失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if(CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + BOND_EXPIRE_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //去掉停用的预警对应的租户Id
        tenantIds.removeAll(ignoreTenantIds);
        if(CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行" + BOND_EXPIRE_WARN_NAME + "完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        List<SqlParam> sqlParamList = new ArrayList<>();
        JSONObject warnInfo = null;
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for(JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue());
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }
        //
        //去除自定义预警设置的对应的租户Id
        tenantIds.removeAll(customTenantIds);

        for(Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for(SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue());
                sqlParamList.add(p);
            }
        }

        List<Map<String, Object>> result = contractService.queryBondExpireWarn(sqlParamList);
        if(CollectionUtils.isEmpty(result)) {
            logger.error("执行" + BOND_EXPIRE_WARN_NAME + "完成，未发现有匹配预警规则的合同信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        return sendWarnToTask(transToWarnVO(result, defaultWarnInfo), defaultWarnInfo.getLong("warnSetId"));
    }

    private List<EarlyWarnTransVO> transToWarnVO(List<Map<String, Object>> result, JSONObject defaultWarnInfo) {
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        for(Map<String, Object> m : result) {
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "")  + m.get("bondCode").toString());
            if("up".equals(m.get("type"))) {
                vo.setPcTitle("对上保证金管理");
                vo.setPcUrl(BOND_UP_PC_URL + m.get("bondId").toString());
            } else {
                vo.setPcTitle("对下保证金管理");
                vo.setPcUrl(BOND_DOWN_PC_URL + m.get("bondId").toString());
            }
            vo.setOrgId(Long.valueOf(m.get("orgId").toString()));
            vo.setOrgName(m.get("orgName").toString());
            vo.setSourceId(m.get("bondId").toString());
            vo.setTenantId(Long.valueOf(m.get("tenantId").toString()));
            vo.setWarnLevel(m.get("warnLevel").toString());
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));
            if (Integer.parseInt(m.get("day").toString()) >= 0){
                tmpContent = "保证金【" + m.get("bondCode").toString() + "】还有" + m.get("day").toString() + "天到期，请及时办理退还；";
            }
            if (Integer.parseInt(m.get("day").toString()) < 0){
                Integer day = -Integer.parseInt(m.get("day").toString());
                //保证金【保证金名称+保函编号】已经过期-X天，请及时办理退还；
                tmpContent = "保证金【" + m.get("bondCode").toString() + "】已经过期" + day + "天，请及时办理退还；";
            }

            vo.setEarlywarnContent(tmpContent);
            resp.add(vo);
        }
        return resp;
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        logger.info("发送任务：{}", JSONObject.toJSONString(warnList));
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if(!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }

    private BigDecimal getBigDecimal(Object obj){
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

}
