package com.ejianc.business.finance.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.bean.ReceiveEntity;
import com.ejianc.business.finance.bean.ReceiveInvoiceEntity;
import com.ejianc.business.finance.bean.ReceiveQuteDetailEntity;
import com.ejianc.business.finance.enums.ReceiveTypeEnum;
import com.ejianc.business.finance.mapper.ReceiveMapper;
import com.ejianc.business.finance.service.IReceiveInvoiceService;
import com.ejianc.business.finance.service.IReceiveQuteDetailService;
import com.ejianc.business.finance.service.IReceiveService;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.utils.BigDecimalUtil;
import com.ejianc.business.finance.utils.FeignUtil;
import com.ejianc.business.finance.vo.ReceiveInvoiceVO;
import com.ejianc.business.finance.vo.ReceiveQuteDetailVO;
import com.ejianc.business.finance.vo.ReceiveVO;
import com.ejianc.business.finance.vo.SumReceiveVO;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service
public class ReceiveServiceImpl extends BaseServiceImpl<ReceiveMapper, ReceiveEntity> implements IReceiveService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String RECEIVE_BILL_CODE = "FINANCE_RECEIVE";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IReceiveInvoiceService invoiceService;

    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private IReceiveQuteDetailService receiveQuteDetailService;
    @Autowired
    private IReceiveService receiveService;

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private FeignUtil feignUtil;
    @Autowired
    private IInvoiceApi invoiceApi;

    @Override
    public ReceiveVO insertOrUpdate(ReceiveVO vo) {
        ReceiveEntity entity = BeanMapper.map(vo, ReceiveEntity.class);
        // 校验同合同、同组织下只能有一张未生效单据
        this.validateBeforeSave(entity);
        // 保存时校验合同version是否一致
        if(!ValidateUtil.validateUpstreamVersion(String.valueOf(vo.getContractId()),
                BillTypeCodeEnum.施工合同.getBillTypeCode(), vo.getContractVersion())){
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }
        // 自动生成编码
        this.autoSetBillCode(entity);
        // 保存主表
        //super.saveOrUpdate(entity);
        super.saveOrUpdate(entity,false);
        // 更新发票
        List<ReceiveInvoiceVO> invoiceVOList = this.updateInvoiceVOS(vo, entity.getId());
        // 返回VO
        ReceiveVO backVO = BeanMapper.map(super.getById(entity.getId()), ReceiveVO.class);
        backVO.setInvoiceVOList(invoiceVOList);// 发票列表
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("receiveId",new Parameter(QueryParam.EQ,entity.getId()));
        List<ReceiveQuteDetailEntity> list = receiveQuteDetailService.queryList(queryParam, false);
        List<ReceiveQuteDetailVO> vos = BeanMapper.mapList(list, ReceiveQuteDetailVO.class);
        backVO.setReceiveQuteDetailList(vos);
        return backVO;
    }

    private List<ReceiveInvoiceVO> updateInvoiceVOS(ReceiveVO receiveVO, Long id) {
        // 新增
        List<ReceiveInvoiceVO> voList = receiveVO.getInvoiceVOList();
        //
        QueryParam param = new QueryParam();
        param.getParams().put("receiveId", new Parameter(QueryParam.EQ, id));
        // 查询关联发票
        List<ReceiveInvoiceEntity> oldInvoiceList = id != null ? invoiceService.queryList(param, false) : new ArrayList<>();
        Map<Long, ReceiveInvoiceEntity> invoiceEntityMap = CollectionUtils.isNotEmpty(oldInvoiceList) ? oldInvoiceList.stream().collect(Collectors.toMap(ReceiveInvoiceEntity::getInvoiceId, Function.identity())) : new HashMap<>();
        //收票改为占用方式
        Map<Long, BigDecimal> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(voList)) {
            for (ReceiveInvoiceVO invoiceVO : voList) {
                // 保存时校验version是否一致
                if (Boolean.FALSE.equals(ValidateUtil.validateUpstreamVersion(String.valueOf(invoiceVO.getInvoiceId()),
                        BillTypeCodeEnum.税务开票.getBillTypeCode(), invoiceVO.getInvoiceVersion()))) {
                    throw new BusinessException("发票已被更新，请刷新后重做！");
                }
                invoiceVO.setReceiveId(id);
            }
            if (CollectionUtils.isNotEmpty(voList)) {
                voList.forEach(t -> {
                    BigDecimal old = t.getInvoiceId() != null && !invoiceEntityMap.isEmpty() && invoiceEntityMap.containsKey(t.getInvoiceId()) ? invoiceEntityMap.get(t.getInvoiceId()).getCurrentInvoiceTaxMny() : BigDecimal.ZERO;
                    map.put(t.getInvoiceId(), BigDecimalUtil.safeSub(t.getCurrentInvoiceTaxMny(), old));
                    invoiceEntityMap.remove(t.getInvoiceId());
                });
            }
            List<ReceiveInvoiceEntity> entityList = BeanMapper.mapList(voList, ReceiveInvoiceEntity.class);
            invoiceService.saveOrUpdateBatch(entityList, entityList.size(), false);
            voList = BeanMapper.mapList(entityList, ReceiveInvoiceVO.class);
        }
        if (!invoiceEntityMap.isEmpty()) {
            invoiceEntityMap.keySet().stream().map(invoiceEntityMap::get).forEach(receiveInvoiceEntity -> map.put(receiveInvoiceEntity.getInvoiceId(), BigDecimalUtil.convertToMinusNumber(receiveInvoiceEntity.getCurrentInvoiceTaxMny())));
        }
        if (!map.isEmpty()) {
            invoiceApi.updateUseInvoiceTaxMny(map);
        }
        // 删除
        List<Long> ids = voList.stream().map(ReceiveInvoiceVO::getId).collect(Collectors.toList());
        QueryWrapper<ReceiveInvoiceEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("receive_id", id);
        wrapper.notIn(!ids.isEmpty(), "id", ids);
        invoiceService.remove(wrapper, false);
        return voList;
    }

    @Override
    public ReceiveVO queryDetail(Long id) {
        // 查询主表
        ReceiveEntity entity = baseMapper.selectById(id);
        ReceiveVO vo = BeanMapper.map(entity, ReceiveVO.class);
        vo.setReceiveTypeName(ReceiveTypeEnum.getNameByCode(vo.getReceiveType()));
        // 查询关联发票
        QueryParam param = new QueryParam();
        param.getParams().put("receiveId", new Parameter(QueryParam.EQ, vo.getId()));
        param.getOrderMap().put("createTime", "desc");
        List<ReceiveInvoiceEntity> invoiceEntityList = invoiceService.queryList(param, false);
        vo.setInvoiceVOList(BeanMapper.mapList(invoiceEntityList, ReceiveInvoiceVO.class));
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("receiveId", new Parameter(QueryParam.EQ,id ));
        List<ReceiveQuteDetailEntity> list = receiveQuteDetailService.queryList(queryParam, false);
        List<ReceiveQuteDetailVO> vos = BeanMapper.mapList(list, ReceiveQuteDetailVO.class);
        vo.setReceiveQuteDetailList(vos);
        return vo;
    }

    @Override
    public List<ReceiveVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<ReceiveVO> resVos = (List<ReceiveVO>) queryPageJson(param, false).get("records");
        // 收款类型
        CommonResponse<List<DefdocDetailVO>> defResp = defdocApi.getDefDocByDefId(330086941239541801L);
        if(!defResp.isSuccess()){
            logger.error("获取自定义档案列表失败：{}", defResp.getMsg());
        }
        if(!resVos.isEmpty()){
            for(int i = 0 ; i< resVos.size(); i++){
                ReceiveVO vo = resVos.get(i);
                vo.setReceiveTypeName(ReceiveTypeEnum.getNameByCode(vo.getReceiveType()));
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
                DefdocDetailVO defdocDetailVO = defResp.getData().stream().filter(defVo -> defVo.getId().equals(vo.getReceiveWay())).findFirst().get();
                vo.setReceiveWayName(defdocDetailVO.getName());
            };
        }
        return resVos;
    }

    @Override
    public String delete(List<Long> ids) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, ids));
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<ReceiveEntity> entityList = super.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(entityList)) {
            QueryWrapper<ReceiveInvoiceEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.in("receive_id",ids);
            List<ReceiveInvoiceEntity> list = invoiceService.list(queryWrapper);
            Map<Long, BigDecimal> map = new HashMap<>();
            if (CollectionUtils.isNotEmpty(list)){
                for (ReceiveInvoiceEntity receiveInvoiceEntity : list) {
                    map.put(receiveInvoiceEntity.getInvoiceId(),BigDecimalUtil.convertToMinusNumber(receiveInvoiceEntity.getCurrentInvoiceTaxMny()));
                }
                invoiceApi.updateUseInvoiceTaxMny(map);
            }
            super.removeByIds(ids, false);
            // 删除发票
            invoiceService.remove(new QueryWrapper<ReceiveInvoiceEntity>().in("receive_id", ids), false);
        }
        return "删除成功！";
    }

    @Override
    public JSONObject queryPageJson(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("contractName");
        fuzzyFields.add("receiveUnitName");
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<ReceiveEntity> pageData = super.queryPage(param,false);
        JSONObject page = new JSONObject();
        List<ReceiveVO> receiveVOS = BeanMapper.mapList(pageData.getRecords(), ReceiveVO.class);
        // 收款类型
        CommonResponse<List<DefdocDetailVO>> defResp = defdocApi.getDefDocByDefId(330086941239541801L);
        if(!defResp.isSuccess()){
            logger.error("获取自定义档案列表失败：{}", defResp.getMsg());
        }
        Map<Long, String> defMap = defResp.getData().stream().collect(Collectors.toMap(DefdocDetailVO::getId, DefdocDetailVO::getName));
        //收款类型名称赋值
        receiveVOS.forEach(item->{
            //DefdocDetailVO defdocDetailVO = defResp.getData().stream().filter(defVo -> defVo.getId().equals(item.getReceiveWay())).findFirst().get();
            item.setReceiveWayName(defMap.get(item.getReceiveWay()));
        });
        page.put("records", receiveVOS);
        page.put("total", pageData.getTotal());
        page.put("current", pageData.getCurrent());
        page.put("size", pageData.getSize());
        page.put("pages", pageData.getPages());
        return page;
    }

    @Override
    public SumReceiveVO getSumReceiveVOList(Long contractId, Long receiveType) {
        QueryParam param = new QueryParam();
        param.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        if (receiveType != null) {
            param.getParams().put("receiveType", new Parameter(QueryParam.EQ, receiveType));
        }
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("createTime", QueryParam.DESC);
        List<ReceiveEntity> entityList = super.queryList(param, false);
        // 累计收款金额
        BigDecimal sumReceiveMny = BigDecimal.ZERO;
        for(ReceiveEntity entity : entityList){
            sumReceiveMny = MathUtil.safeAdd(sumReceiveMny, entity.getReceiveMny());
        }
        SumReceiveVO vo = new SumReceiveVO();
        vo.setContractId(contractId);
        vo.setSumReceiveMny(sumReceiveMny);
        vo.setReceiveVOList(BeanMapper.mapList(entityList, ReceiveVO.class));
        return vo;
    }

    @Override
    public Map<String, Object> countRecAmt(Long tenantId, List<Long> projectIds,List<Long> orgIds, boolean isWithContract) {
        QueryWrapper<ReceiveEntity> queryWrapper = new QueryWrapper<>();
        //指定租户下的
        queryWrapper.eq("tenant_id", tenantId);
        queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
        //有效的单据
        queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
        //匹配的项目范围
        if(CollectionUtils.isNotEmpty(projectIds)) {
            queryWrapper.in("project_id", projectIds);
        } else {
            //2-项目收款
            queryWrapper.isNotNull("project_id");
        }
        if(isWithContract) {
            //查询有合同的收入
            queryWrapper.isNotNull("contract_id");
        }
        if(CollectionUtils.isNotEmpty(orgIds)) {
            queryWrapper.in("org_id", orgIds);
        }
        queryWrapper.select(" round(ifnull(sum(receive_mny),0) / 10000, 2) as amt ");
        return super.getMap(queryWrapper);
    }

    @Override
    public List<ReceiveVO> queryProjectReceiveMny(List<Long> projectIds) {
        return baseMapper.queryProjectReceiveMny(projectIds);
    }

    @Override
    public List<ReceiveVO> queryInfoQuoteId(Long quoteId,Long contractId) {
        return baseMapper.queryInfoQuoteId(quoteId,contractId);
    }

    @Override
    public ReceiveVO queryInfoContractId(Long contractId) {
        return baseMapper.queryInfoContractId(contractId);
    }

    @Override
    public List<ReceiveVO> queryTotalReceiveMny(List<Long> quoteIds) {
        return baseMapper.queryTotalReceiveMny(quoteIds);
    }

    @Override
    public List<ReceiveQuteDetailVO> getLastReceiveTaxMny(List<ReceiveQuteDetailVO> vos) {
        List<Long> quoteIds = vos.stream().map(ReceiveQuteDetailVO::getQuoteId).collect(Collectors.toList());
        List<ReceiveVO> receiveVOS= receiveService.queryTotalReceiveMny(quoteIds);
        Map<Long, List<ReceiveVO>> map =new HashMap<>();
        if (CollectionUtils.isNotEmpty(receiveVOS)){
            map= receiveVOS.stream().collect(Collectors.groupingBy(ReceiveVO::getQuoteId));
        }
        for (ReceiveQuteDetailVO vo : vos) {
            //甲方报量的维度 统计不含本期累计收款金额
            BigDecimal totalReceiveTaxMny = BigDecimal.ZERO;
            BigDecimal totalContractReceiveMny=BigDecimal.ZERO;
            Long contractId = vo.getContractId();
            ReceiveVO receiveVO = receiveService.queryInfoContractId(contractId);
            if (receiveVO!=null){
              totalContractReceiveMny = receiveVO.getTotalContractReceiveMny();
            }
            if (map.containsKey(vo.getQuoteId())){
                ReceiveVO receive = map.get(vo.getQuoteId()).get(0);
                if (receive!=null){
                    totalReceiveTaxMny=receive.getTotalReceiveTaxMny();
                }
            }
            vo.setLastReceiveTaxMny(totalReceiveTaxMny);
            vo.setTotalReceiveTaxMny(totalReceiveTaxMny);

            vo.setLastContractReceiveTaxMny(totalContractReceiveMny);
            vo.setTotalContractReceiveTaxMny(totalContractReceiveMny);
        }
        return vos;
    }

    @Override
    public ReceiveVO queryInfoProjectId(Long projectId) {
        return baseMapper.queryInfoProjectId(projectId);
    }

    /**
     * 校验同合同、同组织下只能有一张未生效单据
     * @param entity
     */
    private void validateBeforeSave(ReceiveEntity entity) {
        LambdaQueryWrapper<ReceiveEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ReceiveEntity::getContractId, entity.getContractId());
//        lambda.eq(ReceiveEntity::getOrgId, entity.getOrgId());
        lambda.eq(ReceiveEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambda.ne(entity.getId() != null && entity.getId() > 0, ReceiveEntity::getId, entity.getId());
        lambda.notIn(ReceiveEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()));
        List<ReceiveEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("该合同下存在未生效单据,不允许新增!");
        }
    }

    /**
     * 自动生成编码
     * @param entity
     */
    private void autoSetBillCode(ReceiveEntity entity) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if(StringUtils.isEmpty(entity.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(RECEIVE_BILL_CODE, tenantId);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        //修改  校验合同编号是否重复
        LambdaQueryWrapper<ReceiveEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ReceiveEntity::getBillCode, entity.getBillCode());
        lambda.eq(ReceiveEntity::getTenantId, tenantId);
        lambda.ne(entity.getId() != null && entity.getId() > 0, ReceiveEntity::getId, entity.getId());
        List<ReceiveEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
    }

}
