package com.ejianc.business.finance.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.accplat.consts.SystemCodeEnum;
import com.ejianc.business.equipment.api.IEquipmentContractApi;
import com.ejianc.business.equipment.api.ISettlementApi;
import com.ejianc.business.equipment.vo.SettlementVO;
import com.ejianc.business.finance.bean.*;
import com.ejianc.business.finance.service.*;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.utils.NumberToCN;
import com.ejianc.business.finance.vo.DeductionVO;
import com.ejianc.business.finance.vo.PayContractVO;
import com.ejianc.business.material.api.IMaterialSettlementApi;
import com.ejianc.business.sub.api.ISubContractForPayApi;
import com.ejianc.business.sub.vo.SubSettleVO;
import com.ejianc.business.voucher.api.IVoucherApi;
import com.ejianc.business.voucher.callable.AccplatVoucherCallable;
import com.ejianc.business.voucher.consts.VoucherFlag;
import com.ejianc.business.voucher.consts.VoucherOptFlag;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * <p>
 * 合同付款申请审批操作业务类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service("payContract")
public class PayContractBpmServiceImpl implements ICommonBusinessService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IPayContractService service;

    @Autowired
    private IPayContractSettleService settleService;

    @Autowired
    private IPayRecordService recordService;

    @Autowired
    private ISubContractForPayApi subApi;

    @Autowired
    private IEquipmentContractApi equipmentApi;

    @Autowired
    private ISettlementApi equipmentSettleApi;

    @Autowired
    private IMaterialSettlementApi materialApi;

    @Autowired
    private IPayContractDeductionService contractDeductionService;

    @Autowired
    private IDeductionService deductionService;

    @Autowired
    private IVoucherApi voucherApi;

    @Autowired
    private HttpServletRequest request;


    /**
     * 提交前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        //TODO
        return CommonResponse.success();
    };

    /**
     * 提交完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode){
        //TODO
        return CommonResponse.success();
    }

    /**
     * 终审审核前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        //TODO
        return CommonResponse.success();
    }

    /**
     * 终审审核完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        PayContractEntity entity = service.selectById(billId);
//        //回写累计付款与累计预付款金额  确认时回写修改
//        service.writeBackSumPayMny(billId, entity, true);
        //回写结算单累计申请金额、剩余可申请金额
        writeBackSumApplyMny(billId, entity, true);

        LambdaUpdateWrapper<PayContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(PayContractEntity::getId, billId);
        updateWrapper.set(PayContractEntity::getApproveTime, new Date());
        service.update(updateWrapper);
        QueryWrapper<PayContractDeductionEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("payapply_id", billId);
        List<PayContractDeductionEntity> list = contractDeductionService.list(wrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            List<Long> dedIds = list.stream().map(PayContractDeductionEntity::getDeductionId).collect(Collectors.toList());
            this.updateDeductionUseFlag(dedIds, 1,entity.getSubContractTypeId(),entity.getSubContractType());
        }
        return CommonResponse.success("回调处理成功！");
    }

    /**
     * 弃审前事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        PayContractEntity entity = service.selectById(billId);
        // 校验审批时间
        if (!validateApproveTime(entity)) return CommonResponse.error("该合同在本组织下非最新付款申请，不能撤回!");
        // 校验存在未生效单据
        if (!validateEffective(entity)) return CommonResponse.error("该合同在本组织下存在未生效付款申请，不能撤回!");
        // 校验结算单预付款抵扣引用
        if(entity.getFeeType() != null && entity.getFeeType() == 1 && entity.getPayStatus() == 2 && !validatePrepayMny(entity)){
            return CommonResponse.error("该合同付款申请已被结算单预付款抵扣引用,不能弃审!");
        }
        return CommonResponse.success("校验通过");
    }

    /**
     * 弃审后事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        PayContractEntity entity = service.selectById(billId);
        // 删除付款记录表，因service.writeBackSumPayMny已全部撤回已支付合同及结算单，则不需要重新撤回结算单已支付金额，如需要再处理
        QueryWrapper<PayRecordEntity> recordQueryWrapper = new QueryWrapper<PayRecordEntity>().eq("payapply_id", billId);
        List<PayRecordEntity> recordEntities = recordService.list(recordQueryWrapper);
        //进行过合并支付的数据不允许撤回
        long count = recordEntities.stream().filter(v -> null != v.getMergeConfirmId()).count();
        if (count>0){
            return CommonResponse.error("该合同付款申请已进行合并支付,不能撤回!");
        }
        // 已确认回写，未确认不回写
        if (2 == entity.getPayStatus()) {
            //回写累计付款与累计预付款金额
            service.writeBackSumPayMny(billId, entity, false);
        }
        //回写结算单累计申请金额、剩余可申请金额
        writeBackSumApplyMny(billId, entity, false);

        // 逆回写时，重置付款申请子表已支付金额
        LambdaUpdateWrapper<PayContractSettleEntity> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(PayContractSettleEntity::getPayapplyId, billId);
        wrapper.set(PayContractSettleEntity::getSumPayMny, BigDecimal.ZERO);
        if (Optional.ofNullable(entity.getCloseFlag()).orElse(false)) {
            // 还原子表申请金额
            wrapper.setSql("body_apply_mny = body_apply_mny_before_close");

            // 还原付款申请主表申请金额
            entity.setApplyMny(entity.getApplyMnyBeforeClose());
            entity.setApplyMnyCn(NumberToCN.number2CN(entity.getApplyMny()));
            // 重置付款申请主表关闭字段
            entity.setApplyMnyBeforeClose(null);
            entity.setCloseFlag(Boolean.FALSE);
            entity.setCloseState(null);
            entity.setCloseTime(null);
            entity.setCloseUser(null);
        }
        settleService.update(wrapper);

        entity.setPayMny(null);
        entity.setPayStatus(1);
        entity.setApproveTime(null);
        entity.setConfirmTime(null);
        service.saveOrUpdate(entity);

        recordService.remove(recordQueryWrapper);
        if(CollectionUtils.isNotEmpty(recordEntities)){
            // 支付记录有凭证则删除
            List<PayRecordEntity> listRec = recordEntities.stream().filter(t -> VoucherFlag.SUCCESS.equals(t.getVoucherFlag())).collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(listRec)){
                ExecutorService threadPool = Executors.newFixedThreadPool(listRec.size());
                String authority = request.getHeader("authority");
                if (authority == null) {
                    logger.info("request-authority为空");
                    authority = (String) InvocationInfoProxy.getExtendAttribute("authority");
                }
                try {
                    for (PayRecordEntity recordEntity : listRec) {
                        VoucherParams voucherParamsDed = VoucherParams.newInstanceByOrgId(PayContractVO.BILL_TYPE_CODE, recordEntity, SystemCodeEnum.FINANCE);
                        Callable<CommonResponse> voucherCallable = new AccplatVoucherCallable(voucherApi, recordService, voucherParamsDed, VoucherOptFlag.DEL, RequestContextHolder.getRequestAttributes(), authority);
                        threadPool.submit(voucherCallable);
                    }
                } catch (Exception e) {
                    logger.error("合同付款申请撤回凭证异常, ", e);
                } finally {
                    threadPool.shutdown();
                }
            }
        }

        QueryWrapper<PayContractDeductionEntity> dedWrapper = new QueryWrapper<>();
        dedWrapper.eq("payapply_id", billId);
        List<PayContractDeductionEntity> payContractDeductionEntityList = contractDeductionService.list(dedWrapper);
        List<Long> dedIdList = payContractDeductionEntityList.stream().map(PayContractDeductionEntity::getDeductionId).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(payContractDeductionEntityList)) {
            //有凭证则删除
            LambdaUpdateWrapper<DeductionEntity> deductionEntityLambdaUpdateWrapper = new LambdaUpdateWrapper<>();
            deductionEntityLambdaUpdateWrapper.in(DeductionEntity::getId, dedIdList);
            deductionEntityLambdaUpdateWrapper.eq(DeductionEntity::getVoucherFlag, VoucherFlag.SUCCESS);
            List<DeductionEntity> deductionEntityList = deductionService.list(deductionEntityLambdaUpdateWrapper);
            if (CollectionUtils.isNotEmpty(deductionEntityList)) {
                ExecutorService threadPool = Executors.newFixedThreadPool(deductionEntityList.size());
                String authority = request.getHeader("authority");
                if (authority == null) {
                    logger.info("request-authority为空");
                    authority = (String) InvocationInfoProxy.getExtendAttribute("authority");
                }
                try {
                    for (DeductionEntity deductionEntity : deductionEntityList) {
                        VoucherParams voucherParamsDed = VoucherParams.newInstanceByOrgId(DeductionVO.BILL_TYPE_CODE, deductionEntity, SystemCodeEnum.FINANCE);
                        Callable<CommonResponse> voucherCallable = new AccplatVoucherCallable(voucherApi, deductionService, voucherParamsDed, VoucherOptFlag.DEL, RequestContextHolder.getRequestAttributes(), authority);
                        threadPool.submit(voucherCallable);
                    }
                } catch (Exception e) {
                    logger.error("合同付款申请扣款单撤回凭证异常, ", e);
                } finally {
                    threadPool.shutdown();
                }
            }
            this.updateDeductionUseFlag(dedIdList, 0, null, null);
        }

        logger.info("进入审批后回写--- entity:" + entity);
        return CommonResponse.success("回调处理成功！");
    }

    /**
     * 回写结算单累计申请金额、剩余可申请金额
     * @param billId
     * @param entity
     * @param flag
     */
    public boolean writeBackSumApplyMny(Long billId, PayContractEntity entity, boolean flag) {
        List<PayContractSettleEntity> settleEntityList = settleService.list(new QueryWrapper<PayContractSettleEntity>().eq("payapply_id", billId));
        // 结算单对应本期申请金额
        Map<Long, BigDecimal> settleApplyMnyMap  = new HashMap<>();
        for(PayContractSettleEntity settleEntity : settleEntityList){
            // 本期申请金额   true为回写，false为逆回写
            BigDecimal applyMny = flag ? settleEntity.getBodyApplyMny() : MathUtil.safeSub(new BigDecimal(0), settleEntity.getBodyApplyMny());
            settleApplyMnyMap.put(settleEntity.getSettleId(), applyMny);
        }
        if(entity.getContractType() == 1){//分包
            for(PayContractSettleEntity settleEntity : settleEntityList){
                subApi.updateSubSettleSumApplyMny(settleEntity.getSettleId(), settleApplyMnyMap.get(settleEntity.getSettleId()));//结算单
            }
        }
        if(entity.getContractType() == 2){//设备采购
            for(PayContractSettleEntity settleEntity : settleEntityList){
                equipmentApi.updatePurchaseSettleSumApplyMny(settleEntity.getSettleId(), settleApplyMnyMap.get(settleEntity.getSettleId()));//结算单
            }
        }
        if(entity.getContractType() == 3){//设备租赁
            for(PayContractSettleEntity settleEntity : settleEntityList){
                equipmentApi.updateRentSettleSumApplyMny(settleEntity.getSettleId(), settleApplyMnyMap.get(settleEntity.getSettleId()));//结算单
            }
        }
        if(entity.getContractType() == 4){//物资采购
            for(PayContractSettleEntity settleEntity : settleEntityList){
                materialApi.updateSettlementBillAlreadyApplyAmount(settleEntity.getSettleId(), settleApplyMnyMap.get(settleEntity.getSettleId()));//结算单
            }
        }
        if(entity.getContractType() == 5){//物资租赁
            //TODO
        }
        return true;
    }

    /**
     * 校验存在未生效单据
     * @param entity
     * @return
     */
    private boolean validateEffective(PayContractEntity entity) {
        Long contractId = entity.getContractId();
        QueryWrapper wrapper = new QueryWrapper<PayContractEntity>();
        wrapper.eq("contract_id", contractId);
        wrapper.eq("org_id", entity.getOrgId());
        wrapper.ne("bill_state", 1);
        wrapper.ne("bill_state", 3);
        List<PayContractEntity> list = service.list(wrapper);
        if(!list.isEmpty()){
            return false;
        }
        return true;
    }

    /**
     * 校验审批时间
     * @param entity
     * @return
     */
    private boolean validateApproveTime(PayContractEntity entity) {
        Long contractId = entity.getContractId();
//        Date approveTime = entity.getConfirmTime() != null ? entity.getConfirmTime() : entity.getApproveTime();
        QueryWrapper wrapper = new QueryWrapper<PayContractEntity>();
        wrapper.eq("contract_id", contractId);
        wrapper.eq("org_id", entity.getOrgId());
        /* edit by yqls 2021-06-16 使用申请时间判断 */
//        wrapper.gt("approve_time", approveTime);
        wrapper.gt("apply_time", entity.getApplyTime());
        List<PayContractEntity> list = service.list(wrapper);
        if(!list.isEmpty()){
            return false;
        }
        return true;
    }
    private boolean updateDeductionUseFlag(List<Long> dedIds, int flag,Long subContractId,String subcontractName) {
        LambdaUpdateWrapper<DeductionEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(DeductionEntity::getId, dedIds);
        updateWrapper.set(DeductionEntity::getDeductionFlag, flag);
        updateWrapper.set(DeductionEntity::getSubContractTypeId, subContractId);
        updateWrapper.set(DeductionEntity::getSubContractType, subcontractName);
        return deductionService.update(updateWrapper);
    }
    /**
     * 结算单预付款抵扣引用
     * 合同、所属组织的剩余预付款必须大于等于当前弃审单据合同预付款金额
     * 剩余预付款 = 合同与所属组织的累计预付款-合同与所属组织累计冲抵金额
     * @param entity
     * @return
     */
    private boolean validatePrepayMny(PayContractEntity entity) {
        Long contractId = entity.getContractId();
        Long orgId = entity.getOrgId();
        QueryWrapper wrapper = new QueryWrapper();
        wrapper.eq("contract_id", contractId);
        wrapper.eq("org_id", orgId);
        List<PayContractEntity> list = service.list(wrapper);
        // 累计预付款
        BigDecimal sumPrepayMny = null;
        for(PayContractEntity data : list){
            if(data.getFeeType() == 1){
                sumPrepayMny = MathUtil.safeAdd(sumPrepayMny, data.getPayMny());
            }
        }
        // 累计冲抵金额
        BigDecimal sumOffsetMny = null;
        if(entity.getContractType() == 1){//分包
            CommonResponse<SubSettleVO> settleResponse = subApi.getSubSettleById(contractId, orgId);
            if(settleResponse.isSuccess()){
                sumOffsetMny = settleResponse.getData().getSumOffsetMny();
            }
        }
        if(entity.getContractType() == 2){//设备采购
            CommonResponse<SettlementVO> settleResponse = equipmentSettleApi.getPurchaseSettlementById(contractId, orgId);
            if(settleResponse.isSuccess()){
                sumOffsetMny = settleResponse.getData().getSumOffsetMny();
            }
        }
        if(entity.getContractType() == 3){//设备租赁
            CommonResponse<SettlementVO> settleResponse = equipmentSettleApi.getRentSettlementById(contractId, orgId);
            if(settleResponse.isSuccess()){
                sumOffsetMny = settleResponse.getData().getSumOffsetMny();
            }
        }
        if(entity.getContractType() == 4){//物资采购
            CommonResponse<BigDecimal> materialResponse = materialApi.getTotalOffsetAmountByContractIdAndOrgId(contractId, orgId);
            if(materialResponse.isSuccess()){
                sumOffsetMny = materialResponse.getData();
            }
        }
        if(entity.getContractType() == 5){//物资租赁
            //TODO
        }
        BigDecimal prepayMny = entity.getPayMny() != null ? entity.getPayMny() : new BigDecimal(0);
        if(prepayMny.compareTo(MathUtil.safeSub(sumPrepayMny, sumOffsetMny)) > 0){
            logger.info("prepayMny:" + prepayMny + "> sumPrepayMny:" + sumPrepayMny + " - sumOffsetMny:" + sumOffsetMny);
            return false;
        }
        return true;
    }
}
