package com.ejianc.controller;

import cn.hutool.core.util.RandomUtil;
import cn.hutool.crypto.digest.MD5;
import cn.hutool.http.ContentType;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.entity.TemplateEntity;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.ITemplateCategoryService;
import com.ejianc.service.ITemplateService;
import com.ejianc.vo.FileUploadResult;
import feign.Response;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * 模板控制器
 * @author: 刘家乐
 * @version: 1
 * @document: 模板管理相关接口
 */
@RestController
@RequestMapping("template")
public class TemplateController {

    // ==================== 常量定义 ====================
    
    /** 业务代码 */
    private static final String BILL_CODE = "LJL_TEMPLATE";
    
    /** RPA相关配置常量 */
    private static final String RPA_BASE_URL = "http://10.136.208.23:16106/";
    private static final String RPA_APP_KEY = "OL5E5raV";
    private static final String RPA_SECRET_KEY = "r7Oi2ITQee5OHuYCBAsShCD7wgP7xoqHJjr1lucr";
    private static final String RPA_TOKEN_URL = "/rpa/open/api/token/applyNoUser";
    private static final String RPA_UPLOAD_URL = "/rpa/api/upload/uploadFile";
    private static final String RPA_SUCCESS_CODE = "0000";
    private static final int RPA_TOKEN_TIMEOUT = 10000; // 10秒
    private static final int RPA_UPLOAD_TIMEOUT = 30000; // 30秒

    // ==================== 成员变量 ====================
    
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BaseHost;

    @Autowired
    private ITemplateService templateService;

    @Autowired
    private ITemplateCategoryService categoryService;

    @Autowired
    private IAttachmentApi attachmentApi;

    // ==================== 公共方法 ====================

    /**
     * 分页查询模板列表（封装写法）
     * @param params 查询参数
     * @return 返回分页数据
     */
    @ApiOperation(value = "分页查询模板列表（封装写法）")
    @RequestMapping(value = "/queryPage", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<TemplateEntity>> queryPage(@RequestBody Map<String, Object> params) {
        // 从Map中获取分页参数
        long current = Long.parseLong(params.get("pageIndex").toString());
        long size = Long.parseLong(params.get("pageSize").toString());

        // 创建分页对象
        Page<TemplateEntity> page = new Page<>(current, size);

        // 创建查询条件
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 根据Map中的参数添加查询条件
        if (params.containsKey("templateName") && StringUtils.isNotEmpty((String) params.get("templateName"))) {
            queryWrapper.like(TemplateEntity::getTemplateName, params.get("templateName"));
        }

        if (params.containsKey("categoryId") && params.get("categoryId") != null) {
            queryWrapper.eq(TemplateEntity::getCategoryId, params.get("categoryId"));
        }

        if (params.containsKey("enableStatus") && params.get("enableStatus") != null) {
            queryWrapper.eq(TemplateEntity::getEnableStatus, params.get("enableStatus"));
        }

        if (params.containsKey("fileType") && StringUtils.isNotEmpty((String) params.get("fileType"))) {
            queryWrapper.eq(TemplateEntity::getFileType, params.get("fileType"));
        }

        // 添加排序条件
        queryWrapper.orderByDesc(TemplateEntity::getCreateTime);

        // 执行分页查询
        IPage<TemplateEntity> result = templateService.page(page, queryWrapper);

        return CommonResponse.success("查询列表数据成功！", result);
    }

    /**
     * 复杂查询条件的原生MyBatis-Plus写法
     * @param current 当前页
     * @param size 每页大小
     * @param templateName 模板名称（模糊查询）
     * @param categoryId 分类ID（精确查询）
     * @param fileTypes 文件类型列表（IN查询）
     * @param minFileSize 最小文件大小
     * @param maxFileSize 最大文件大小
     * @return 分页数据
     */
    @ApiOperation(value = "复杂查询条件的原生MyBatis-Plus写法")
    @GetMapping("/queryPageComplex")
    @ResponseBody
    public CommonResponse<IPage<TemplateEntity>> queryPageComplex(
            @ApiParam(value = "当前页", defaultValue = "1") @RequestParam(defaultValue = "1") long current,
            @ApiParam(value = "每页大小", defaultValue = "10") @RequestParam(defaultValue = "10") long size,
            @ApiParam(value = "模板名称（模糊查询）") @RequestParam(required = false) String templateName,
            @ApiParam(value = "分类ID（精确查询）") @RequestParam(required = false) Long categoryId,
            @ApiParam(value = "文件类型列表（IN查询）") @RequestParam(required = false) String[] fileTypes,
            @ApiParam(value = "最小文件大小") @RequestParam(required = false) Integer minFileSize,
            @ApiParam(value = "最大文件大小") @RequestParam(required = false) Integer maxFileSize) {

        // 创建分页对象
        Page<TemplateEntity> page = new Page<>(current, size);

        // 创建查询条件
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 模板名称模糊查询
        if (StringUtils.isNotEmpty(templateName)) {
            queryWrapper.like(TemplateEntity::getTemplateName, templateName);
        }

        // 分类ID精确查询
        if (categoryId != null) {
            queryWrapper.eq(TemplateEntity::getCategoryId, categoryId);
        }

        // 文件类型IN查询
        if (fileTypes != null && fileTypes.length > 0) {
            queryWrapper.in(TemplateEntity::getFileType, (Object[]) fileTypes);
        }

        // 文件大小范围查询
        if (minFileSize != null) {
            queryWrapper.ge(TemplateEntity::getFileSize, minFileSize);
        }
        if (maxFileSize != null) {
            queryWrapper.le(TemplateEntity::getFileSize, maxFileSize);
        }

        // 只查询启用的模板
        queryWrapper.eq(TemplateEntity::getEnableStatus, 1);

        // 添加排序条件
        queryWrapper.orderByDesc(TemplateEntity::getCreateTime)
                .orderByAsc(TemplateEntity::getSequence);

        // 执行分页查询
        IPage<TemplateEntity> result = templateService.page(page, queryWrapper);

        return CommonResponse.success("查询列表数据成功！", result);
    }

    /**
     * 使用原生MyBatis-Plus的selectMaps方法进行统计查询
     * @return 统计结果
     */
    @ApiOperation(value = "使用原生MyBatis-Plus的selectMaps方法进行统计查询")
    @GetMapping("/statistics")
    @ResponseBody
    public CommonResponse<Map<String, Object>> getStatistics() {
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();

        // 统计总模板数
        long totalCount = templateService.count(queryWrapper);

        // 统计各文件类型的模板数量
        queryWrapper.select(TemplateEntity::getFileType, TemplateEntity::getId);
        queryWrapper.groupBy(TemplateEntity::getFileType);

        // 统计启用的模板数量
        LambdaQueryWrapper<TemplateEntity> enabledWrapper = new LambdaQueryWrapper<>();
        enabledWrapper.eq(TemplateEntity::getEnableStatus, 1);
        long enabledCount = templateService.count(enabledWrapper);

        // 构建统计结果
        Map<String, Object> statistics = new HashMap<>();
        statistics.put("totalCount", totalCount);
        statistics.put("enabledCount", enabledCount);
        statistics.put("disabledCount", totalCount - enabledCount);

        return CommonResponse.success("统计查询成功！", statistics);
    }

    /**
     * 根据两个文件ID获取文件信息并上传到RPA进行对比
     * @param firstFileId 第一个文件ID
     * @param secondFileId 第二个文件ID
     * @return 包含对比URL的响应
     */
    @RequestMapping(value = "/compareFilesById", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> compareFilesById(@RequestParam("firstFileId") Long firstFileId,
                                                       @RequestParam("secondFileId") Long secondFileId) {
        logger.info("开始文件对比，文件ID: {}, {}", firstFileId, secondFileId);

        try {
            // 1. 参数校验
            CommonResponse<JSONObject> validationResult = validateParameters(firstFileId, secondFileId);
            if (!validationResult.isSuccess()) {
                return validationResult;
            }

            // 2. 获取RPA访问令牌
            String openToken = getOpenToken();
            if (StringUtils.isBlank(openToken)) {
                return CommonResponse.error("获取RPA访问令牌失败");
            }

            // 3. 处理文件上传
            FileUploadResult firstFileResult = processFileUpload(firstFileId, "第一个文件");
            if (!firstFileResult.isSuccess()) {
                return CommonResponse.error(firstFileResult.getErrorMessage());
            }

            FileUploadResult secondFileResult = processFileUpload(secondFileId, "第二个文件");
            if (!secondFileResult.isSuccess()) {
                return CommonResponse.error(secondFileResult.getErrorMessage());
            }

            // 4. 构建并返回结果
            JSONObject result = buildComparisonResult(openToken, firstFileResult, secondFileResult);
            logSuccess(firstFileId, secondFileId, firstFileResult, secondFileResult);

            return CommonResponse.success("文件对比准备成功", result);

        } catch (Exception e) {
            logger.error("文件对比处理失败，文件ID: {}, {}", firstFileId, secondFileId, e);
            return CommonResponse.error("文件对比处理失败: " + e.getMessage());
        }
    }

    // ==================== 私有方法（按调用顺序排列） ====================

    // -------- 第一层：compareFilesById直接调用的方法 --------

    /**
     * 参数校验
     */
    private CommonResponse<JSONObject> validateParameters(Long firstFileId, Long secondFileId) {
        if (firstFileId == null || secondFileId == null) {
            return CommonResponse.error("文件ID不能为空");
        }
        if (firstFileId.equals(secondFileId)) {
            return CommonResponse.error("不能对比同一个文件");
        }
        return CommonResponse.success("参数校验通过", null);
    }

    /**
     * 获取RPA访问令牌
     * @return RPA访问令牌，失败时返回null
     */
    private String getOpenToken() {
        try {
            // 生成签名参数
            String timestamp = System.currentTimeMillis() + "";
            String random = RandomUtil.randomString(12);
            String sign = MD5.create().digestHex(RPA_APP_KEY + timestamp + RPA_SECRET_KEY + random);

            // 构建请求体
            JSONObject requestBody = new JSONObject()
                    .fluentPut("appKey", RPA_APP_KEY)
                    .fluentPut("timestamp", timestamp)
                    .fluentPut("random", random)
                    .fluentPut("sign", sign);

            logger.debug("请求RPA访问令牌，参数: {}", requestBody.toString());

            // 发送请求
            HttpResponse response = HttpRequest.post(RPA_BASE_URL + RPA_TOKEN_URL)
                    .contentType(ContentType.JSON.getValue())
                    .body(requestBody.toString())
                    .timeout(RPA_TOKEN_TIMEOUT)
                    .execute();

            return parseTokenResponse(response);

        } catch (Exception e) {
            logger.error("获取RPA访问令牌时发生异常", e);
            return null;
        }
    }

    /**
     * 处理单个文件的上传流程
     * @param fileId 文件ID
     * @param fileDescription 文件描述（用于错误信息）
     * @return 文件上传结果
     */
    private FileUploadResult processFileUpload(Long fileId, String fileDescription) {
        try {
            // 获取文件响应
            Response fileResponse = attachmentApi.downloadFileById(fileId);
            if (fileResponse == null || fileResponse.body() == null) {
                return FileUploadResult.failure(fileDescription + "无法获取，文件ID: " + fileId);
            }

            // 获取文件详细信息
            CommonResponse<AttachmentVO> attachmentResponse = attachmentApi.queryDetail(String.valueOf(fileId));
            if (!attachmentResponse.isSuccess() || attachmentResponse.getData() == null) {
                return FileUploadResult.failure(fileDescription + "详细信息获取失败，文件ID: " + fileId);
            }

            String fileName = attachmentResponse.getData().getFileName();
            if (StringUtils.isBlank(fileName)) {
                return FileUploadResult.failure(fileDescription + "文件名为空，文件ID: " + fileId);
            }

            // 上传文件到RPA
            String rpaFileId = uploadToRPA(fileResponse, fileId, fileName);
            if (StringUtils.isBlank(rpaFileId)) {
                return FileUploadResult.failure(fileDescription + "上传RPA失败，文件ID: " + fileId);
            }

            return FileUploadResult.success(fileName, rpaFileId);

        } catch (Exception e) {
            logger.error("处理文件上传时发生异常，文件ID: {}", fileId, e);
            return FileUploadResult.failure(fileDescription + "处理失败: " + e.getMessage());
        }
    }

    /**
     * 构建对比结果
     */
    private JSONObject buildComparisonResult(String openToken, FileUploadResult firstFileResult, FileUploadResult secondFileResult) {
        String comparisonUrl = buildComparisonUrl(openToken, firstFileResult, secondFileResult);
        
        JSONObject result = new JSONObject();
        result.put("url", comparisonUrl);
        result.put("firstFileName", firstFileResult.getFileName());
        result.put("secondFileName", secondFileResult.getFileName());
        result.put("firstFileRpaId", firstFileResult.getRpaFileId());
        result.put("secondFileRpaId", secondFileResult.getRpaFileId());
        
        return result;
    }

    /**
     * 记录成功日志
     */
    private void logSuccess(Long firstFileId, Long secondFileId, FileUploadResult firstFileResult, FileUploadResult secondFileResult) {
        logger.info("文件对比准备完成，文件ID: {}, {}, RPA文件ID: {}, {}",
                firstFileId, secondFileId, firstFileResult.getRpaFileId(), secondFileResult.getRpaFileId());
    }

    // -------- 第二层：getOpenToken调用的方法 --------

    /**
     * 解析RPA令牌响应
     * @param response HTTP响应
     * @return RPA访问令牌，失败时返回null
     */
    private String parseTokenResponse(HttpResponse response) {
        if (!response.isOk()) {
            logger.error("获取RPA访问令牌HTTP请求失败，状态码: {}", response.getStatus());
            return null;
        }

        String responseBody = response.body();
        logger.debug("RPA获取token响应: {}", responseBody);

        try {
            // 解析响应
            cn.hutool.json.JSONObject jsonResponse = JSONUtil.parseObj(responseBody);
            if (RPA_SUCCESS_CODE.equals(jsonResponse.getStr("code"))) {
                // 获取data对象
                cn.hutool.json.JSONObject dataObj = jsonResponse.getJSONObject("data");
                if (dataObj != null) {
                    // 从data对象中获取openToken
                    String openToken = dataObj.getStr("openToken");
                    if (StringUtils.isNotBlank(openToken)) {
                        logger.info("成功获取RPA访问令牌");
                        return openToken;
                    } else {
                        logger.error("RPA响应中openToken为空");
                    }
                } else {
                    logger.error("RPA响应中data对象为空");
                }
            } else {
                logger.error("获取RPA访问令牌失败，错误码: {}, 错误信息: {}",
                        jsonResponse.getStr("code"), jsonResponse.getStr("msg"));
            }
        } catch (Exception e) {
            logger.error("解析RPA令牌响应时发生异常", e);
        }

        return null;
    }

    // -------- 第二层：processFileUpload调用的方法 --------

    /**
     * 上传文件到RPA系统
     * @param fileResponse 文件响应对象
     * @param fileId 文件ID
     * @param fileName 文件名
     * @return RPA文件ID，失败时返回null
     */
    private String uploadToRPA(Response fileResponse, Long fileId, String fileName) {
        // 参数校验
        if (fileResponse == null || fileResponse.body() == null) {
            logger.error("文件响应为空，文件ID: {}", fileId);
            return null;
        }
        
        if (StringUtils.isBlank(fileName)) {
            logger.error("文件名为空，文件ID: {}", fileId);
            return null;
        }

        InputStream inputStream = null;
        File tempFile = null;
        
        try {
            // 获取文件输入流
            inputStream = fileResponse.body().asInputStream();
            if (inputStream == null) {
                logger.error("文件流为空，文件ID: {}", fileId);
                return null;
            }

            // 获取RPA访问令牌
            String token = getOpenToken();
            if (StringUtils.isBlank(token)) {
                logger.error("获取RPA访问令牌失败，文件ID: {}", fileId);
                return null;
            }

            // 创建临时文件
            tempFile = createTempFile(inputStream, fileName, fileId);
            if (tempFile == null) {
                return null;
            }

            // 上传文件到RPA
            return uploadFileToRPA(tempFile, token, fileId);

        } catch (IOException e) {
            logger.error("处理文件流时发生IO异常，文件ID: {}", fileId, e);
            return null;
        } catch (Exception e) {
            logger.error("上传文件到RPA时发生异常，文件ID: {}", fileId, e);
            return null;
        } finally {
            // 清理资源
            closeQuietly(inputStream);
            deleteQuietly(tempFile);
        }
    }

    // -------- 第二层：buildComparisonResult调用的方法 --------

    /**
     * 构建文件对比URL
     * @param openToken RPA访问令牌
     * @param firstFileResult 第一个文件结果
     * @param secondFileResult 第二个文件结果
     * @return 对比URL
     */
    private String buildComparisonUrl(String openToken, FileUploadResult firstFileResult, FileUploadResult secondFileResult) {
        return String.format("http://10.136.208.23:16106/app-aiservice/contract_comparision?" +
                        "openToken=%s" +
                        "&ignoreWatermark" +
                        "&ignoreSeal" +
                        "&basicFileId=%s" +
                        "&basicFileName=%s" +
                        "&targetFileId=%s" +
                        "&targetFileName=%s",
                openToken,
                firstFileResult.getRpaFileId(),
                firstFileResult.getFileName(),
                secondFileResult.getRpaFileId(),
                secondFileResult.getFileName());
    }

    // -------- 第三层：uploadToRPA调用的方法 --------

    /**
     * 创建临时文件
     * @param inputStream 输入流
     * @param fileName 文件名
     * @param fileId 文件ID
     * @return 临时文件，失败时返回null
     */
    private File createTempFile(InputStream inputStream, String fileName, Long fileId) {
        File tempFile = null;
        try {
            // 获取文件扩展名
            String extension = getFileExtension(fileName);
            tempFile = File.createTempFile(fileName, extension);
            
            // 将InputStream的内容写入临时文件
            try (FileOutputStream outputStream = new FileOutputStream(tempFile)) {
                byte[] buffer = new byte[8192]; // 增加缓冲区大小
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
                outputStream.flush();
            }
            
            logger.debug("临时文件创建成功，文件ID: {}, 临时文件路径: {}", fileId, tempFile.getAbsolutePath());
            return tempFile;
            
        } catch (IOException e) {
            logger.error("创建临时文件失败，文件ID: {}", fileId, e);
            deleteQuietly(tempFile);
            return null;
        }
    }

    /**
     * 上传文件到RPA系统
     * @param tempFile 临时文件
     * @param token RPA访问令牌
     * @param fileId 文件ID
     * @return RPA文件ID，失败时返回null
     */
    private String uploadFileToRPA(File tempFile, String token, Long fileId) {
        try {
            HttpResponse response = HttpRequest.post(RPA_BASE_URL + RPA_UPLOAD_URL)
                    .header("Open-Token", token)
                    .form("file", tempFile)
                    .timeout(RPA_UPLOAD_TIMEOUT)
                    .execute();

            return parseUploadResponse(response, fileId);
            
        } catch (Exception e) {
            logger.error("上传文件到RPA时发生异常，文件ID: {}", fileId, e);
            return null;
        }
    }

    // -------- 第四层：createTempFile调用的方法 --------

    /**
     * 获取文件扩展名
     * @param fileName 文件名
     * @return 文件扩展名（包含点号）
     */
    private String getFileExtension(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return ".tmp";
        }
        
        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex > 0 && lastDotIndex < fileName.length() - 1) {
            return fileName.substring(lastDotIndex);
        }
        
        return ".docx"; // 默认扩展名
    }

    // -------- 第四层：uploadFileToRPA调用的方法 --------

    /**
     * 解析RPA上传响应
     * @param response HTTP响应
     * @param fileId 文件ID
     * @return RPA文件ID，失败时返回null
     */
    private String parseUploadResponse(HttpResponse response, Long fileId) {
        if (!response.isOk()) {
            logger.error("文件上传RPA HTTP请求失败，状态码: {}, 文件ID: {}", response.getStatus(), fileId);
            return null;
        }

        String responseBody = response.body();
        logger.debug("RPA文件上传响应: {}", responseBody);

        try {
            // 解析响应
            cn.hutool.json.JSONObject jsonResponse = JSONUtil.parseObj(responseBody);
            if (RPA_SUCCESS_CODE.equals(jsonResponse.getStr("code"))) {
                String rpaFileId = jsonResponse.getStr("data");
                if (StringUtils.isNotBlank(rpaFileId)) {
                    logger.info("文件上传RPA成功，文件ID: {}, RPA文件ID: {}", fileId, rpaFileId);
                    return rpaFileId;
                } else {
                    logger.error("RPA响应中文件ID为空，文件ID: {}", fileId);
                }
            } else {
                logger.error("文件上传RPA失败，错误码: {}, 错误信息: {}, 文件ID: {}",
                        jsonResponse.getStr("code"), jsonResponse.getStr("message"), fileId);
            }
        } catch (Exception e) {
            logger.error("解析RPA上传响应时发生异常，文件ID: {}", fileId, e);
        }

        return null;
    }

    // -------- 工具方法（被多个方法调用） --------

    /**
     * 安全关闭输入流
     * @param inputStream 输入流
     */
    private void closeQuietly(InputStream inputStream) {
        if (inputStream != null) {
            try {
                inputStream.close();
            } catch (IOException e) {
                logger.warn("关闭文件流时发生异常", e);
            }
        }
    }

    /**
     * 安全删除文件
     * @param file 文件
     */
    private void deleteQuietly(File file) {
        if (file != null && file.exists()) {
            try {
                if (file.delete()) {
                    logger.debug("临时文件删除成功: {}", file.getAbsolutePath());
                } else {
                    logger.warn("临时文件删除失败: {}", file.getAbsolutePath());
                }
            } catch (Exception e) {
                logger.warn("删除临时文件时发生异常: {}", file.getAbsolutePath(), e);
            }
        }
    }
}





