package com.ejianc.util;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Base64;

/**
 * 文件工具类
 * @author: 系统生成
 * @version: 1
 */
public class FileUtils {

    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    /**
     * 验证base64字符串是否有效
     *
     * @param base64String base64字符串
     * @return 是否有效
     */
    public static boolean isValidBase64(String base64String) {
        if (StringUtils.isBlank(base64String)) {
            return false;
        }

        try {
            Base64.getDecoder().decode(base64String);
            return true;
        } catch (IllegalArgumentException e) {
            logger.debug("无效的base64字符串: {}", e.getMessage());
            return false;
        }
    }

    /**
     * 构建base64数据URI
     *
     * @param fileName     文件名
     * @param base64String base64编码的文件内容
     * @return 完整的数据URI
     */
    public static String buildDataUri(String fileName, String base64String) {
        String extension = getFileExtension(fileName);
        return "data:application/" + extension + ";base64," + base64String;
    }

    /**
     * 获取文件扩展名
     *
     * @param fileName 文件名
     * @return 文件扩展名
     */
    public static String getFileExtension(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "tmp";
        }

        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex > 0 && lastDotIndex < fileName.length() - 1) {
            return fileName.substring(lastDotIndex + 1);
        }

        return "docx";
    }

    /**
     * 根据文件名获取文件类型
     * @param fileName 文件名
     * @return 文件类型
     */
    public static String getFileTypeByFileName(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "w"; // 默认word类型
        }

        String extension = getFileExtension(fileName).toLowerCase();
        switch (extension) {
            case "pdf":
                return "f"; // pdf
            case "doc":
            case "docx":
                return "w"; // word
            case "xls":
            case "xlsx":
                return "s"; // excel
            case "ppt":
            case "pptx":
                return "p"; // powerpoint
            default:
                return "w"; // 默认word类型
        }
    }

    /**
     * 根据文件名获取Content-Type
     * @param fileName 文件名
     * @return Content-Type
     */
    public static String getContentTypeByFileName(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "application/octet-stream";
        }

        String extension = getFileExtension(fileName).toLowerCase();
        switch (extension) {
            case "pdf":
                return "application/pdf";
            case "doc":
                return "application/msword";
            case "docx":
                return "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
            case "xls":
                return "application/vnd.ms-excel";
            case "xlsx":
                return "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
            case "txt":
                return "text/plain";
            case "jpg":
            case "jpeg":
                return "image/jpeg";
            case "png":
                return "image/png";
            case "gif":
                return "image/gif";
            default:
                return "application/octet-stream";
        }
    }
} 