package com.ejianc.service.impl;

import com.ejianc.entity.TemplateEntity;
import com.ejianc.entity.UpLoad;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentRequestVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.IUploadFileService;
import com.ejianc.service.ITemplateService;
import feign.Response;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import com.ejianc.util.FileUtils;
import java.util.ArrayList;
import java.util.List;

/**
 * 文件上传服务实现类
 * @author: 系统生成
 * @version: 1
 */
@Service
public class UploadFileServiceImpl implements IUploadFileService {

    private static final Logger logger = LoggerFactory.getLogger(UploadFileServiceImpl.class);

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private ITemplateService templateService;

  

    @Override
    public CommonResponse<List<AttachmentVO>> uploadFile(UpLoad upload) {
        logger.info("开始处理文件上传，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId());

        try {
            // 构建数据URI
            String dataUri = FileUtils.buildDataUri(upload.getFileName(), upload.getBase64String());

            // 删除已有文件
            deleteExistingFiles(upload);

            // 上传新文件
            return uploadNewFile(upload, dataUri);

        } catch (Exception e) {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId(), e);
            return CommonResponse.error("文件上传失败: " + e.getMessage());
        }
    }

    @Override
    public void uploadTemplate(UpLoad upload) throws IOException {
        logger.info("开始处理模板文件上传，文件ID: {}, 模板名称: {}, 文件名: {}", 
                   upload.getFileId(), upload.getTemplateName(), upload.getFileName());
        
        try {
            // 验证文件名是否已设置
            if (StringUtils.isBlank(upload.getFileName())) {
                throw new IOException("文件名不能为空");
            }

            // 下载并上传文件
            uploadFileFromAttachment(upload.getFileId(), upload.getFileName());

            // 保存模板信息
            TemplateEntity templateEntity = createTemplateEntity(upload);
            templateService.save(templateEntity);
            logger.info("模板信息保存成功: {}", templateEntity.getTemplateName());
            
        } catch (Exception e) {
            logger.error("模板文件上传失败，文件ID: {}", upload.getFileId(), e);
            throw new IOException("模板文件上传失败: " + e.getMessage(), e);
        }
    }

    @Override
    public void deleteTemplate(UpLoad upload) throws IOException {
        logger.info("开始处理模板删除，文件ID: {}, 模板名称: {}",
                upload.getFileId(), upload.getFileName());

        try {
            // 调用另一个服务器的删除文件接口
            callDeleteFileInterface(upload);
            
            // 删除本地模板信息
            String fileId = upload.getFileId();
            templateService.removeById(fileId);
            
            logger.info("模板删除成功，文件ID: {}", upload.getFileId());

        } catch (Exception e) {
            logger.error("模板删除失败，文件ID: {}", upload.getFileId(), e);
            throw new IOException("模板删除失败: " + e.getMessage(), e);
        }
    }

    @Override
    public void uploadFileFromAttachment(String fileId,String fileName) throws IOException {
        // 下载文件
        Response response = attachmentApi.downloadFileById(Long.valueOf(fileId));
        if (response == null || response.body() == null) {
            throw new IOException("无法下载文件，文件ID: " + fileId);
        }

        // 转换为MultipartFile并上传
        try (InputStream inputStream = response.body().asInputStream()) {
            MultipartFile multipartFile = createMultipartFile(inputStream, fileName);
            callUploadFileInterface(multipartFile);
            logger.info("文件上传成功，文件名: {}", fileName);
        }
    }



    @Override
    public void callUploadFileInterface(MultipartFile multipartFile) {
        try {
            // 设置请求头
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.MULTIPART_FORM_DATA);

            // 创建文件资源
            ByteArrayResource fileResource = new ByteArrayResource(multipartFile.getBytes()) {
                @Override
                public String getFilename() {
                    return multipartFile.getOriginalFilename();
                }
            };

            // 构建请求体
            MultiValueMap<String, Object> body = new LinkedMultiValueMap<>();
            body.add("file", fileResource);

            // 创建HTTP实体
            HttpEntity<MultiValueMap<String, Object>> requestEntity = new HttpEntity<>(body, headers);

            // 调用接口 - 这里需要根据实际情况修改URL
            String uploadUrl = "http://10.136.208.154:8081/upload/uploadFile";
            
            ResponseEntity<String> response = restTemplate.exchange(
                uploadUrl,
                HttpMethod.POST,
                requestEntity,
                String.class
            );

            // 处理响应
            if (response.getStatusCode() == HttpStatus.OK) {
                logger.info("文件上传成功，响应: {}", response.getBody());
            } else {
                logger.error("文件上传失败，状态码: {}, 响应: {}", response.getStatusCode(), response.getBody());
            }

        } catch (Exception e) {
            logger.error("调用上传接口时发生异常", e);
        }
    }

    @Override
    public void callDeleteFileInterface(UpLoad upload) {
        try {
            // 设置请求头
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            // 创建HTTP实体
            HttpEntity<UpLoad> requestEntity = new HttpEntity<>(upload, headers);

            // 调用另一个服务器的删除文件接口
            String deleteUrl = "http://10.136.208.154:8081/upload/deleteFile";

            //String deleteUrl = "http://localhost:8081/upload/deleteFile";


            ResponseEntity<String> response = restTemplate.exchange(
                deleteUrl,
                HttpMethod.POST,
                requestEntity,
                String.class
            );

            // 处理响应
            if (response.getStatusCode() == HttpStatus.OK) {
                logger.info("远程文件删除成功，文件ID: {}, 响应: {}", upload.getFileId(), response.getBody());
            } else {
                logger.error("远程文件删除失败，文件ID: {}, 状态码: {}, 响应: {}", 
                           upload.getFileId(), response.getStatusCode(), response.getBody());
                throw new RuntimeException("远程文件删除失败，状态码: " + response.getStatusCode());
            }

        } catch (Exception e) {
            logger.error("调用删除文件接口时发生异常，文件ID: {}", upload.getFileId(), e);
            throw new RuntimeException("调用删除文件接口失败: " + e.getMessage(), e);
        }
    }



    /**
     * 删除已有文件
     */
    private void deleteExistingFiles(UpLoad upload) {
        try {
            Long sourceId = Long.valueOf(upload.getSourceId());
            attachmentApi.deleteFileByParam(sourceId, upload.getBillType(), upload.getSourceType());
            logger.debug("已删除已有文件，来源ID: {}", upload.getSourceId());
        } catch (NumberFormatException e) {
            logger.warn("来源ID格式错误，跳过删除操作: {}", upload.getSourceId());
        } catch (Exception e) {
            logger.warn("删除已有文件失败，继续上传: {}", e.getMessage());
        }
    }

    /**
     * 上传新文件
     */
    private CommonResponse<List<AttachmentVO>> uploadNewFile(UpLoad upload, String dataUri) {
        List<String> fileList = new ArrayList<>();
        fileList.add(dataUri);

        AttachmentRequestVO request = new AttachmentRequestVO();
        request.setBillType(upload.getBillType());
        request.setSourceId(upload.getSourceId());
        request.setSourceType(upload.getSourceType());
        request.setFileList(fileList);

        CommonResponse<List<AttachmentVO>> response = attachmentApi.uploadForBase64(request);

        if (response.isSuccess()) {
            logger.info("文件上传成功，单据类型: {}, 来源ID: {}, 文件数量: {}",
                    upload.getBillType(), upload.getSourceId(),
                    response.getData() != null ? response.getData().size() : 0);
        } else {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}, 错误信息: {}",
                    upload.getBillType(), upload.getSourceId(), response.getData());
        }

        return response;
    }



    /**
     * 将InputStream转换为MultipartFile
     */
    private MultipartFile createMultipartFile(InputStream inputStream, String fileName) throws IOException {
        // 读取输入流的所有字节 - Java 8 兼容版本
        byte[] fileBytes = readAllBytes(inputStream);
        
        return new MultipartFile() {
            @Override
            public String getName() {
                return "file";
            }

            @Override
            public String getOriginalFilename() {
                return fileName;
            }

            @Override
            public String getContentType() {
                return FileUtils.getContentTypeByFileName(fileName);
            }

            @Override
            public boolean isEmpty() {
                return fileBytes.length == 0;
            }

            @Override
            public long getSize() {
                return fileBytes.length;
            }

            @Override
            public byte[] getBytes() throws IOException {
                return fileBytes;
            }

            @Override
            public InputStream getInputStream() throws IOException {
                return new java.io.ByteArrayInputStream(fileBytes);
            }

            @Override
            public void transferTo(java.io.File dest) throws IOException, IllegalStateException {
                Files.write(dest.toPath(), fileBytes);
            }
        };
    }

    /**
     * 创建TemplateEntity对象
     */
    private TemplateEntity createTemplateEntity(UpLoad upload) {
        TemplateEntity templateEntity = new TemplateEntity();
        
        // 设置基本信息
        templateEntity.setTemplateName(upload.getTemplateName());
        templateEntity.setCategoryName(upload.getCategoryName());
        templateEntity.setBelongOrgName(upload.getBelongOrgName());
        templateEntity.setFileName(upload.getFileName());

        // 设置文件ID
        if (StringUtils.isNotBlank(upload.getFileId())) {
            templateEntity.setFileId(Long.valueOf(upload.getFileId()));
        }
        
        // 设置默认值
        templateEntity.setEnableStatus(1); // 默认启用
        templateEntity.setSequence(1); // 默认排序号
        templateEntity.setFileType(upload.getFileType());

        logger.info("TemplateEntity创建完成: 模板名称={}, 分类名称={}, 所属组织={}, 文件名={}", 
                   templateEntity.getTemplateName(), templateEntity.getCategoryName(), 
                   templateEntity.getBelongOrgName(), templateEntity.getFileName());
        
        return templateEntity;
    }

    /**
     * Java 8 兼容的读取所有字节方法
     */
    private byte[] readAllBytes(InputStream inputStream) throws IOException {
        java.io.ByteArrayOutputStream buffer = new java.io.ByteArrayOutputStream();
        int nRead;
        byte[] data = new byte[1024];
        while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
            buffer.write(data, 0, nRead);
        }
        buffer.flush();
        return buffer.toByteArray();
    }
} 