package com.ejianc.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.entity.TemplateEntity;
import com.ejianc.entity.UpLoad;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.IUploadFileService;
import com.ejianc.service.ITemplateService;
import com.ejianc.util.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.List;

/**
 * 文件上传控制器
 * @author: 系统生成
 * @version: 1
 */
@RestController
@RequestMapping(value = "/upload")
public class UploadFileController {

    private static final Logger logger = LoggerFactory.getLogger(UploadFileController.class);

    @Autowired
    private IUploadFileService uploadFileService;

    @Autowired
    private ITemplateService templateService;

    @Autowired
    private IAttachmentApi attachmentApi;

    /**
     * PageOffice调用上传
     * @param upload 上传参数
     * @return 上传结果
     */
    @PostMapping(value = "/uploadFile")
    public CommonResponse<List<AttachmentVO>> uploadFile(@RequestBody UpLoad upload) {
        logger.info("接收到文件上传请求，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId());
        
        // 参数校验
        CommonResponse<List<AttachmentVO>> validationResult = validateUploadFileParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        return uploadFileService.uploadFile(upload);
    }

    /**
     * 上传模板文件
     * @param upload 上传参数
     * @throws IOException IO异常
     */
    @PostMapping(value = "/uploadTemplate")
    @ResponseBody
    public CommonResponse<Void> uploadFileTemplate(@RequestBody UpLoad upload) throws IOException {
        logger.info("接收到模板文件上传请求，文件ID: {}, 模板名称: {}", 
                   upload.getFileId(), upload.getTemplateName());
        
        // 参数校验
        CommonResponse<Void> validationResult = validateTemplateParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        // 获取文件名并进行去重检查
        String fileName = getFileName(upload.getFileId());
        CommonResponse<Void> duplicateCheckResult = checkFileNameDuplicate(fileName);
        if (!duplicateCheckResult.isSuccess()) {
            return duplicateCheckResult;
        }
        
        // 设置文件名到upload对象
        upload.setFileName(fileName);
        
        // 调用Service层方法，传递文件名
        uploadFileService.uploadTemplate(upload);
        return CommonResponse.success("模板文件上传成功");
    }

    /**
     * 删除模板文件
     * @param upload 删除参数
     * @throws IOException IO异常
     */
    @PostMapping(value = "/deleteTemplate")
    @ResponseBody
    public CommonResponse<Void> deleteTemplate(@RequestBody UpLoad upload) throws IOException {
        logger.info("接收到模板删除请求，文件ID: {}, 模板名称: {}",
                upload.getFileId(), upload.getFileName());
        
        // 参数校验
        CommonResponse<Void> validationResult = validateDeleteTemplateParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        uploadFileService.deleteTemplate(upload);
        return CommonResponse.success("模板文件删除成功");
    }

    /**
     * 校验文件上传参数
     * @param upload 上传参数
     * @return 校验结果
     */
    private CommonResponse<List<AttachmentVO>> validateUploadFileParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.error("上传参数为空");
            return CommonResponse.error("上传参数不能为空");
        }

        // 校验base64字符串
        if (StringUtils.isBlank(upload.getBase64String())) {
            logger.error("base64字符串为空");
            return CommonResponse.error("base64字符串不能为空");
        }

        // 校验必要参数
        if (StringUtils.isBlank(upload.getBillType()) ||
                StringUtils.isBlank(upload.getSourceId()) ||
                StringUtils.isBlank(upload.getSourceType())) {
            logger.error("必要参数缺失: billType={}, sourceId={}, sourceType={}",
                    upload.getBillType(), upload.getSourceId(), upload.getSourceType());
            return CommonResponse.error("必要参数缺失: 单据类型、来源ID、来源类型不能为空");
        }

        // 校验base64字符串有效性
        if (!FileUtils.isValidBase64(upload.getBase64String())) {
            logger.error("无效的base64字符串");
            return CommonResponse.error("无效的base64字符串格式");
        }

        logger.info("文件上传参数校验通过");
        return CommonResponse.success("参数校验通过", null);
    }

    /**
     * 校验模板上传参数
     * @param upload 上传参数
     * @return 校验结果
     */
    private CommonResponse<Void> validateTemplateParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.warn("模板上传参数为空");
            return CommonResponse.error("模板上传参数不能为空");
        }

        // 校验文件ID
        if (StringUtils.isBlank(upload.getFileId())) {
            logger.warn("模板文件ID为空");
            return CommonResponse.error("模板文件ID不能为空");
        }

        // 校验文件ID格式
        try {
            Long.valueOf(upload.getFileId());
        } catch (NumberFormatException e) {
            logger.warn("无效的模板文件ID格式: {}", upload.getFileId());
            return CommonResponse.error("无效的模板文件ID格式");
        }

        // 校验模板名称
        if (StringUtils.isBlank(upload.getTemplateName())) {
            logger.warn("模板名称为空");
            return CommonResponse.error("模板名称不能为空");
        }

        // 校验分类名称
        if (StringUtils.isBlank(upload.getCategoryName())) {
            logger.warn("分类名称为空");
            return CommonResponse.error("分类名称不能为空");
        }

        // 校验所属组织名称 - 修复：belongOrgName现在是HashMap类型
        if (upload.getBelongOrgName() == null || upload.getBelongOrgName().isEmpty()) {
            logger.warn("所属组织信息为空");
            return CommonResponse.error("所属组织信息不能为空");
        }

        logger.info("模板上传参数校验通过");
        return CommonResponse.success("参数校验通过");
    }

    /**
     * 校验模板删除参数
     * @param upload 删除参数
     * @return 校验结果
     */
    private CommonResponse<Void> validateDeleteTemplateParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.warn("模板删除参数为空");
            return CommonResponse.error("模板删除参数不能为空");
        }

        // 校验文件ID
        if (StringUtils.isBlank(upload.getFileId())) {
            logger.warn("删除模板文件ID为空");
            return CommonResponse.error("删除模板文件ID不能为空");
        }

        // 校验文件ID格式
        try {
            Long.valueOf(upload.getFileId());
        } catch (NumberFormatException e) {
            logger.warn("无效的删除模板文件ID格式: {}", upload.getFileId());
            return CommonResponse.error("无效的删除模板文件ID格式");
        }

        // 校验模板名称
        if (StringUtils.isBlank(upload.getFileName())) {
            logger.warn("删除模板名称为空");
            return CommonResponse.error("删除模板名称不能为空");
        }

        logger.info("模板删除参数校验通过");
        return CommonResponse.success("参数校验通过");
    }

    /**
     * 获取文件名
     * @param fileId 文件ID
     * @return 文件名
     * @throws IOException IO异常
     */
    private String getFileName(String fileId) throws IOException {
        // 获取文件信息
        CommonResponse<AttachmentVO> attachmentResponse = attachmentApi.queryDetail(fileId);
        if (!attachmentResponse.isSuccess() || attachmentResponse.getData() == null) {
            throw new IOException("无法获取文件信息，文件ID: " + fileId);
        }

        String fileName = attachmentResponse.getData().getFileName();
        if (StringUtils.isBlank(fileName)) {
            throw new IOException("文件名为空，文件ID: " + fileId);
        }
        return fileName;
    }

    /**
     * 检查文件名是否重复
     * @param fileName 文件名
     * @return 检查结果
     */
    private CommonResponse<Void> checkFileNameDuplicate(String fileName) {
        try {
            // 按照fileName查询判断去重
            LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(TemplateEntity::getFileName, fileName);
            List<TemplateEntity> templateEntities = templateService.list(queryWrapper);

            // 检查是否存在重复的文件名
            if (templateEntities != null && !templateEntities.isEmpty()) {
                logger.warn("发现重复的文件名: {}, 已存在的模板数量: {}", fileName, templateEntities.size());
                
                // 构建重复信息
                StringBuilder duplicateInfo = new StringBuilder();
                duplicateInfo.append("文件名 '").append(fileName).append("' 已存在，重复的模板信息：\n");
                
                return CommonResponse.error("文件重复: " + duplicateInfo.toString());
            }

            return CommonResponse.success("文件名检查通过");
            
        } catch (Exception e) {
            logger.error("检查文件名重复时发生异常，文件名: {}", fileName, e);
            return CommonResponse.error("检查文件名重复失败: " + e.getMessage());
        }
    }
}

