package com.ejianc.poc.gyy.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.gyypoc.vo.PurchaseProcessVO;
import com.ejianc.foundation.bpm.api.IBpmApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.poc.gyy.bean.PurchaseBillEntity;
import com.ejianc.poc.gyy.bean.PurchaseProcessEntity;
import com.ejianc.poc.gyy.bean.SoftReceiveCheckEntity;
import com.ejianc.poc.gyy.service.IPurchaseBillService;
import com.ejianc.poc.gyy.service.IPurchaseProcessService;
import com.ejianc.poc.gyy.service.ISoftReceiveCheckService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 采购流程审批回调类
 *
 * @author CJ
 * @Description: 采购流程审批回调类
 * @date 2021/3/25 14:55
 */
@Service(value = "purchaseProcess")
public class PurchaseProcessBpmCallbackServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private final static String APPROVE_SIGN_MEETING_INIT = "init";
    private final static String APPROVE_SIGN_MEETING_APPROVE = "meetingApprove";
    private final static String APPROVE_SIGN_MEETING_REC = "rec";

    private final static String PURCHASE_BILL_BILL_TYPE_CODE = "BT210323000000002";
    //申请报告审批页面地址
    private final static String PC_URL = "ejc-gyypoc-frontend/#/purchaseBill/approveCard?id={id}";
    private final static String MA_URL = "ejc-gyypoc-mobile/#/applyApprove?id={id}";

    private final static String REC_PC_URL = "ejc-gyypoc-frontend/#/receive/approve?id={id}";
    private final static String REC_MA_URL = "ejc-gyypoc-mobile/#/receive/approve?id={id}";

    @Autowired
    private IPurchaseProcessService purchaseProcessService;


    @Autowired
    private IPurchaseBillService purchaseBillService;

    @Autowired
    private IBpmApi bpmApi;

    @Autowired
    private ISoftReceiveCheckService softReceiveCheckService;

    /**
     * 审批节点审批中时节点审批前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeInApprovalBack(Long billId, Integer state, String billTypeCode, String sign) {
        logger.info("采购流程审批节点审批中时节点审批前回调 billId-{}, sing-{}", billId, sign);
        Long userId =  InvocationInfoProxy.getUserid();
        String userCode = InvocationInfoProxy.getUsercode();
        PurchaseProcessEntity processEntity = purchaseProcessService.selectById(billId);

        switch (sign) {
            case APPROVE_SIGN_MEETING_APPROVE: {
                QueryWrapper<PurchaseBillEntity> billQuery = new QueryWrapper<>();
                billQuery.eq("purchase_process_id", billId);
                billQuery.notIn("bill_state", new Integer[]{BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode()});

                List<PurchaseBillEntity> unFinishList = purchaseBillService.list(billQuery);
                if(CollectionUtils.isNotEmpty(unFinishList)) {
                    logger.error("采购流程[billId-{}]存在未审批完成的申请报告，无法执审批操作！", billId);
                    return CommonResponse.error("存在未审批完成的申请报告，无法执行该操作！");
                }
                break;
            }
            case APPROVE_SIGN_MEETING_INIT: {
                //查询申请报告并提交
                QueryWrapper<PurchaseBillEntity> query = new QueryWrapper<>();
                query.eq("purchase_process_id", billId);
                query.eq("create_user_id", userId);
                PurchaseBillEntity purchaseBill = purchaseBillService.getOne(query);
                if(null == purchaseBill) {
                    logger.error("采购流程审提交填报失败，未找到【采购流程：billId-{}, 申请用户:userId-{}】对应的申请报告！", billId, userId);
                    return CommonResponse.error("采购流程审提交填报失败，未找到匹配的申请报告！");
                }
                String msg = submitBill(userId,PURCHASE_BILL_BILL_TYPE_CODE, purchaseBill.getId(),
                        processEntity.getName() + "-" + purchaseBill.getApplyOrgName(),PC_URL,MA_URL);

                if(StringUtils.isNotBlank(msg)) {
                    logger.error("申请报告【id: {}】提交流程失败，原因：{}", purchaseBill.getId(), msg);
                    return CommonResponse.error("申请报告填报提交失败！");
                }
                break;
            }
            case APPROVE_SIGN_MEETING_REC: {
                //查询申请报告并提交
                QueryWrapper<SoftReceiveCheckEntity> recQuery = new QueryWrapper<>();
                recQuery.eq("process_id", billId);
                recQuery.eq("create_user_code", userCode);
                SoftReceiveCheckEntity rec = softReceiveCheckService.getOne(recQuery);
                if(null == rec) {
                    logger.error("软件验收提交填报失败，未找到【采购流程：billId-{}, 申请用户:userCode-{}】对应的申请报告！", billId, userCode);
                    return CommonResponse.error("采购流程审提交填报失败，未找到匹配的申请报告！");
                }

                String msg = submitBill(userId,PURCHASE_BILL_BILL_TYPE_CODE, rec.getId(),
                        processEntity.getName() + "-" + rec.getName() + " 验收", REC_PC_URL, REC_MA_URL);

                if(StringUtils.isNotBlank(msg)) {
                    logger.error("软件验收【id: {}】提交流程失败，原因：{}", rec.getId(), msg);
                    return CommonResponse.error("软件验收填报提交失败！");
                }
                break;
            }
        }
        return CommonResponse.success("未发现未完成的申请报告，可以执行该操作！");
    }

    private String submitBill(Long userId, String billTypeCode, Long billId, String processName, String pcUrl, String maUrl) {
        Map<String, Object> submitParam = new HashMap<>(20);
        submitParam.put("userId", userId);
        submitParam.put("billtypeId", billTypeCode);
        submitParam.put("assignAble", false);
        submitParam.put("businessKey", billId);
        submitParam.put("messageType", new String[]{"sys","weixin"});
        submitParam.put("billOrgId", InvocationInfoProxy.getOrgId());
        submitParam.put("procInstName", processName);
        submitParam.put("formurl", pcUrl);
        submitParam.put("maurl", maUrl);
        logger.info("单据【id: {}】提交流程参数：{}", billId, JSONObject.toJSONString(submitParam));
        CommonResponse<String> submitResp = bpmApi.doSubmit(submitParam);
        logger.info("单据【id: {}】提交流程结果：{}", billId, JSONObject.toJSONString(submitResp));

        if(!submitResp.isSuccess()) {
            logger.error("单据【id: {}】提交流程失败，原因：{}", billId, submitResp.getMsg());
            return submitResp.getMsg();
        }

        return null;
    }

    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        logger.info("采购流程【billId-{}】审批完成回调，将该流程的流流程状态设置为已结束", billId);
        PurchaseProcessEntity processEntity = purchaseProcessService.selectById(billId);

        //将采购流程的流程状态设置为已结束
        processEntity.setProcessState(PurchaseProcessVO.PROCESS_STATE_END);
        purchaseProcessService.saveOrUpdate(processEntity, false);
        return CommonResponse.success("采购流程审批完成回调处理成功！");
    }

    /**
     * 审批节点审批中时节点审批后回调
     *
     * @param billId 单据id
     * @param state 单据状态
     * @param billTypeCode 单据类型code
     * @param taskId 该流程id
     * @param isEnd 该节点是否结束
     * @param sign 节点标识符
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> afterInApprovalBack(Long billId, Integer state, String billTypeCode, String taskId, Boolean isEnd, String sign) {
        logger.info("采购流程【billId-{},sign-{},isEnd-{}】审批完成回调，将该流程的流流程状态设置为汇总审批", billId, sign, isEnd);
        PurchaseProcessEntity processEntity = purchaseProcessService.selectById(billId);

        if(isEnd) {
            switch (sign) {
                case APPROVE_SIGN_MEETING_INIT:
                    //将采购流程的流程状态设置为汇总审批
                    processEntity.setProcessState(PurchaseProcessVO.PROCESS_STATE_APPROVE);
                    logger.info("采购流程【billId-{}】节点审批完成回调，将该流程的流流程状态设置为汇总审批", billId);
                    break;
                case APPROVE_SIGN_MEETING_APPROVE:
                    //将采购流程的流程状态设置为采购
                    processEntity.setProcessState(PurchaseProcessVO.PROCESS_STATE_PURCHASE);
                    logger.info("采购流程【billId-{}】节点审批完成回调，将该流程的流流程状态设置为采购", billId);
                    break;
            }

            purchaseProcessService.saveOrUpdate(processEntity, false);
        } else if(APPROVE_SIGN_MEETING_REC.equals(sign)) {
            if(!PurchaseProcessVO.PROCESS_STATE_REC.equals(processEntity.getProcessState())) {
                //将采购流程的流程状态设置为验收
                processEntity.setProcessState(PurchaseProcessVO.PROCESS_STATE_REC);
                logger.info("采购流程【billId-{}】节点审批完成回调，将该流程的流流程状态设置为验收", billId);

                purchaseProcessService.saveOrUpdate(processEntity, false);
            }
        }
        return CommonResponse.success("采购流程节点审批完成回调处理成功！");
    }
}
