package com.ejianc.foundation.usercenter.pm.encoder;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.codec.digest.MessageDigestAlgorithms;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * 1. 将原始密码 SHA1 加密后转成 16 进制字符串 2. 用生成的 16 进制字符串与用户名和用户盐进行加密 1. 将用户名 SHA1 加密后转成
 * 16 进制字符串 2. 用用户盐的 16 进制字符串作为盐，密码密文串拼接用户名串作为待加密密文，使用 SHA1 加密 3. 获得的加密串转成 16
 * 进制字符串作为用户密码
 *
 */
public class UsernameSaltPasswordEncoder implements PasswordEncoder {

	private String username;

	private String salt;

	@SuppressWarnings("unused")
	private UsernameSaltPasswordEncoder() {}

	public UsernameSaltPasswordEncoder(String username, String salt) {
		this.username = username;
		this.salt = salt;
	}

	@Override
	public String encode(String rawPassword) {
		String encodePassword = encodeBySHA1AndHex(rawPassword);
		String encodedUsername = encodeBySHA1AndHex(this.username);

		return encodeBySHA1WithSaltAndHex((encodePassword + encodedUsername), decodeHex(this.salt));
	}

	/**
	 * 字符串转为 16 进制格式
	 *
	 * @param input
	 *            待转换的字符串
	 * @return 转换后 16 进制的字符串
	 */
	private byte[] decodeHex(String input) {
		if (input == null) {
			return null;
		}

		try {
			return Hex.decodeHex(input.toCharArray());
		} catch (DecoderException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 字符串先用 SHA1 转换，然后将 SHA1 转换后的字符串转换为 16 进制格式（小写）的字符串
	 *
	 * @param input
	 *            待转换的字符串
	 * @return 转换后的字符串
	 */
	private String encodeBySHA1AndHex(String input) {
		byte[] bytes = DigestUtils.sha1(input.getBytes());

		return Hex.encodeHexString(bytes);
	}

	/**
	 * 字符串先用 SHA1 加盐转换，然后将 SHA1 转换后的字符串转换为 16 进制格式（小写）的字符串
	 *
	 * @param input
	 *            待转换的字符串
	 * @param salt
	 *            盐
	 * @return 转换后的字符串
	 */
	private String encodeBySHA1WithSaltAndHex(String input, byte[] salt) {
		try {
			MessageDigest digest = MessageDigest.getInstance(MessageDigestAlgorithms.SHA_1);
			digest.update(salt);

			byte[] bytes = digest.digest(input.getBytes());

			return Hex.encodeHexString(bytes);

		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}

	}
}
