package com.ejianc.foundation.utils.weixinpay;

import cn.hutool.http.Header;
import cn.hutool.http.HttpRequest;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.wechat.pay.contrib.apache.httpclient.util.AesUtil;
import okhttp3.HttpUrl;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Map;
import java.util.Random;

public class WechatPayApiV3Util {
    private static Logger logger = LoggerFactory.getLogger(WechatPayApiV3Util.class);

    /**
     * 获取微信支付平台证书  注意不是商户api证书
     */
    private static final String WechatPaySerial = "https://api.mch.weixin.qq.com/v3/certificates";


    /**
     * 微信支付API v3 签名
     *
     * @param method       请求类型GET、POST
     * @param url          请求地址
     * @param body         请求数据 GET: 传"" POST: json串
     * @param merchantId   商户号
     * @param certSerialNo 商户证书（Api证书）序列号
     * @param filename     商户证书（Api证书） 私钥
     * @return
     * @throws Exception
     */
    public static String getToken(String method, String url, String body, String merchantId, String certSerialNo, String privateKey) throws Exception {
        String signStr = "";
        HttpUrl httpurl = HttpUrl.parse(url);
        // 随机字符串
        String nonceStr = getRandomString(32);
        // 时间戳
        long timestamp = System.currentTimeMillis() / 1000;
        if (StringUtils.isEmpty(body)) {
            body = "";
        }
        String message = buildMessage(method, httpurl, timestamp, nonceStr, body);
        String signature = sign(message.getBytes("utf-8"), privateKey);
        signStr = "mchid=\"" + merchantId + "\","
                + "nonce_str=\"" + nonceStr + "\","
                + "timestamp=\"" + timestamp + "\","
                + "serial_no=\"" + certSerialNo + "\","
                + "signature=\"" + signature + "\"";
        return signStr;
    }

    public static String buildMessage(String method, HttpUrl url, long timestamp, String nonceStr, String body) {
        String canonicalUrl = url.encodedPath();
        if (url.encodedQuery() != null) {
            canonicalUrl += "?" + url.encodedQuery();
        }
        return method + "\n"
                + canonicalUrl + "\n"
                + timestamp + "\n"
                + nonceStr + "\n"
                + body + "\n";
    }


    public static String sign(byte[] message, String privateKey) throws Exception {
        Signature sign = Signature.getInstance("SHA256withRSA");
        sign.initSign(getPrivateKey(privateKey));
        sign.update(message);
        return Base64.encodeBase64String(sign.sign());
    }

    /**
     * 获取私钥。
     *
     * @return 私钥对象
     */
    public static PrivateKey getPrivateKey(String content) throws IOException {
        // 编译后的相对路径
//        ClassPathResource classPathResource = new ClassPathResource(filename);
//        InputStream inputStream = classPathResource.getInputStream();
//        Scanner scanner = new Scanner(inputStream, "UTF-8");
//        String content = scanner.useDelimiter("\\A").next();
        // 绝对路径
//        String content = new String(Files.readAllBytes(Paths.get(filename)), "utf-8");
        try {
            String privateKey = content.replace("-----BEGIN PRIVATE KEY-----", "")
                    .replace("-----END PRIVATE KEY-----", "")
                    .replaceAll("\\s+", "");
            KeyFactory kf = KeyFactory.getInstance("RSA");
            return kf.generatePrivate(
                    new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey)));
        } catch (NoSuchAlgorithmException e) {
            logger.info("当前Java环境不支持RSA！------{}",e);
            throw new RuntimeException("当前Java环境不支持RSA", e);
        } catch (InvalidKeySpecException e) {
            logger.info("无效的密钥格式！------{}",e);
            throw new RuntimeException("无效的密钥格式");
        }
    }


    /**
     * 获取商户证书。
     *
     * @return X509证书
     */
    public static X509Certificate getCertificate() throws IOException {
        String APICLIENT_CERT = "pem/apiclient_cert.pem";
        InputStream fis = WechatPayApiV3Util.class.getClassLoader().getResourceAsStream(APICLIENT_CERT);
        try (BufferedInputStream bis = new BufferedInputStream(fis)) {
            CertificateFactory cf = CertificateFactory.getInstance("X509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(bis);
            cert.checkValidity();
            return cert;
        } catch (CertificateExpiredException e) {
            logger.info("证书已过期！------{}",e);
            throw new RuntimeException("证书已过期", e);
        } catch (CertificateNotYetValidException e) {
            logger.info("证书尚未生效！------{}",e);
            throw new RuntimeException("证书尚未生效", e);
        } catch (CertificateException e) {
            logger.info("无效的证书文件！------{}",e);
            throw new RuntimeException("无效的证书文件", e);
        }
    }

    /**
     *  生成随机数
     * @param length
     * @return
     */
    public static String getRandomString(int length) {
//      创建一个随机数生成器。
        SecureRandom random = new SecureRandom();
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < length; ++i) {
//          返回下一个伪随机数，它是此随机数生成器的序列中均匀分布的 int 值。
            int number = random.nextInt(3);
            long result = 0;
//          Math 类包含用于执行基本数学运算的方法，如初等指数、对数、平方根和三角函数。
//          Math.random()  返回带正号的 double 值，该值大于等于 0.0 且小于 1.0。
//          Math.round(Math.random() * 25 + 97)  返回最接近参数的 long。
            switch (number) {
                case 0:
                    result = Math.round(Math.random() * 25 + 65);
                    sb.append(String.valueOf((char) result));
                    break;
                case 1:
                    result = Math.round(Math.random() * 25 + 97);
                    sb.append(String.valueOf((char) result));
                    break;
                case 2:
                    sb.append(String.valueOf(new Random().nextInt(10)));
                    break;
            }
        }

        return sb.toString();
    }

    /**
     * 对称解密，异步通知的加密数据
     * @param resource 加密数据
     * @param apiV3Key apiV3密钥
     * @param type 1-支付，2-退款
     * @return
     */
    public static Map<String, Object> decryptFromResource(String resource, String apiV3Key, Integer type) {

        String msg = type==1?"支付成功":"退款成功";
        try {
            //通知数据
            Map<String, String> resourceMap = JSONObject.parseObject(resource, new TypeReference<Map<String, String>>() {});
            //数据密文
            String ciphertext = resourceMap.get("ciphertext");
            //随机串
            String nonce = resourceMap.get("nonce");
            //附加数据
            String associatedData = resourceMap.get("associated_data");

            AesUtil aesUtil = new AesUtil(apiV3Key.getBytes(StandardCharsets.UTF_8));
            String resourceStr = aesUtil.decryptToString(associatedData.getBytes(StandardCharsets.UTF_8),
                    nonce.getBytes(StandardCharsets.UTF_8),
                    ciphertext);

            return JSONObject.parseObject(resourceStr, new TypeReference<Map<String, Object>>(){});
        }catch (Exception e){
            logger.info("回调参数，解密失败！------{}",e);
            throw new RuntimeException("回调参数，解密失败！");
        }
    }


    public static void main(String[] args) throws Exception {
        String method = "POST";
        JSONObject body = new JSONObject();
        body.put("mchid", "1519756431");
        body.put("out_trade_no", "H51217752501201407033dsd0001");
        body.put("appid", "wxa453c95adb68fcdc");
        body.put("description", "测试测试");
        body.put("notify_url", "https://weixin.qq.com/");
        JSONObject amount = new JSONObject();
        amount.put("total", 1);
        amount.put("currency", "CNY");
        body.put("amount", amount);

        String url = "https://api.mch.weixin.qq.com/v3/pay/transactions/native";
        String mch_id = "1519756431";
        String token = WechatPayApiV3Util.getToken(method, url, body.toJSONString(), mch_id, "75831FE21DE9A7779EB4E63D8E6DBB3A9D7FF4ED", "D:\\WXCertUtil\\cert\\apiclient_key.pem");

        try {
            String result = HttpRequest.post(url)
                    .header(Header.CONTENT_TYPE, "application/json")
                    .header("ACCEPT", "application/json")
                    // 签名
                    .header("Authorization", "WECHATPAY2-SHA256-RSA2048" + " "
                            + token)
                    .body(body.toJSONString())
                    .execute().body();
            //应该创建 支付表数据
            if (result != null) {
                //业务逻辑-----------
                System.out.println("微信接口调用成功");
                System.out.println(result);

            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}