package com.ejianc.foundation.cache;

import java.nio.charset.Charset;
import java.util.List;

import com.ejianc.foundation.permission.vo.ButtonVO;
import com.ejianc.framework.core.kit.collection.ListUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.ejianc.foundation.permission.vo.AppVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;

@Component
public class AppCacheManage {

	public static final String DEFAULT_CHARSET = "UTF-8";
	// 所有应用缓存
	private final static String APP_LIST_KEY = "app_list_key:";
	// 所有按钮缓存
	private final static String BTN_KEY = "btn_key:";

	@Autowired
	private PermissionCacheManage permissionCacheManage;

	@Autowired
	private CacheManager cacheManager;

	private Serializer serializer = new DefaultJDKSerializer();
	
	/**
	 * 把应用list放入缓存
	 * 
	 * @param tenantId
	 * @param appVoList
	 */
	public void putAppListToCache(Long tenantId, List<AppVO> appVoList) {
		if(appVoList != null && appVoList.size() > 0) {
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				String key = APP_LIST_KEY + tenantId;
				for (AppVO appVo : appVoList) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
					byte[] fieldBytes = appVo.getId().toString().getBytes(Charset.forName(DEFAULT_CHARSET));
					byte[] contentBytes = serializer.marshalToByte(JSON.toJSONString(appVo));
					pipeline.hset(keyBytes, fieldBytes, contentBytes);
				}
				pipeline.exec();
			});
		}
	}
	
	/**
	 * 把某个应用放入缓存
	 * 
	 * @param tenantId
	 * @param appVo
	 */
	public void putAppBeanToCache(Long tenantId, AppVO appVo) {
		String key = APP_LIST_KEY + tenantId;
		cacheManager.hset(key, appVo.getId().toString(), JSON.toJSONString(appVo));
	}
	/**
	 * 把某个应用的按钮放入缓存
	 *
	 * @param vos
	 */
	public void putBtnToCache(List<ButtonVO> vos) {
		if(ListUtil.isNotEmpty(vos)){
			vos.forEach(btn->{
				String key = BTN_KEY + btn.getPageCode() + btn.getCode();
				cacheManager.set(key,JSON.toJSONString(btn));
			});
		}
	}
	/**
	 * 把某个应用的按钮缓存删除
	 *
	 * @param vos
	 */
	public void rmBtnFromCache(List<ButtonVO> vos) {
		if(ListUtil.isNotEmpty(vos)){
			vos.forEach(btn->{
				String key = BTN_KEY + btn.getPageCode() + btn.getCode();
				cacheManager.removeCache(key);
			});
		}
	}
	/**
	 * 获取app
	 *
	 * @return
	 */
	public ButtonVO getBtnCache(String pageCode,String uiKey) {
		String key = BTN_KEY + pageCode + uiKey;
		String content = cacheManager.get(key);
		return JSON.parseObject(content, ButtonVO.class);
	}
	/**
	 * 把某个按钮的权限缓存删除
	 *
	 * @param vos
	 */
	public void rmBtnRoleFromCache(List<ButtonVO> vos,List<Long> roleIds) {
		if(ListUtil.isNotEmpty(vos) && ListUtil.isNotEmpty(roleIds)){
			vos.forEach(btn->roleIds.forEach(roleId->permissionCacheManage.removeRoleAppsBtnCache(roleId,btn.getPageCode()+btn.getCode())));
		}
	}
	
	/**
	 * 获取app
	 * 
	 * @param tenantId
	 * @param appId
	 * @return
	 */
	public AppVO getAppsCache(Long tenantId, Long appId) {
		String key = APP_LIST_KEY + tenantId;
		String content = cacheManager.hget(key, appId.toString());
		AppVO appVo = JSON.parseObject(content, AppVO.class);
		return appVo;
	}

	/**
	 * 删除某个租户下的应用
	 * 
	 * @param tenantId
	 * @param appId
	 */
	public void removeAppCache(Long tenantId, Long appId) {
		String key = APP_LIST_KEY + tenantId;
		cacheManager.hdel(key, appId.toString());
	}
}
