package com.ejianc.business.income.controller;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.api.IRecoveriesApi;
import com.ejianc.business.finance.vo.ReceiveVO;
import com.ejianc.business.income.service.IReportService;
import com.ejianc.business.income.utils.BigDecimalUtil;
import com.ejianc.business.income.vo.MonthESDataVO;
import com.ejianc.business.income.vo.TableRowDataVO;
import com.ejianc.business.income.vo.report.ProjectMnyVO;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedLongTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedSum;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;

@RestController
@RequestMapping("/report")
public class ReportController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IReportService reportService;


    @Autowired
    private IOrgApi orgApi;

    @Autowired
    IProjectApi projectApi;

    @Autowired
    IRecoveriesApi recoveriesApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private RestHighLevelClient client;


    //利润表
    private final String PROFIT_STATISTICS = "profit_statistics";

    //资金表
    private final String CASH_STATISTICS = "cash_statistics";


    public final static Integer QUERY_TIMEOUT = 60;


    /**
     * @description: 中奥资金表
     * @author songlx
     * @date: 2023/11/9
     */
    @RequestMapping(value = "/queryCashStatistics", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<TableRowDataVO>> queryCashStatistics(@RequestParam String curYear, @RequestParam Long projectId) {
        if(StringUtils.isBlank(curYear)){
            throw new BusinessException("年份不能为空！");
        }
        if(projectId == null){
            throw new BusinessException("项目不能为空！");
        }
        List<TableRowDataVO> resVO = this.queryCashStatisticsData(curYear, projectId);
        return CommonResponse.success("查询利润表成功！", resVO);
    }

    private List<TableRowDataVO> queryCashStatisticsData(String curYear, Long projectId) {
        //款项性质、月份维度数据，包含收入合计、支出合计、利润
        Map<String, MonthESDataVO> dataMap = new HashMap<>();
        //返回给前端表格数据
        List<TableRowDataVO> tableRowDataVOS = new LinkedList<>();

        try {
            SearchRequest searchRequest = new SearchRequest(CASH_STATISTICS);
            SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
            //1、查询项目当前年所有月份数据，放入dataMap
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("curYear", curYear));
            boolQuery.must(QueryBuilders.termQuery("projectId", projectId));
            sourceBuilder.query(boolQuery);
            sourceBuilder.from(0);
            sourceBuilder.size(10000);
            searchRequest.source(sourceBuilder);
            sourceBuilder.trackTotalHits(true);
            sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS));

            List<MonthESDataVO> list = new ArrayList<>();
            SearchResponse response = null;

            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, MonthESDataVO.class));
            }
            if (CollectionUtils.isNotEmpty(list)) {
                dataMap = list.stream().collect(Collectors.toMap(
                        obj -> obj.getFeeType() + "_" + obj.getCurMonth(), Function.identity(), (key1, key2) -> key1));
            }

            //2、查询项目上期余额初始值（年份小于当前年，款型性质分组合计当前项目计算），放入dataMap
            SearchRequest searchRequestForQiChu = new SearchRequest(CASH_STATISTICS);
            BoolQueryBuilder boolQueryForQiChu = QueryBuilders.boolQuery();
            boolQueryForQiChu.must(QueryBuilders.rangeQuery("curYear").lt(curYear));
            boolQueryForQiChu.must(QueryBuilders.termQuery("projectId", projectId));

            SearchSourceBuilder sourceBuilderForQiChu = new SearchSourceBuilder();
            sourceBuilderForQiChu.from(0);
            sourceBuilderForQiChu.size(10000);
            sourceBuilderForQiChu.query(boolQueryForQiChu);
            sourceBuilderForQiChu.trackTotalHits(true);
            sourceBuilderForQiChu.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
            searchRequestForQiChu.source(sourceBuilderForQiChu);
            SearchResponse search = client.search(searchRequestForQiChu, RequestOptions.DEFAULT);

            SearchHits hits4QiChu = search.getHits();
            List<MonthESDataVO> list4QiChu = new ArrayList<>();
            for (SearchHit hit : hits4QiChu) {
                String sourceAsString = hit.getSourceAsString();
                list4QiChu.add(JSON.parseObject(sourceAsString, MonthESDataVO.class));
            }
            if (CollectionUtils.isNotEmpty(list4QiChu)) {
                Map<Integer, BigDecimal> qichuTaxMnySumMap = list4QiChu.stream().collect(Collectors.groupingBy(MonthESDataVO::getFeeType, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));
                BigDecimal lastSurplusIncome = BigDecimal.ZERO;
                BigDecimal lastSurplusCost = BigDecimal.ZERO;
                for (Integer feeType : qichuTaxMnySumMap.keySet()) {
                    BigDecimal taxMny = qichuTaxMnySumMap.get(feeType);
                    if(!feeType.toString().startsWith("106")){
                        if(feeType.toString().startsWith("1")){
                            lastSurplusIncome = ComputeUtil.safeAdd(lastSurplusIncome, taxMny);
                        } else {
                            lastSurplusCost = ComputeUtil.safeAdd(lastSurplusCost, taxMny);
                        }
                    }
                    MonthESDataVO qichuObj = new MonthESDataVO();
                    qichuObj.setCurMonth(0);
                    qichuObj.setFeeType(feeType);
                    qichuObj.setTaxMny(taxMny);
                    dataMap.put(feeType + "_0", qichuObj);
                    //logger.info( "feeType:{}, qichuObj:{}", feeType, JSONObject.toJSONString(qichuObj));
                }
                MonthESDataVO lastSurplusObj = new MonthESDataVO();
                lastSurplusObj.setCurMonth(1);
                lastSurplusObj.setFeeType(106);
                lastSurplusObj.setTaxMny(ComputeUtil.safeSub(lastSurplusIncome, lastSurplusCost));
                dataMap.put( "106_1", lastSurplusObj);
            }
            //3、计算纵向经营活动流入小计、经营活动流出小计、利经营活动产生的现金流量净额：将数据放入dataMap
            List<MonthESDataVO> incomeESData = new ArrayList<>();
            List<MonthESDataVO> costESData = new ArrayList<>();
            for (String feeTypeCurMonth : dataMap.keySet()) {
                if(!feeTypeCurMonth.startsWith("106")){
                    if (feeTypeCurMonth.startsWith("1")) {
                        incomeESData.add(dataMap.get(feeTypeCurMonth));
                    } else {
                        costESData.add(dataMap.get(feeTypeCurMonth));
                    }
                }
            }
            //3.1 经营活动流入小计
            Map<Integer, BigDecimal> incomeTaxMnySumGroup = incomeESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));

            if (MapUtils.isNotEmpty(incomeTaxMnySumGroup)) {
                for (Integer key : incomeTaxMnySumGroup.keySet()) {
                    MonthESDataVO incomeSumObj = new MonthESDataVO();
                    incomeSumObj.setTaxMny(incomeTaxMnySumGroup.get(key));
                    dataMap.put("199_" + key, incomeSumObj);
                }
            }

            //3.2 经营活动流出小计
            Map<Integer, BigDecimal> costTaxMnySumGroup = costESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));
            if (MapUtils.isNotEmpty(costTaxMnySumGroup)) {
                for (Integer key : costTaxMnySumGroup.keySet()) {
                    MonthESDataVO costSumObj = new MonthESDataVO();
                    costSumObj.setTaxMny(costTaxMnySumGroup.get(key));
                    dataMap.put("299_" + key, costSumObj);
                }
            }

            //3.3 计算2月份以后的上期余额、经营活动产生的现金流量净额= 上期余额+本月收入-本月支出
            for (int i = 1; i <= 12; i++) {
                Integer year = DateUtil.thisYear();//获取当前年份
                Integer mon = DateUtil.thisMonth()+1;//获取当前月份DateUtil.thisMonth()获取的月份从0开始计数
                if (year.compareTo(Integer.valueOf(curYear))==1) {//当前年份大于所选年份则展示全部数据
                    if (i > 1) {
                        MonthESDataVO lastSurplusObj2 = new MonthESDataVO();
                        int lastmon = i - 1;
                        MonthESDataVO curIncomeDataVO = dataMap.get("199_" + lastmon);
                        MonthESDataVO curCostDataVO = dataMap.get("299_" + lastmon);
                        BigDecimal curIncomeMny = curIncomeDataVO != null ? curIncomeDataVO.getTaxMny() : null;
                        BigDecimal curCostMny = curCostDataVO != null ? curCostDataVO.getTaxMny() : null;
                        MonthESDataVO _lastSurplusDataVO = dataMap.get("106_" + lastmon);
                        BigDecimal _lastSurplusMny = _lastSurplusDataVO != null ? _lastSurplusDataVO.getTaxMny() : null;
                        BigDecimal curSurplusMny = ComputeUtil.safeAdd(_lastSurplusMny, ComputeUtil.safeSub(curIncomeMny, curCostMny));
                        lastSurplusObj2.setCurMonth(i);
                        lastSurplusObj2.setFeeType(106);
                        lastSurplusObj2.setTaxMny(curSurplusMny);
                        dataMap.put("106_" + i, lastSurplusObj2);
                    }
                    MonthESDataVO profitSumObj = new MonthESDataVO();
                    MonthESDataVO incomeSum = dataMap.get("199_" + i);
                    MonthESDataVO costSumObj = dataMap.get("299_" + i);
                    BigDecimal incomeSumMny = incomeSum != null ? incomeSum.getTaxMny() : null;
                    BigDecimal costSumMny = costSumObj != null ? costSumObj.getTaxMny() : null;
                    MonthESDataVO lastSurplusMonthESDataVO = dataMap.get("106_" + i);
                    BigDecimal lastSurplusMny = lastSurplusMonthESDataVO != null ? lastSurplusMonthESDataVO.getTaxMny() : null;
                    BigDecimal surplusMny = ComputeUtil.safeAdd(lastSurplusMny, ComputeUtil.safeSub(incomeSumMny, costSumMny));
                    profitSumObj.setTaxMny(surplusMny);
                    dataMap.put("301_" + i, profitSumObj);
                }else if (year.compareTo(Integer.valueOf(curYear))==0){//当前年份等于所选年份则展示当前月份之前数据
                    if (mon<i){
                        continue;
                    }
                    if (i > 1) {
                        MonthESDataVO lastSurplusObj2 = new MonthESDataVO();
                        int lastmon = i - 1;
                        MonthESDataVO curIncomeDataVO = dataMap.get("199_" + lastmon);
                        MonthESDataVO curCostDataVO = dataMap.get("299_" + lastmon);
                        BigDecimal curIncomeMny = curIncomeDataVO != null ? curIncomeDataVO.getTaxMny() : null;
                        BigDecimal curCostMny = curCostDataVO != null ? curCostDataVO.getTaxMny() : null;
                        MonthESDataVO _lastSurplusDataVO = dataMap.get("106_" + lastmon);
                        BigDecimal _lastSurplusMny = _lastSurplusDataVO != null ? _lastSurplusDataVO.getTaxMny() : null;
                        BigDecimal curSurplusMny = ComputeUtil.safeAdd(_lastSurplusMny, ComputeUtil.safeSub(curIncomeMny, curCostMny));
                        lastSurplusObj2.setCurMonth(i);
                        lastSurplusObj2.setFeeType(106);
                        lastSurplusObj2.setTaxMny(curSurplusMny);
                        dataMap.put("106_" + i, lastSurplusObj2);
                    }
                    MonthESDataVO profitSumObj = new MonthESDataVO();
                    MonthESDataVO incomeSum = dataMap.get("199_" + i);
                    MonthESDataVO costSumObj = dataMap.get("299_" + i);
                    BigDecimal incomeSumMny = incomeSum != null ? incomeSum.getTaxMny() : null;
                    BigDecimal costSumMny = costSumObj != null ? costSumObj.getTaxMny() : null;
                    MonthESDataVO lastSurplusMonthESDataVO = dataMap.get("106_" + i);
                    BigDecimal lastSurplusMny = lastSurplusMonthESDataVO != null ? lastSurplusMonthESDataVO.getTaxMny() : null;
                    BigDecimal surplusMny = ComputeUtil.safeAdd(lastSurplusMny, ComputeUtil.safeSub(incomeSumMny, costSumMny));
                    profitSumObj.setTaxMny(surplusMny);
                    dataMap.put("301_" + i, profitSumObj);
                }
            }

            //4、获取税金
            List<TableRowDataVO> profitTableRowDataVOS = this.queryProfitStatisticsData(curYear, projectId);
            TableRowDataVO taxRowDataVO = null;
            if(CollectionUtils.isNotEmpty(profitTableRowDataVOS)){
                for (TableRowDataVO profitTableRowDataVO : profitTableRowDataVOS) {
                    if(profitTableRowDataVO.getFeeType().equals(Integer.valueOf(208))){
                        taxRowDataVO = profitTableRowDataVO;
                    }
                }
            }

            //5、组装表格最终数据，将数据转成行数据
            Map<Integer, String> feeTypeMap = TableRowDataVO.CASH_FEE_TYPE_MAP;
            for (Integer feeType : feeTypeMap.keySet()) {
                TableRowDataVO rowDataVO = new TableRowDataVO();
                rowDataVO.setFeeType(feeType, TableRowDataVO.CASH_FEE_TYPE_MAP);
                for (int i = 1; i <= 12; i++) {
                    String dataKey = feeType + "_" + i;
                    MonthESDataVO monthESDataVO = dataMap.get(dataKey);
                    if (monthESDataVO != null && monthESDataVO.getTaxMny() != null) {
                        BeanUtils.setProperty(rowDataVO, "taxMny" + i, monthESDataVO.getTaxMny());
                    }
                }
                tableRowDataVOS.add(rowDataVO.sumYearTotalAndStartTotal());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return tableRowDataVOS;
    }



    /**
     * @description: 中奥利润表
     * @author songlx
     * @date: 2023/11/9
     */
    @RequestMapping(value = "/queryProfitStatistics", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<TableRowDataVO>> queryProfitStatistics(@RequestParam String curYear, @RequestParam Long projectId) {
        if(StringUtils.isBlank(curYear)){
            throw new BusinessException("年份不能为空！");
        }
        if(projectId == null){
            throw new BusinessException("项目不能为空！");
        }
        List<TableRowDataVO> resVO = this.queryProfitStatisticsData(curYear, projectId);
        return CommonResponse.success("查询利润表成功！", resVO);
    }

    private List<TableRowDataVO> queryProfitStatisticsData(String curYear, Long projectId) {
        //款项性质、月份维度数据，包含收入合计、支出合计、利润
        Map<String, MonthESDataVO> dataMap = new HashMap<>();
        //返回给前端表格数据
        List<TableRowDataVO> tableRowDataVOS = new LinkedList<>();

        try {
            SearchRequest searchRequest = new SearchRequest(PROFIT_STATISTICS);
            SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
            //1、查询项目当前年所有月份数据，放入dataMap
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("curYear", curYear));
            boolQuery.must(QueryBuilders.termQuery("projectId", projectId));
            sourceBuilder.query(boolQuery);
            sourceBuilder.from(0);
            sourceBuilder.size(10000);
            searchRequest.source(sourceBuilder);
            sourceBuilder.trackTotalHits(true);
            sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS));

            List<MonthESDataVO> list = new ArrayList<>();
            SearchResponse response = null;

            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, MonthESDataVO.class));
            }
            if (CollectionUtils.isNotEmpty(list)) {
                dataMap = list.stream().collect(Collectors.toMap(
                        obj -> obj.getFeeType() + "_" + obj.getCurMonth(), Function.identity(), (key1, key2) -> key1));
            }

            //2、查询项目期初值数据（年份小于当前年，款型性质分组合计当前项目计算），放入dataMap
            SearchRequest searchRequestForQiChu = new SearchRequest(PROFIT_STATISTICS);
            BoolQueryBuilder boolQueryForQiChu = QueryBuilders.boolQuery();
            boolQueryForQiChu.must(QueryBuilders.rangeQuery("curYear").lt(curYear));
            boolQueryForQiChu.must(QueryBuilders.termQuery("projectId", projectId));

            SearchSourceBuilder sourceBuilderForQiChu = new SearchSourceBuilder();
            sourceBuilderForQiChu.from(0);
            sourceBuilderForQiChu.size(10000);
            sourceBuilderForQiChu.query(boolQueryForQiChu);
            sourceBuilderForQiChu.trackTotalHits(true);
            sourceBuilderForQiChu.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
            searchRequestForQiChu.source(sourceBuilderForQiChu);
            SearchResponse search = client.search(searchRequestForQiChu, RequestOptions.DEFAULT);

            SearchHits hits4QiChu = search.getHits();
            List<MonthESDataVO> list4QiChu = new ArrayList<>();
            for (SearchHit hit : hits4QiChu) {
                String sourceAsString = hit.getSourceAsString();
                list4QiChu.add(JSON.parseObject(sourceAsString, MonthESDataVO.class));
            }
            if (CollectionUtils.isNotEmpty(list4QiChu)) {
                Map<Integer, BigDecimal> qichuTaxMnySumMap = list4QiChu.stream().collect(Collectors.groupingBy(MonthESDataVO::getFeeType, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));
                Map<Integer, BigDecimal> qichuMnySumMap = list4QiChu.stream().collect(Collectors.groupingBy(MonthESDataVO::getFeeType, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getMny, ComputeUtil::safeAdd)));
                Map<Integer, BigDecimal> qichuTaxSumMap = list4QiChu.stream().collect(Collectors.groupingBy(MonthESDataVO::getFeeType, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTax, ComputeUtil::safeAdd)));
                for (Integer feeType : qichuTaxMnySumMap.keySet()) {
                    MonthESDataVO qichuObj = new MonthESDataVO();
                    qichuObj.setCurMonth(0);
                    qichuObj.setFeeType(feeType);
                    qichuObj.setTaxMny(qichuTaxMnySumMap.get(feeType));
                    qichuObj.setMny(qichuMnySumMap.get(feeType));
                    qichuObj.setTax(qichuTaxSumMap.get(feeType));
                    dataMap.put(feeType + "_0", qichuObj);
                    logger.info( "feeType:{}, qichuObj:{}", feeType, JSONObject.toJSONString(qichuObj));
                }
            }
           /* Aggregations aggregations = search.getAggregations();
            ParsedLongTerms terms = aggregations.get("feeTypeGroup");
            List<? extends Terms.Bucket> buckets = terms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                Integer feeType = Integer.valueOf(bucket.getKey().toString());
                Aggregations bucketAggregations = bucket.getAggregations();
                MonthESDataVO qichuObj = new MonthESDataVO();
                qichuObj.setCurMonth(0);
                qichuObj.setFeeType(feeType);
                ParsedSum taxMnySum = bucketAggregations.get("taxMny");
                BigDecimal taxMny = ComputeUtil.toBigDecimal(taxMnySum.getValue());
                qichuObj.setTaxMny(taxMny);
                ParsedSum mnySum = bucketAggregations.get("mny");
                BigDecimal mny = ComputeUtil.toBigDecimal(mnySum.getValue());
                qichuObj.setMny(mny);
                ParsedSum taxSum = bucketAggregations.get("tax");
                BigDecimal tax = ComputeUtil.toBigDecimal(taxSum.getValue());
                qichuObj.setTax(tax);
                dataMap.put(feeType + "_0", qichuObj);
                logger.info( "feeType:{}, qichuObj:{}", feeType, JSONObject.toJSONString(qichuObj));
            }*/

            //3、计算纵向收入小计、支出小计、利润：将dataMap的数据按照收入、支出分组计算出来收入小计、支出小计、利润放入dataMap
            List<MonthESDataVO> incomeESData = new ArrayList<>();
            List<MonthESDataVO> costESData = new ArrayList<>();
            for (String feeTypeCurMonth : dataMap.keySet()) {
                if (feeTypeCurMonth.startsWith("1")) {
                    incomeESData.add(dataMap.get(feeTypeCurMonth));
                } else {
                    costESData.add(dataMap.get(feeTypeCurMonth));
                }
            }
            //3.1 收入小计
            Map<Integer, BigDecimal> incomeTaxMnySumGroup = incomeESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));
            Map<Integer, BigDecimal> incomeMnySumGroup = incomeESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getMny, ComputeUtil::safeAdd)));
            Map<Integer, BigDecimal> incomeTaxSumGroup = incomeESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTax, ComputeUtil::safeAdd)));

            if (MapUtils.isNotEmpty(incomeTaxMnySumGroup)) {
                for (Integer key : incomeTaxMnySumGroup.keySet()) {
                    MonthESDataVO incomeSumObj = new MonthESDataVO();
                    incomeSumObj.setTaxMny(incomeTaxMnySumGroup.get(key));
                    incomeSumObj.setMny(incomeMnySumGroup.get(key));
                    incomeSumObj.setTax(incomeTaxSumGroup.get(key));
                    dataMap.put("199_" + key, incomeSumObj);
                }
            }

            //3.2 支出小计
            Map<Integer, BigDecimal> costTaxMnySumGroup = costESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTaxMny, ComputeUtil::safeAdd)));
            Map<Integer, BigDecimal> costMnySumGroup = costESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getMny, ComputeUtil::safeAdd)));
            Map<Integer, BigDecimal> costTaxSumGroup = costESData.stream().collect(Collectors.groupingBy(MonthESDataVO::getCurMonth, Collectors.reducing(BigDecimal.ZERO, MonthESDataVO::getTax, ComputeUtil::safeAdd)));

            if (MapUtils.isNotEmpty(incomeTaxMnySumGroup)) {
                for (Integer key : incomeTaxMnySumGroup.keySet()) {
                    MonthESDataVO costSumObj = new MonthESDataVO();
                    costSumObj.setTaxMny(costTaxMnySumGroup.get(key));
                    costSumObj.setMny(costMnySumGroup.get(key));
                    costSumObj.setTax(costTaxSumGroup.get(key));
                    dataMap.put("299_" + key, costSumObj);
                }
            }

            //3.3 利润
            for (int i = 0; i <= 12; i++) {
                MonthESDataVO profitSumObj = new MonthESDataVO();
                MonthESDataVO incomeSum = dataMap.get("199_" + i);
                MonthESDataVO costSumObj = dataMap.get("299_" + i);
                BigDecimal incomeSumTaxMny = incomeSum != null ? incomeSum.getTaxMny() : null;
                BigDecimal incomeSumMny = incomeSum != null ? incomeSum.getMny() : null;
                BigDecimal incomeSumTax = incomeSum != null ? incomeSum.getTax() : null;
                BigDecimal costSumTaxMny = costSumObj != null ? costSumObj.getTaxMny() : null;
                BigDecimal costSumMny = costSumObj != null ? costSumObj.getMny() : null;
                BigDecimal costSumTax = costSumObj != null ? costSumObj.getTax() : null;
                profitSumObj.setTaxMny(ComputeUtil.safeSub(incomeSumTaxMny, costSumTaxMny));
                profitSumObj.setMny(ComputeUtil.safeSub(incomeSumMny, costSumMny));
                profitSumObj.setTax(ComputeUtil.safeSub(incomeSumTax, costSumTax));
                dataMap.put("301_" + i, profitSumObj);
            }

            //4、组装表格最终数据，将数据转成行数据
            Map<Integer, String> feeTypeMap = TableRowDataVO.PROFIT_FEE_TYPE_MAP;
            for (Integer feeType : feeTypeMap.keySet()) {
                TableRowDataVO rowDataVO = new TableRowDataVO();
                rowDataVO.setFeeType(feeType, TableRowDataVO.PROFIT_FEE_TYPE_MAP);
                //税金，只需要放入期初税金
                if(Integer.valueOf(208).equals(feeType)){
                    for (int i = 0; i <= 12; i++) {
                        String dataKey = feeType + "_" + i;
                        MonthESDataVO monthESDataVO = dataMap.get(dataKey);
                        if (monthESDataVO != null && monthESDataVO.getTax()!=null) {
                        BeanUtils.setProperty(rowDataVO, "tax" + i, monthESDataVO.getTax());
                        }
                    }

                } else {
                    for (int i = 0; i <= 12; i++) {
                        String dataKey = feeType + "_" + i;
                        MonthESDataVO monthESDataVO = dataMap.get(dataKey);
                        if (monthESDataVO != null && monthESDataVO.getTaxMny() != null) {
                            BeanUtils.setProperty(rowDataVO, "taxMny" + i, monthESDataVO.getTaxMny());
                            BeanUtils.setProperty(rowDataVO, "mny" + i, monthESDataVO.getMny());
                            BeanUtils.setProperty(rowDataVO, "tax" + i, monthESDataVO.getTax());
                        }
                    }
                }
                tableRowDataVOS.add(rowDataVO.sumTotal());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return tableRowDataVOS;
    }


    public static void main(String[] args) {
        for(int i = 0; i <= 12; i++){
            System.out.println("private BigDecimal taxMny" + i + ";");
            System.out.println("private BigDecimal mny" + i + ";");
            System.out.println("private BigDecimal tax" + i + ";");
        }

        StringBuilder taxMnyTotal = new StringBuilder();
        StringBuilder mnyTotal = new StringBuilder();
        StringBuilder taxTotal = new StringBuilder();
        for(int i = 0; i <= 12; i++){
            taxMnyTotal.append("this.taxMny" + i + ", ");
            mnyTotal.append("this.mny" + i + ", ");
            taxTotal.append("this.tax" + i + ", ");
        }

        System.out.println(taxMnyTotal.toString());
        System.out.println(mnyTotal.toString());
        System.out.println(taxTotal.toString());
    }

    /**
     * 项目回款台账
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/projectPageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> projectPageList(@RequestBody QueryParam param) {
        return CommonResponse.success("查询列表数据成功！", reportService.pageList(param));
    }

    /**
     * 月度经营报表
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/monthPageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> monthPageList(@RequestBody QueryParam param) {
        return CommonResponse.success("查询列表数据成功！", reportService.monthPageList(param));
    }

    @RequestMapping(value = "getProjectInOutQueryParam", method = RequestMethod.GET)
    public CommonResponse<QueryParam> getProjectInOutQueryParam() {
        try {
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
            String orgType = InvocationInfoProxy.getOrgType();
            if ("5".equals(orgType)) {
                queryParam.getParams().put("orgId", new Parameter(QueryParam.EQ, 1));
            }
            return CommonResponse.success(queryParam);
        } catch (Exception e) {
            logger.error("系统异常：" + e.getMessage());
            e.printStackTrace();
        }
        return CommonResponse.error("查询失败");
    }

    /**
     * 项目回款台账导出
     *
     * @param param
     * @return
     */
    @PostMapping("projectExport")
    public void projectExport(@RequestBody QueryParam param, HttpServletResponse response) {
        param.setPageIndex(1);
        param.setPageSize(10000);
        JSONObject pageList = reportService.pageList(param);
        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", pageList.get("records"));
        ExcelExport.getInstance().export("projectReport-export.xlsx", beans, response);
    }

    /**
     * 月度经营报表导出
     *
     * @param param
     * @return
     */
    @PostMapping("monthExport")
    public void monthExport(@RequestBody QueryParam param, HttpServletResponse response) {
        param.setPageIndex(1);
        param.setPageSize(10000);
        JSONObject pageList = reportService.monthPageList(param);
        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", pageList.get("records"));
        ExcelExport.getInstance().export("monthReport-export.xlsx", beans, response);
    }
    /***************  移动看板start  *****************/
    /***************  移动看板end    *****************/

    /**
     * 回款登记
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/queryProjectMnyList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<ProjectMnyVO>> queryProjectMnyList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("projectName");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        UserContext userContextCache = sessionManager.getUserContext();
        String authOrgIds = userContextCache.getAuthOrgIds();
        //先使用pc端的orgid
        if (InvocationInfoProxy.getOrgId() != null) {
            Long orgId = InvocationInfoProxy.getOrgId();
            if (null != orgId && orgId > 0) {
                List<Long> commonOrgIds = new ArrayList<>();
                CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
                if (!orgListResp.isSuccess()) {
                    throw new BusinessException("列表查询失败, 查询组织信息失败！");
                }
                commonOrgIds.addAll(orgListResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
                if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                    param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
                }
            }
        } else {
            if (StringUtils.isNotEmpty(authOrgIds)) {
                String[] split = authOrgIds.split(",");
                List<Long> orgList = new ArrayList<>();
                List<String> collect = Arrays.stream(split).collect(Collectors.toList());
                collect.forEach(item -> {
                    orgList.add(Long.parseLong(item));
                });
                param.getParams().put("orgId", new Parameter(QueryParam.IN, orgList));
            }
        }
        //截止时间
        Map<String, Parameter> paramsmap = param.getParams();
        if (paramsmap.containsKey("quoteDate")) {
            Date date = DateUtil.date();
            int year = DateUtil.year(date);
            String quoteDate = (String) paramsmap.get("quoteDate").getValue();
            quoteDate = year + "-" + quoteDate + "-31";
            param.getParams().put("quote_date", new Parameter(QueryParam.LE, quoteDate));
            paramsmap.remove("quoteDate");
        }

        QueryWrapper wrapper = changeToQueryWrapper(param);
        wrapper.groupBy("project_id");
        Page<ProjectMnyVO> page = new Page<>(param.getPageIndex(), param.getPageSize());
        List<ProjectMnyVO> list = reportService.queryProjectMnyList(page, wrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            List<Long> projectIds = list.stream().map(ProjectMnyVO::getProjectId).collect(Collectors.toList());
            CommonResponse<List<ProjectRegisterVO>> projectRes = projectApi.queryProjectByIds(projectIds);
            CommonResponse<List<ReceiveVO>> recoveries = recoveriesApi.queryProjectReceiveMny(projectIds);
            Map<Long, ProjectRegisterVO> projectMap = new HashMap<>();
            if (projectRes.isSuccess() && CollectionUtils.isNotEmpty(projectRes.getData())) {
                projectMap = projectRes.getData().stream().collect(Collectors.toMap(k -> k.getId(), (k) -> k));
            }
            Map<Long, ReceiveVO> recoverieMap = new HashMap<>();
            if (recoveries.isSuccess() && CollectionUtils.isNotEmpty(recoveries.getData())) {
                recoverieMap = recoveries.getData().stream().collect(Collectors.toMap(k -> k.getProjectId(), (k) -> k));
            }
            for (ProjectMnyVO mnyVO : list) {
                Long projectId = mnyVO.getProjectId();
                ProjectRegisterVO projectRegisterVO = projectMap.get(projectId);
                if (null != projectRegisterVO) {
                    mnyVO.setParentOrgId(projectRegisterVO.getOrgId());
                    mnyVO.setParentOrgName(projectRegisterVO.getOrgName());
                    mnyVO.setProjectManager(projectRegisterVO.getProjectManagerName());
                    mnyVO.setPubUnitName(projectRegisterVO.getPubUnitName());
                    mnyVO.setInvestorName(projectRegisterVO.getInvestorName());
                }
                ReceiveVO receiveVO = recoverieMap.get(projectId);
                if (null != receiveVO) {
                    mnyVO.setReceiveMny(receiveVO.getReceiveMny());
                }
                if (mnyVO.getParentOrgId() == null)
                    logger.error("此数据【projectId：{}】的ParentOrgId为空！", mnyVO.getProjectId());
            }
            if (CollectionUtils.isNotEmpty(list)) {
                Map<Long, List<ProjectMnyVO>> map = list.stream().filter(s -> s != null && s.getParentOrgId() != null).collect(Collectors.groupingBy(ProjectMnyVO::getParentOrgId));
                for (ProjectMnyVO v : list) {
                    Long parentorgid = v.getParentOrgId();
                    List<ProjectMnyVO> ll = map.get(parentorgid);
                    if (CollectionUtils.isNotEmpty(ll)) {
                        v.setOrgRowSpan(ll.size());
                        map.remove(parentorgid);
                    } else {
                        v.setOrgRowSpan(0);
                    }
                }
            }
        }

        Map<Long, List<ProjectMnyVO>> map = new HashMap<>();
        map = list.stream().filter(s -> s != null).filter(s -> s.getProjectId() != null).collect(Collectors.groupingBy(ProjectMnyVO::getProjectId));
        for (ProjectMnyVO mnyVO : list) {
            Long projectId = mnyVO.getProjectId();
            List<ProjectMnyVO> projectMnyVOS = map.get(projectId);
            if (CollectionUtils.isNotEmpty(projectMnyVOS)) {
                BigDecimal sumreceivemny = projectMnyVOS.stream().filter(s -> s.getReceiveMny() != null).map(ProjectMnyVO::getReceiveMny).reduce(BigDecimal.ZERO, BigDecimal::add);
                mnyVO.setSumReceiveMny(sumreceivemny);
                BigDecimal sumQuoteTaxMny = projectMnyVOS.stream().filter(s -> s.getQuoteTaxMny() != null).map(ProjectMnyVO::getQuoteTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
                mnyVO.setSumQuoteTaxMny(sumQuoteTaxMny);
                BigDecimal yun = BigDecimalUtil.safeSub(sumQuoteTaxMny, sumreceivemny);
                mnyVO.setYue(yun);
            }
        }
        List<ProjectMnyVO> collect = list.stream().filter(s -> s.getParentOrgId() != null).sorted(Comparator.comparing(ProjectMnyVO::getParentOrgId)).collect(Collectors.toList());
        IPage<ProjectMnyVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(collect);
        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    @RequestMapping(value = "excelExportInAndOutStoreList", method = RequestMethod.POST)
    @ResponseBody
    public void excelExportInAndOutStoreList(@RequestBody QueryParam param, HttpServletResponse response) throws IOException {
        param.setPageSize(10000);
        param.setPageIndex(1);
        CommonResponse<IPage<ProjectMnyVO>> iPageCommonResponse = queryProjectMnyList(param);
        List<ProjectMnyVO> records = iPageCommonResponse.getData().getRecords();
        int num = 0;

        if (ListUtil.isNotEmpty(records)) {
            ServletOutputStream outputStream = response.getOutputStream();
            XSSFWorkbook workbook = new XSSFWorkbook();
            XSSFSheet sheet = workbook.createSheet();
            sheet.createFreezePane(0, 2, 0, 2);
            sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, 7));
            sheet.setColumnWidth(7, 60 * 135);
            //首行标题
            XSSFRow row = sheet.createRow(0);
            CellStyle style = workbook.createCellStyle();
            style.setFillForegroundColor(IndexedColors.PALE_BLUE.getIndex());
            style.setFillPattern(CellStyle.SOLID_FOREGROUND);
            style.setAlignment(CellStyle.ALIGN_CENTER);

            XSSFCell cell = row.createCell(0);
            cell.setCellValue("回款汇总");
            cell.setCellStyle(style);
            //第二行数据
            row = sheet.createRow(1);

            cell = row.createCell(0);
            cell.setCellValue("区域");

            cell.setCellStyle(style);
            cell = row.createCell(1);
            cell.setCellValue("项目经理");

            cell.setCellStyle(style);
            cell = row.createCell(2);
            cell.setCellStyle(style);
            cell.setCellValue("项目");

            cell = row.createCell(3);
            cell.setCellValue("投资主体");
            cell.setCellStyle(style);

            cell = row.createCell(4);
            cell.setCellValue("发包单位");
            cell.setCellStyle(style);

            cell = row.createCell(5);
            cell.setCellValue("回款金额");
            cell.setCellStyle(style);

            cell = row.createCell(6);
            cell.setCellValue("实际回款");
            cell.setCellStyle(style);

            cell = row.createCell(7);
            cell.setCellValue("余额");
            cell.setCellStyle(style);


            for (int i = 0; i < 7; i++) {
                sheet.autoSizeColumn(i);
                sheet.setColumnWidth(i, sheet.getColumnWidth(i) * 60 / 10);
            }
            for (int i = 0; i < records.size(); i++) {
                ProjectMnyVO vo = records.get(i);
                row = sheet.createRow(i + 2);
                if (vo.getOrgRowSpan() > 0) {
                    cell = row.createCell(0);
                    cell.setCellValue(vo.getParentOrgName() == null ? "" : vo.getParentOrgName().toString());
                    sheet.addMergedRegion(new CellRangeAddress(row.getRowNum(), row.getRowNum() + (vo.getOrgRowSpan() - 1), 0, 0));
                }
                cell = row.createCell(1);
                cell.setCellValue(vo.getProjectManager() != null ? vo.getProjectManager() : null);

                cell = row.createCell(2);
                cell.setCellValue(vo.getProjectName() != null ? vo.getProjectName().toString() : null);

                cell = row.createCell(3);
                cell.setCellValue(vo.getInvestorName() != null ? vo.getInvestorName().toString() : null);

                cell = row.createCell(4);
                cell.setCellValue(vo.getPubUnitName() != null ? vo.getPubUnitName().toString() : null);

                cell = row.createCell(5);
                cell.setCellValue(vo.getSumQuoteTaxMny() != null ? vo.getSumQuoteTaxMny().setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");

                cell = row.createCell(6);
                cell.setCellValue(vo.getReceiveMny() != null ? vo.getReceiveMny().setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");

                cell = row.createCell(7);
                cell.setCellValue(vo.getYue() != null ? vo.getYue().setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");
            }
            List<Integer> list1 = new ArrayList<>();
            records.forEach(item -> {
                if (item.getOrgRowSpan() > 0) {
                    Integer orgRowSpan = item.getOrgRowSpan();
                    list1.add(orgRowSpan);
                    int sum1 = (int) list1.stream().collect(Collectors.summarizingInt(value -> value)).getSum();
                    sheet.shiftRows(sum1 + 2, sheet.getLastRowNum() + 1, 1, true, false);
                    XSSFRow newRow = sheet.createRow(sum1 + 2);
                    XSSFCell newCell = newRow.createCell(0);
                    newCell = newRow.createCell(0);
                    newCell.setCellValue(item.getParentOrgName() == null ? "" : item.getParentOrgName().toString());
                    newCell = newRow.createCell(1);
                    newCell.setCellValue("合计");

                    BigDecimal sumQuoteTaxMny = records.stream().filter(s -> (s.getParentOrgId().equals(item.getParentOrgId()) && s.getSumQuoteTaxMny() != null)).map(ProjectMnyVO::getSumQuoteTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
                    newCell = newRow.createCell(5);
                    newCell.setCellValue(sumQuoteTaxMny != null ? sumQuoteTaxMny.setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");

                    BigDecimal receiveMny = records.stream().filter(s -> s.getParentOrgId().equals(item.getParentOrgId())).map(ProjectMnyVO::getSumReceiveMny).reduce(BigDecimal.ZERO, BigDecimal::add);
                    newCell = newRow.createCell(6);
                    newCell.setCellValue(receiveMny != null ? receiveMny.setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");
                    BigDecimal yun = BigDecimalUtil.safeSub(sumQuoteTaxMny, receiveMny);
                    newCell = newRow.createCell(7);
                    newCell.setCellValue(yun != null ? yun.setScale(2, BigDecimal.ROUND_HALF_UP).toString() : "0.00");
                    list1.remove(orgRowSpan);
                    list1.add(orgRowSpan + 1);
                }
            });
            workbook.write(outputStream);
        }

    }

}
