package com.ejianc.business.income.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.income.bean.ProductionEntity;
import com.ejianc.business.income.history.ProductionHistoryVo;
import com.ejianc.business.income.service.IProductionService;
import com.ejianc.business.income.utils.ComputeUtil;
import com.ejianc.business.income.utils.DateUtil;
import com.ejianc.business.income.vo.ProductionDetailVo;
import com.ejianc.business.income.vo.ProductionVo;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 产值进度 前端控制器
 * </p>
 *
 * @author yuezx
 * @since 2020-06-04
 */
@RestController
@RequestMapping("/production")
public class ProductionController {
    @Autowired
    private IProductionService productionService;
    @Autowired
    private IOrgApi orgApi;
    /**
     * 新增或编辑
     * @param productionVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<ProductionVo> saveOrUpdate(@RequestBody ProductionVo productionVo) {
        CommonResponse<ProductionVo> response = productionService.saveOrUpdate(productionVo);
        return response;
    }
    /**
     * 根据主键ID查询详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ProductionVo>  queryDetail(@RequestParam Long id) {
        ProductionVo vo= productionService.queryDetail(id);
        return CommonResponse.success(vo);
    }
    /**
     * 查询分页列表
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/pageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<ProductionEntity>> pageList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        //项目名称、合同名称、变更摘要、变更签证编号、经办人
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getOrderMap().put("createTime","desc");
        return CommonResponse.success("查询列表数据成功！",productionService.queryPage(param,false));
    }
    /**
     * 删除
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<ProductionVo> vos) {
        productionService.deleteProduction(vos);
        return CommonResponse.success("删除成功");
    }
    /**
     * 产值进度历史记录
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryProductionHistory", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ProductionHistoryVo> queryProductionHistory(@RequestParam Long id) {
        ProductionHistoryVo vo = productionService.queryProductionHistory(id);
        return CommonResponse.success(vo);
    }
    /**
     * 查询产值进度参照分页列表
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/pageRef", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<ProductionEntity>> pageRef(@RequestParam(value = "contractId") String contractId,
                                                          @RequestParam(value = "searchText", required = false) String searchText,
                                                          @RequestParam(value = "productionDate", required = false) String productionDate) {
        QueryParam queryParam = new QueryParam();
        queryParam.setSearchText(searchText);
        /** 模糊搜索配置字段示例 */
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("contractName");
        queryParam.getFuzzyFields().add("employeeName");
        queryParam.getParams().put("contractId",new Parameter(QueryParam.EQ,contractId));
        queryParam.getParams().put("quoteFlag",new Parameter(QueryParam.EQ,0));
        if(null!=productionDate && productionDate.length()>0){
            queryParam.getParams().put("productionDate",new Parameter(QueryParam.LE,productionDate));
        }
        List<Integer> billStatus=new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStatus));
        queryParam.getOrderMap().put("productionDate","desc");
        queryParam.getOrderMap().put("createTime","desc");
        return CommonResponse.success("查询产值进度参照成功！",productionService.queryList(queryParam,false));
    }

    /**
     * 产值进度明细参照
     * @param productionId
     * @return
     */
    @RequestMapping(value = "/detailRef", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<ProductionDetailVo>> detailRef(@RequestParam Long productionId) {
        return CommonResponse.success("查询产值进度明细成功！",productionService.detailRef(productionId));
    }
    /**
     * 导出
     *
     * @param param
     * @return
     */
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        //项目名称、合同名称、变更摘要、变更签证编号、经办人
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getOrderMap().put("createTime","desc");
        param.setPageIndex(1);
        param.setPageSize(-1);
        IPage<ProductionEntity> pageData = productionService.queryPage(param, false);

        Map<String, Object> beans = new HashMap<String, Object>();
        List<ProductionVo> list = new ArrayList<>();
        if (null != pageData.getRecords() && CollectionUtils.isNotEmpty(pageData.getRecords())) {
            list = BeanMapper.mapList(pageData.getRecords(), ProductionVo.class);
            list.forEach(vo -> {
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            });

        }
        beans.put("records", list);
        ExcelExport.getInstance().export("production-export.xlsx", beans, response);
    }

    /**
     * 查询默认统计日期
     * @param contractId
     * @return
     */
    @RequestMapping(value = "/queryDefaultDate", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ProductionVo> queryDefaultDate(Long contractId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        //已生效状态的单据
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));// 已生效
        queryParam.getOrderMap().put("productionDate", QueryParam.DESC);
        List<ProductionEntity> list = productionService.queryList(queryParam,false);
        Date nextMonth = CollectionUtils.isNotEmpty(list) ? DateUtil.nextMonth(list.get(0).getProductionDate()) : new Date();
        ProductionVo vo = new ProductionVo();
        vo.setProductionDate(nextMonth);
        return CommonResponse.success("查询数据成功！", vo);
    }

    /**
     * 根据合同主键，统计日期获取本年完工产值
     * @param contractId
     * @return
     */
    @GetMapping(value = "/getYearFinishMny")
    @ResponseBody
    public CommonResponse<ProductionVo> getYearFinishMny(@RequestParam(value = "contractId") Long contractId,
            @RequestParam(value = "productionDate") Date productionDate) {
        LambdaQueryWrapper<ProductionEntity> lambda = new LambdaQueryWrapper();
        lambda.eq(ProductionEntity::getContractId, contractId);
        lambda.in(ProductionEntity::getBillState, Arrays.asList("1","3"));
        lambda.last("and date_format(production_date,'%Y')='" + DateUtil.getYear(productionDate)+ "'");
        List<ProductionEntity> list = productionService.list(lambda);
        ProductionVo vo = new ProductionVo();
        if(CollectionUtils.isNotEmpty(list)){
            BigDecimal yearFinishTaxMny = null;
            BigDecimal yearFinishMny = null;
            for(ProductionEntity entity : list){
                yearFinishTaxMny = ComputeUtil.safeAdd(yearFinishTaxMny, entity.getFinishTaxMny());
                yearFinishMny = ComputeUtil.safeAdd(yearFinishMny, entity.getFinishMny());
            }
            vo.setYearFinishTaxMny(yearFinishTaxMny);
            vo.setYearFinishMny(yearFinishMny);
        }
        return CommonResponse.success("查询数据成功！", vo);
    }

    /**
     * 根据合同主键，查询是否有非生效单据、所有月份及最大月份
     * @param contractId
     * @return
     */
    @GetMapping(value = "/getValidate")
    @ResponseBody
    public CommonResponse<JSONObject> getValidate(@RequestParam(value = "id", required = false) Long id, @RequestParam(value = "contractId") Long contractId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getOrderMap().put("productionDate", QueryParam.DESC);
        if(id != null){
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, id));//  编辑过滤自身
        }
        // 所有月份
        List<ProductionEntity> list = productionService.queryList(queryParam);
        List<String> allMonth = list.stream().map(x->DateUtil.getYearAndMonth(x.getProductionDate())).distinct().collect(Collectors.toList());
        JSONObject json = new JSONObject();
        json.put("allMonth", allMonth);
        // 最大月份
        json.put("maxDate", CollectionUtils.isNotEmpty(list) ? DateUtil.getYearAndMonth(list.get(0).getProductionDate()) : null);

        // 是否有非生效单据
        queryParam.getParams().put("billState", new Parameter(QueryParam.NOT_IN, "1,3"));// 非生效态
        List<ProductionEntity> list2 = productionService.queryList(queryParam);
        json.put("unEffective", CollectionUtils.isNotEmpty(list2) ? true : false);

        return CommonResponse.success("查询数据成功！", json);
    }
}
