package com.ejianc.business.income.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.income.bean.ContractEntity;
import com.ejianc.business.income.history.ProductionPlanHistoryVo;
import com.ejianc.business.income.service.IContractService;
import com.ejianc.business.income.utils.ComputeUtil;
import com.ejianc.business.income.utils.MonthUtil;
import com.ejianc.business.income.vo.ProductionPlanVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.income.mapper.ProductionPlanMapper;
import com.ejianc.business.income.bean.ProductionPlanEntity;
import com.ejianc.business.income.service.IProductionPlanService;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 产值计划
 * 
 * @author generator
 * 
 */
@Service("productionPlanService")
public class ProductionPlanServiceImpl extends BaseServiceImpl<ProductionPlanMapper, ProductionPlanEntity> implements IProductionPlanService{
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final String PRODUCTION_PLAN = "PRODUCTION_PLAN";

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IContractService contractService;

    @Autowired
    private ProductionPlanMapper planMapper;

    @Override
    public CommonResponse<ProductionPlanVO> saveOrUpdate(ProductionPlanVO saveorUpdateVO) {

        ContractEntity contractEntity = contractService.selectById(saveorUpdateVO.getContractId());
        // 竣工校验
        if(contractEntity != null && contractEntity.getIsFinish() == 1){
            return CommonResponse.error("该合同已做竣工结算，不能做产值计划！");
        }

        Jedis jedis = jedisPool.getResource();
        boolean locked = false;
        try {
            locked = RedisTool.tryLock(jedis, String.valueOf(saveorUpdateVO.getContractId()), "saveOrUpdate", 1000);
            logger.info("判断单据单据锁结果------" + locked);
            if (saveorUpdateVO.getContractVersion() != null && saveorUpdateVO.getContractVersion() != 0) {
                if (locked) {
                    Integer version = contractEntity.getVersion() == null ? 0 : contractEntity.getVersion();
                    Integer conVersion = saveorUpdateVO.getContractVersion();
                    if (!version.equals(conVersion)) {
                        return CommonResponse.error("施工合同已被更新，请刷新后重做！");
                    }
                }else{
                    return CommonResponse.error("出现并发操作,请稍后重试！");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (locked) {
                RedisTool.releaseLock(jedis, String.valueOf(saveorUpdateVO.getContractId()), "saveOrUpdate");
            }
            jedis.close();
        }

        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);

        // 保存后后校验
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, saveorUpdateVO.getContractId()));
        queryParam.getOrderMap().put("planMonth", QueryParam.DESC);
        if (saveorUpdateVO.getId() != null && saveorUpdateVO.getId() > 0) {
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, saveorUpdateVO.getId())); // 编辑过滤自身
        }
        List<ProductionPlanEntity> entities = super.queryList(queryParam, false);

        if (ListUtil.isNotEmpty(entities)) {
            List<ProductionPlanEntity> billStateList = entities.stream().filter(list -> !billStatus.contains(list.getBillState())).collect(Collectors.toList());
            if (ListUtil.isNotEmpty(billStateList)) {
                throw new BusinessException("当前合同存在非审批通过态的产值计划，不允许保存！");
            }

            List<String> allMonth = entities.stream().map(ProductionPlanEntity::getPlanMonth).collect(Collectors.toList());
            if (allMonth.contains(saveorUpdateVO.getPlanMonth())) {
                throw new BusinessException("该合同在该计划月份已制定产值计划，不允许保存！");
            }

            DateTime planMonth = DateUtil.parse(saveorUpdateVO.getPlanMonth(), "yyyy-MM");
            DateTime maxDate = DateUtil.parse(entities.get(0).getPlanMonth(), "yyyy-MM");
            if (planMonth.compareTo(maxDate) < 0) {
                throw new BusinessException("该合同存在比当前计划月份更晚的产值计划，不允许保存！");
            }
        }
        DateTime planMonthDate = DateUtil.parse(saveorUpdateVO.getPlanMonth(), "yyyy-MM");
        saveorUpdateVO.setPlanMonthDate(planMonthDate);
        Long tenantId = InvocationInfoProxy.getTenantid();
        ProductionPlanEntity entity = null;
        if (saveorUpdateVO.getId() != null && saveorUpdateVO.getId() > 0) {
            entity = BeanMapper.map(saveorUpdateVO, ProductionPlanEntity.class);
        }else {
            // 新增
            if (StringUtils.isEmpty(saveorUpdateVO.getBillCode())) {
                CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(PRODUCTION_PLAN, tenantId);
                if (billCode.isSuccess()) {
                    saveorUpdateVO.setBillCode(billCode.getData());
                } else {
                    throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
                }
            }

            entity = BeanMapper.map(saveorUpdateVO, ProductionPlanEntity.class);
        }

        super.saveOrUpdate(entity, false);
        ProductionPlanVO vo = BeanMapper.map(entity, ProductionPlanVO.class);

        return CommonResponse.success("保存或修改单据成功！",vo);
    }

    @Override
    public CommonResponse<ProductionPlanVO> getPlanMonth(Long contractId) {
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getOrderMap().put("planMonth", QueryParam.DESC);
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStatus));
        List<ProductionPlanEntity> entities = super.queryList(queryParam, false);

        if (ListUtil.isEmpty(entities)) {
            return CommonResponse.success("查询计划月份成功！", new ProductionPlanVO());
        }

        ProductionPlanVO vo = BeanMapper.map(entities.get(0), ProductionPlanVO.class);
        // 月份加一个月
        vo.setPlanMonth(MonthUtil.monthAdd(new SimpleDateFormat("yyyy-MM"), vo.getPlanMonth(), 1));

        return CommonResponse.success("查询计划月份成功！", vo);
    }

    @Override
    public CommonResponse<ProductionPlanVO> getYearPlanTaxMny(Long contractId, Date planMonth) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getOrderMap().put("planMonth", QueryParam.ASC);
        List<ProductionPlanEntity> entities = super.queryList(queryParam, false);

        if (ListUtil.isEmpty(entities)) {
            return CommonResponse.success("查询本年计划产值成功！", new ProductionPlanVO());
        }

        Map<String, List<ProductionPlanEntity>> map = entities.stream()
                .collect(Collectors.groupingBy(list -> list.getPlanMonth().split("-")[0]));

        String year = String.valueOf(DateUtil.year(planMonth));
        if (map.containsKey(year)) {
            List<ProductionPlanEntity> list = map.get(year);
            ProductionPlanVO vo = BeanMapper.map(list.get(0), ProductionPlanVO.class);
            return CommonResponse.success("查询本年计划产值成功！", vo);
        }else {
            return CommonResponse.success("查询本年计划产值成功！", new ProductionPlanVO());
        }

    }

    @Override
    public ProductionPlanHistoryVo queryProductionPlanHistory(Long id) {
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        List<ProductionPlanEntity> planEntities = planMapper.selectList(new QueryWrapper<ProductionPlanEntity>()
                .eq("contract_id", id)
                .in("bill_state", billStatus).orderByDesc("plan_month", "create_time"));

        ContractEntity contractEntity = contractService.selectById(id);
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();

        ProductionPlanHistoryVo vo = new ProductionPlanHistoryVo();
        vo.setContractId(id);
        vo.setIsFinish(contractEntity.getIsFinish());
        vo.setContractTaxMny(contractTaxMny);

        BigDecimal totalPlanTaxMny = null;
        BigDecimal totalPlanRate = null;
        if (ListUtil.isNotEmpty(planEntities)) {
            totalPlanTaxMny = planEntities.stream().filter(entity -> entity.getPlanTaxMny() != null)
                    .map(ProductionPlanEntity::getPlanTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);

            totalPlanRate = ComputeUtil.safeDiv(totalPlanTaxMny, contractTaxMny);
        }

        vo.setTotalPlanTaxMny(totalPlanTaxMny);
        vo.setTotalPlanRate(ComputeUtil.safeMultiply(totalPlanRate, new BigDecimal("100")));
        vo.setProductionPlanRecord(BeanMapper.mapList(planEntities, ProductionPlanVO.class));

        return vo;
    }

    @Override
    public List<ProductionPlanVO> queryTotalPlanMny(List<Long> orgIds, Integer yearFlag) {
        return planMapper.queryTotalPlanMny(orgIds, yearFlag);
    }

}
