package com.ejianc.business.market.controller.api;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.market.bean.ProjectRegisterEntity;
import com.ejianc.business.market.bean.ProjectSubEntity;
import com.ejianc.business.market.service.IProjectRegisterService;
import com.ejianc.business.market.service.IProjectSubService;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.market.vo.ProjectSubVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/api/project/")
public class ProjectApi {

	@Autowired
	private IOrgApi orgApi;

    @Autowired
    private IProjectRegisterService projectRegisterService;

    @GetMapping("queryProjectBySourceId")
    public CommonResponse<ProjectRegisterVO> queryProjectBySourceId(@RequestParam(value = "sourceId") String sourceId) {
    	ProjectRegisterVO vo = new ProjectRegisterVO();
    	QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
    	queryWrapper.eq("dr", 0);
    	queryWrapper.eq("source_id", sourceId);
    	ProjectRegisterEntity entity = projectRegisterService.getOne(queryWrapper);
    	if(entity!=null){
    		vo = BeanMapper.map(entity, ProjectRegisterVO.class);
    	}
    	return CommonResponse.success("查询成功！", vo);
    }

	@GetMapping("queryProjectDetail")
	public CommonResponse<ProjectRegisterVO> queryProjectDetail(@RequestParam(value = "projectId") Long projectId) {
		ProjectRegisterVO vo = new ProjectRegisterVO();
		ProjectRegisterEntity entity = projectRegisterService.selectById(projectId);
		if(entity!=null){
			vo = BeanMapper.map(entity, ProjectRegisterVO.class);
		}
		return CommonResponse.success("查询成功！", vo);
	}


    /**
     * @Author mrsir_wxp
     * @Date 2020/8/21 根据项目部id查询项目信息
     * @Description queryProjectByProjectDepartmentId
     * @Param [sourceId]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.market.vo.ProjectRegisterVO>
     */
    @GetMapping("queryProjectByProjectDepartmentId")
    public CommonResponse<ProjectRegisterVO> queryProjectByProjectDepartmentId(@RequestParam(value = "projectDepartmentId") Long projectDepartmentId) {
    	ProjectRegisterVO vo = new ProjectRegisterVO();
    	QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
    	queryWrapper.eq("project_department_id", projectDepartmentId);
    	ProjectRegisterEntity entity = projectRegisterService.getOne(queryWrapper);
    	if(entity!=null){
    		vo = BeanMapper.map(entity, ProjectRegisterVO.class);
    	}
    	return CommonResponse.success("查询成功！", vo);
    }

	/**
	 * 根据项目ID查询项目列表
	 *
	 * @param
	 * @return
	 */
    @PostMapping("queryProjectByIds")
    public CommonResponse<List<ProjectRegisterVO>> queryProjectByIds(@RequestBody List<Long> ids) {
    	List<ProjectRegisterVO> vos = new ArrayList<>();
    	QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
    	queryWrapper.eq("dr", 0);
    	queryWrapper.in("id", ids);
    	List<ProjectRegisterEntity> entitys = projectRegisterService.list(queryWrapper);
    	if(entitys!=null){
    		vos = BeanMapper.mapList(entitys, ProjectRegisterVO.class);
    	}
    	return CommonResponse.success("查询成功！", vos);
    }

	/**
	 * 根据ids查询项目详情
	 *
	 * @param ids
	 * @return
	 */
	@PostMapping(value = "queryProjectArray")
	public CommonResponse<JSONArray> queryProjectArray(@RequestBody List<Long> ids){
		List<ProjectRegisterVO> vos = new ArrayList<>();
		QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("dr", 0);
		queryWrapper.in("id", ids);
		List<ProjectRegisterEntity> entitys = projectRegisterService.list(queryWrapper);
//		if(entitys!=null){
//			vos = BeanMapper.mapList(entitys, ProjectRegisterVO.class);
//		}
		return CommonResponse.success("查询成功！", JSONArray.parseArray(JSONObject.toJSONString(entitys)));
	}

	/**
	 * 根据条件查询对应项目ID列表
	 *
	 * @param projectRange 1-全部项目，2-在建项目，3-完工项目，4-本年新签项目
	 * @return
	 */
	@GetMapping(value = "getProjectIdsByProperties")
	CommonResponse<List<Long>> getProjectIdsByProperties(@RequestParam(value = "projectRange") int projectRange) {
		QueryParam queryParam = new QueryParam();
		//查询本租户的项目
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//启用状态
		queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, 0));
		boolean queryNewSigned = false;
		if(projectRange == 2 || projectRange ==3) {
			queryParam.getParams().put("status", new Parameter(QueryParam.EQ, projectRange));
		} else if(projectRange == 4) {
			queryNewSigned = true;
		}
		return CommonResponse.success(projectRegisterService.queryProIdsByProperties(queryParam, queryNewSigned));
	}

	/**
	 * 供方协同---用户获取有权限的项目，为空代表全部项目的权限
	 *
	 * @return
	 */
	@GetMapping(value = "queryProjectByUserId")
	public CommonResponse<List<ProjectRegisterVO>> queryProjectByUserId(@RequestParam(value = "userId") Long userId) {
		return projectRegisterService.queryProjectByUserId(userId);
	}

	/***
	 * @Author mrsir_wxp
	 * @Date 2021/2/1 查询本下启用的项目，参数为需要查询的条数，默认十条
	 * @Description queryProjectList
	 * @Param [num]
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.util.List<com.ejianc.business.market.vo.ProjectRegisterVO>>
	 */
	@GetMapping(value = "queryProjectList")
	public CommonResponse<List<ProjectRegisterVO>> queryProjectList(@RequestParam(value = "num",defaultValue = "10") int num) {
		QueryParam queryParam = new QueryParam();
		queryParam.setPageSize(num);
		queryParam.setPageIndex(1);
		//查询本租户的项目
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		CommonResponse<List<OrgVO>> response = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
		List<OrgVO> orgVOList = response.getData();
		if(!response.isSuccess()){
			return CommonResponse.error("本下组织查询失败！");
		}
		queryParam.getParams().put("projectDepartmentId",new Parameter(QueryParam.IN,orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList())));
		/** 项目状态：0-启用 */
		queryParam.getParams().put("validStatus",new Parameter(QueryParam.EQ, 0));
		/** 创建时间倒序 */
		queryParam.getOrderMap().put("createTime",QueryParam.DESC);
		IPage<ProjectRegisterEntity> page = projectRegisterService.queryPage(queryParam,false);
		System.out.println("查询本下启用的项目，参数为需要查询的条数，默认十条 page.getTotal = "+page.getTotal());
		List<ProjectRegisterVO> res = BeanMapper.mapList(page.getRecords(),ProjectRegisterVO.class);
		System.out.println("查询本下启用的项目，参数为需要查询的条数，默认十条 res.size = "+res.size());
		return CommonResponse.success(res);
	}


	@PostMapping(value = "queryProjectPage")
	public CommonResponse<Page<ProjectRegisterVO>> queryProjectPage(@RequestBody QueryParam queryParam) {
		//查询本租户的项目
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		CommonResponse<List<OrgVO>> response = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
		List<OrgVO> orgVOList = response.getData();
		if(!response.isSuccess()){
			return CommonResponse.error("本下组织查询失败！");
		}
		queryParam.getParams().put("projectDepartmentId",new Parameter(QueryParam.IN,orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList())));
		/** 项目状态：0-启用 */
		queryParam.getParams().put("validStatus",new Parameter(QueryParam.EQ, 0));
		/** 创建时间倒序 */
		queryParam.getOrderMap().putIfAbsent("createTime", QueryParam.DESC);
		IPage<ProjectRegisterEntity> pageData = projectRegisterService.queryPage(queryParam,false);
		Page<ProjectRegisterVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
		if(CollectionUtils.isNotEmpty(pageData.getRecords())){
			List<ProjectRegisterVO> list = BeanMapper.mapList(pageData.getRecords(),ProjectRegisterVO.class);
			result.setRecords(list);
		}
		return CommonResponse.success(result);
	}

	/**
	 * @Author zhangwx
	 * @Date 2021/06/07 根据组织主键查询本下所有项目
	 * @Description queryProjectByProjectDepartmentId
	 * @Param [orgId]
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.util.List<com.ejianc.business.market.vo.ProjectRegisterVO>>
	 */
	@GetMapping(value = "queryChildrenProjectByOrgId")
	CommonResponse<List<ProjectRegisterVO>> queryChildrenProjectByOrgId(@RequestParam(value = "orgId") Long orgId){
		List<Long> departmentIdList = new ArrayList<>();
		List<ProjectRegisterVO> result = new ArrayList<>();
		CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
		if (childOrgResp.isSuccess() && CollectionUtils.isNotEmpty(childOrgResp.getData())) {
			for(OrgVO orgVO : childOrgResp.getData()){
				if(null != orgVO.getOrgType() && orgVO.getOrgType() == 5){
					departmentIdList.add(orgVO.getId());
				}
			}
		}
		if(CollectionUtils.isNotEmpty(departmentIdList)){
			LambdaQueryWrapper<ProjectRegisterEntity> queryWrapper = new LambdaQueryWrapper();
			queryWrapper.in(ProjectRegisterEntity::getProjectDepartmentId, departmentIdList);
			List<ProjectRegisterEntity> list = projectRegisterService.list(queryWrapper);
			if(CollectionUtils.isNotEmpty(list)){
				result = BeanMapper.mapList(list, ProjectRegisterVO.class);
			}
		}
		return CommonResponse.success(result);
	}

	/**
	 * 根据组织主键查询本下项目地址，过滤条件项目状态、关注类型
	 * @return
	 */
	@GetMapping(value = "queryProjects")
	CommonResponse<List<Map>> queryProjects(@RequestParam(value = "orgId") Long orgId,
											@RequestParam(value = "statusList", required = false) List<String> statusList,
											@RequestParam(value = "focusTypeList", required = false) List<String> focusTypeList,
											@RequestParam(value = "engineeringTypeList", required = false) List<Long> engineeringTypeList,
											@RequestParam(value = "searchText", required = false) String searchText){
		List<Long> departmentIdList = new ArrayList<>();
		List<Map> result = new ArrayList<>();
		CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
		if (childOrgResp.isSuccess() && CollectionUtils.isNotEmpty(childOrgResp.getData())) {
			for(OrgVO orgVO : childOrgResp.getData()){
				if(null != orgVO.getOrgType() && orgVO.getOrgType() == 5){
					departmentIdList.add(orgVO.getId());
				}
			}
		}

		if(CollectionUtils.isNotEmpty(departmentIdList)){
			QueryParam queryParam = new QueryParam();
			queryParam.getParams().put("dr", new Parameter(QueryParam.EQ, 0));
			queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIdList));
			if (statusList != null && statusList.size() > 0) {
				queryParam.getParams().put("status", new Parameter(QueryParam.IN, statusList));
			}
			if (focusTypeList != null && focusTypeList.size() > 0) {
				queryParam.getParams().put("focusType", new Parameter(QueryParam.IN, focusTypeList));
			}
			if (engineeringTypeList != null && engineeringTypeList.size() > 0) {
				queryParam.getParams().put("engineeringType", new Parameter(QueryParam.IN, engineeringTypeList));
			}
			if(StringUtils.isNotBlank(searchText)) {
				queryParam.setSearchText(searchText);
				queryParam.getFuzzyFields().add("name");
				queryParam.getFuzzyFields().add("projectManagerName");
			}
			queryParam.getOrderMap().put("createTime","desc");
			List<ProjectRegisterEntity> list = projectRegisterService.queryList(queryParam);
			if(CollectionUtils.isNotEmpty(list)){
				for(ProjectRegisterEntity entity : list){
					Map map=new HashMap();
					map.put("id",entity.getId());
					map.put("name",entity.getName());
					map.put("lat",entity.getLat());
					map.put("lng",entity.getLng());
					map.put("focusType",entity.getFocusType());
					map.put("projectDepartmentId",entity.getProjectDepartmentId());
					map.put("engineeringTypeName",entity.getEngineeringTypeName());
					map.put("projectManagerName",entity.getProjectManagerName());
					result.add(map);
				}
			}
		}
		return CommonResponse.success(result);
	}

}
