package com.ejianc.business.material.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.cost.api.ICostDetailApi;
import com.ejianc.business.cost.vo.CostDetailVO;
import com.ejianc.business.material.bean.InstoreEntity;
import com.ejianc.business.material.bean.InstoreMaterialEntity;
import com.ejianc.business.material.bean.OutStoreEntity;
import com.ejianc.business.material.mapper.InstoreMapper;
import com.ejianc.business.material.mapper.InstoreMaterialMapper;
import com.ejianc.business.material.pub.MaterialStoreState;
import com.ejianc.business.material.pub.MaterialStoreType;
import com.ejianc.business.material.service.IInstoreService;
import com.ejianc.business.material.service.IOutStoreService;
import com.ejianc.business.material.vo.InstoreAccountSumVO;
import com.ejianc.business.material.vo.InstoreMaterialVO;
import com.ejianc.business.material.vo.InstoreVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


@Service
public class InstoreService  extends BaseServiceImpl<InstoreMapper, InstoreEntity> implements IInstoreService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    IOutStoreService iOutStoreService;
    @Autowired
    private ICostDetailApi iCostDetailApi;
    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private InstoreMaterialMapper instoreMaterialMapper;


    @Override
    public IPage<InstoreVO> queryForList(QueryParam queryParam, boolean isEs) {
        IPage<InstoreVO> voPage = null;
        IPage<InstoreEntity> entityPage = super.queryPage(queryParam,isEs);
        if(entityPage!=null){
            voPage = new Page<>();
            voPage.setCurrent(entityPage.getCurrent());
            voPage.setPages(entityPage.getPages());
            voPage.setTotal(entityPage.getTotal());
            voPage.setSize(queryParam.getPageSize());
            voPage.setRecords(BeanMapper.mapList(entityPage.getRecords(),InstoreVO.class));
        }
        return voPage;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void sureToReceive(InstoreEntity entity) {
        entity.getInstoreMaterialList().forEach(e->{
            e.setStoreId(entity.getStoreId());
            e.setStoreState(MaterialStoreState.STORED.getCode());
        });
        super.saveOrUpdate(entity,false);
        OutStoreEntity outStoreEntity = iOutStoreService.selectById(entity.getOutId());
        if(outStoreEntity == null ){
            throw new BusinessException("没有找到对应调拨出库单，确认收料失败！");
        }
        outStoreEntity.setReceiveState(entity.getReceiveState());
        outStoreEntity.setReceivePerson(entity.getReceivePerson());
        outStoreEntity.getOutStoreSubEntities().forEach(o-> o.setStoreState(MaterialStoreState.USED.getCode()));
        iOutStoreService.saveOrUpdate(outStoreEntity,false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void sureToReturn(InstoreEntity entity) {
        super.saveOrUpdate(entity,false);
        OutStoreEntity outStoreEntity = iOutStoreService.selectById(entity.getOutId());
        if(outStoreEntity == null ){
            throw new BusinessException("没有找到对应调拨出库单，确认收料失败！");
        }
        outStoreEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
        outStoreEntity.setReceiveState(entity.getReceiveState());
        outStoreEntity.setReceivePerson(entity.getReceivePerson());
        outStoreEntity.setReturnReason(entity.getPickReturnReason());
        outStoreEntity.getOutStoreSubEntities().forEach(o-> o.setStoreState(MaterialStoreState.OCCUPY.getCode()));
        iOutStoreService.saveOrUpdate(outStoreEntity,false);
    }

    @Override
    public void processCost(InstoreEntity entity) {
        String factor = "1";
        if(MaterialStoreType.RETURN_IN_STORE.getCode().equals(entity.getInstoreType())){
            factor = "-1";
        }
        if(ListUtil.isNotEmpty(entity.getInstoreMaterialList())){
            List<CostDetailVO> list = new ArrayList<>();
            boolean canPush = true;
            for (int i = 0;i<entity.getInstoreMaterialList().size();i++){
                InstoreMaterialEntity sub = entity.getInstoreMaterialList().get(i);
                if(sub.getSubjectId()!=null && sub.getSubjectId()>0L){
                    CostDetailVO c = new CostDetailVO();
                    c.setSubjectId(sub.getSubjectId());
                    c.setSourceId(entity.getId());
                    c.setSourceDetailId(sub.getId());
                    c.setProjectId(entity.getProjectId());
                    c.setHappenTaxMny(sub.getAmount()==null?new BigDecimal("0.00"):sub.getAmount().multiply(new BigDecimal(factor)));
                    if(sub.getAmount()==null){
                        c.setHappenMny(new BigDecimal("0.00"));
                    }else {
                        BigDecimal rate = new BigDecimal("1.00");
                        if("-1".equals(factor)){//物资退库
                            rate = rate.add(sub.getTaxRate()==null?new BigDecimal("0.00"):sub.getTaxRate().divide(new BigDecimal("100.00"),8,BigDecimal.ROUND_HALF_UP));
                        }else {
                            rate = rate.add(entity.getTaxRate()==null?new BigDecimal("0.00"):entity.getTaxRate().divide(new BigDecimal("100.00"),8,BigDecimal.ROUND_HALF_UP));
                        }
                        BigDecimal happenMny = sub.getAmount().divide(rate,8,BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(factor));
                        c.setHappenMny(happenMny);
                    }
                    c.setHappenDate(entity.getInstoreDate());
                    c.setMemo(entity.getNote());
                    c.setCreateUserName(sessionManager.getUserContext().getUserName());
                    c.setSourceType(MaterialStoreType.getStoreTypeNameByCode(entity.getInstoreType()));
                    c.setSourceTabType(MaterialStoreType.getStoreTypeNameByCode(entity.getInstoreType())+"子表");
                    list.add(c);
                }else {
                    canPush = false;
                    break;
                }
            }
            if(canPush){
                CommonResponse<String> response = iCostDetailApi.saveSubject(list);
                logger.info("推送成本"+MaterialStoreType.getStoreTypeNameByCode(entity.getInstoreType())+"结果:"+response.isSuccess()+" msg:"+response.getMsg()+" billId="+entity.getId());
                entity.setRelationFlag("1");
            }else {
                CommonResponse<String> response = iCostDetailApi.deleteSubject(entity.getId());
                logger.info("删除成本"+MaterialStoreType.getStoreTypeNameByCode(entity.getInstoreType())+"结果:"+response.isSuccess()+" msg:"+response.getMsg()+" billId="+entity.getId());
                entity.setRelationFlag("0");
            }
        }else {
            iCostDetailApi.deleteSubject(entity.getId());
            entity.setRelationFlag("0");
        }
    }

    @Override
    public List<InstoreMaterialVO> instoreNumCount(Map<String, Object> queryParam) {
        return instoreMaterialMapper.instoreNumCount(queryParam);
    }

    @Override
    public List<InstoreAccountSumVO> amountSum(List<Long> projectIds) {
        return instoreMaterialMapper.amountSum(projectIds);
    }
}
