package com.ejianc.business.material.controller;

import com.ejianc.business.material.bean.ContractChangeEntity;
import com.ejianc.business.material.bean.ContractDetailChangeEntity;
import com.ejianc.business.material.service.IContractChangeService;
import com.ejianc.business.material.service.IMaterialContractService;
import com.ejianc.business.material.vo.*;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description: 物资合同变更
 * @date 2020/6/12 11:43
 */
@RestController
@RequestMapping(value = "/materialContractChange/")
public class MaterialContractChangeController {

    @Autowired
    private IContractChangeService contractChangeService;

    @Autowired
    private IMaterialContractService materialContractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final String DEFAULT_RULE_CODE = "materialContract";

    /**
     * 合同变更保存/更新
     *
     * @param contractChangeVO
     * @return
     */
    @PostMapping("saveOrUpdate")
    public CommonResponse<ContractChangeVO> save(@RequestBody ContractChangeVO contractChangeVO) {
        //查询变更合同是否已有处在变更中的变更记录
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("material_contract_id", new Parameter(QueryParam.EQ, contractChangeVO.getMaterialContractId()));
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);
        c1.getParams().put("bill_state", new Parameter(QueryParam.NE, BillStateEnum.COMMITED_STATE.getBillStateCode()));
        queryParam.getComplexParams().add(c1);
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NE, BillStateEnum.PASSED_STATE.getBillStateCode()));


        queryParam.getParams().put("id", new Parameter(QueryParam.NE, contractChangeVO.getId()));


        List<ContractChangeEntity> list = contractChangeService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("保存失败，当前合同存在未审批通过的变更记录！");
        }
        if (StringUtils.isBlank(contractChangeVO.getCode())) {
            //自动生成编码
            contractChangeVO.setCode(getAutoCode());
        }

        boolean checkResult = materialContractService.codeCheck(contractChangeVO.getMaterialContractId(), contractChangeVO.getCode());
        if (!checkResult) {
            return CommonResponse.error("保存失败，合同编码重复");
        }

        ContractChangeVO resp = contractChangeService.save(contractChangeVO);
        this.queryPrice(resp);
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 获取自动生成编码并校验重复
     *
     * @return
     */
    private String getAutoCode() {
        String code = null;

        while (StringUtils.isBlank(code)) {
            CommonResponse<String> resp = billCodeApi.getCodeBatchByRuleCode(DEFAULT_RULE_CODE, InvocationInfoProxy.getTenantid());
            if (!resp.isSuccess()) {
                throw new BusinessException("保存合同变更信息失败，自动生成编码失败！");
            }
            boolean checkResult = materialContractService.codeCheck(null, resp.getData());
            if (checkResult) {
                code = resp.getData();
            }
        }

        return code;
    }

    /**
     * 合同变更详情查询
     *
     * @param id
     * @return
     */
    @GetMapping("queryDetail")
    public CommonResponse<ContractChangeVO> detail(@RequestParam(value = "id") Long id) {
        ContractChangeVO vo = null;
        ContractChangeEntity e = contractChangeService.selectById(id);
        if (e != null) {
            vo = BeanMapper.map(e, ContractChangeVO.class);
            this.queryPrice(vo);
        }

        return CommonResponse.success("查询成功！", vo);
    }

    private void queryPrice(ContractChangeVO vo) {
        //根据单据参数控制查询总计划量\价,历史价格区间
        List<ContractDetailChangeVO> materialDetailList = vo.getMaterialDetailList();

        if (CollectionUtils.isNotEmpty(materialDetailList)) {
            MaterialPriceVO priceVO = new MaterialPriceVO(vo.getProjectId());
            List<MaterialPriceVO> detail = new ArrayList<>();
            materialDetailList.forEach(item -> {
                MaterialPriceVO materialPriceVO = new MaterialPriceVO();
                materialPriceVO.setMaterialId(Long.valueOf(item.getMaterialId()));
                detail.add(materialPriceVO);
            });
            priceVO.setDetail(detail);
            Integer isJc = vo.getPurchaseMode().equals("1") ? 1 : 0;
            MaterialPriceVO materialPriceVO = materialContractService.queryPrice(isJc, vo.getMaterialContractId(), priceVO);
            List<MaterialPriceVO> dd = materialPriceVO.getDetail();
            if (CollectionUtils.isNotEmpty(dd)) {
                Map<Long, MaterialPriceVO> priceVOMap = dd.stream().collect(Collectors.toMap(MaterialPriceVO::getMaterialId, account -> account, (v1, v2) -> v2));
                for (ContractDetailChangeVO v : materialDetailList) {
                    String materialId = v.getMaterialId();
                    MaterialPriceVO p = priceVOMap.get(Long.valueOf(materialId));
                    if (null != p) {
                        v.setPlanNum(p.getPlanNum());
                        v.setPlanPrice(p.getPlanPrice());
                        v.setMinPrice(p.getMinPrice());
                        v.setMaxPrice(p.getMaxPrice());
                        v.setPriceArea(p.getPriceArea());
                    }
                }
            }
        }
    }

    /**
     * 合同变更删除
     *
     * @param vos
     * @return
     */
    @PostMapping("delete")
    public CommonResponse<String> delete(@RequestBody List<ContractChangeVO> vos) {
        contractChangeService.deleteById(vos.get(0).getId(), false);
        return CommonResponse.success("删除成功！");
    }
}
