package com.ejianc.business.material.controller;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.material.bean.ContractChangeEntity;
import com.ejianc.business.material.service.IContractChangeService;
import com.ejianc.business.material.service.IMaterialContractService;
import com.ejianc.business.material.utlis.ArchivesUtil;
import com.ejianc.business.material.vo.ContractChangeVO;
import com.ejianc.business.material.vo.ContractDetailChangeVO;
import com.ejianc.business.material.vo.MaterialPriceVO;
import com.ejianc.foundation.share.api.IMaterialApi;
import com.ejianc.foundation.share.vo.MaterialVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description: 物资合同变更
 * @date 2020/6/12 11:43
 */
@RestController
@RequestMapping(value = "/materialContractChange/")
public class MaterialContractChangeController {

    @Autowired
    private IContractChangeService contractChangeService;

    @Autowired
    private IMaterialContractService materialContractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final String DEFAULT_RULE_CODE = "materialContract";
    @Autowired
    private ArchivesUtil archivesUtil;
    @Resource
    private IMaterialApi materialApi;

    /**
     * 合同变更保存/更新
     *
     * @param contractChangeVO
     *
     * @return
     */
    @PostMapping("saveOrUpdate")
    public CommonResponse<ContractChangeVO> save(@RequestBody ContractChangeVO contractChangeVO) {
        //查询变更合同是否已有处在变更中的变更记录
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("material_contract_id", new Parameter(QueryParam.EQ, contractChangeVO.getMaterialContractId()));
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);
        c1.getParams().put("bill_state", new Parameter(QueryParam.NE, BillStateEnum.COMMITED_STATE.getBillStateCode()));
        queryParam.getComplexParams().add(c1);
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NE, BillStateEnum.PASSED_STATE.getBillStateCode()));


        queryParam.getParams().put("id", new Parameter(QueryParam.NE, contractChangeVO.getId()));


        List<ContractChangeEntity> list = contractChangeService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("保存失败，当前合同存在未审批通过的变更记录！");
        }
        if (StringUtils.isBlank(contractChangeVO.getCode())) {
            //自动生成编码
            contractChangeVO.setCode(getAutoCode());
        }
        String changeVersionStr = contractChangeVO.getChangeVersion() > 9 ? String.valueOf(contractChangeVO.getChangeVersion()) : ("0" + contractChangeVO.getChangeVersion());
        contractChangeVO.setChangeCode("BG" + contractChangeVO.getCode() + "-" + changeVersionStr);
        boolean checkResult = materialContractService.codeCheck(contractChangeVO.getMaterialContractId(), contractChangeVO.getCode());
        if (!checkResult) {
            return CommonResponse.error("保存失败，合同编码重复");
        }
        List<ContractDetailChangeVO> materialDetailList = contractChangeVO.getMaterialDetailList();
        if (CollectionUtils.isNotEmpty(materialDetailList)) {
            String sectionNames = null;
            Set sectionNameSet = new HashSet();
            for (ContractDetailChangeVO materialContractDetailSubVO : materialDetailList) {
                String onlykey = materialContractDetailSubVO.getMaterialTypeName() + materialContractDetailSubVO.getMaterialName() + materialContractDetailSubVO.getMeasureUnit() + materialContractDetailSubVO.getSpec()+materialContractDetailSubVO.getMaterialCode();
                materialContractDetailSubVO.setOnlyKey(onlykey);
                if (!"del".equals(materialContractDetailSubVO.getRowState())) {
                    if (StringUtils.isNotBlank(sectionNames)) {
                        if (!sectionNameSet.contains(materialContractDetailSubVO.getSectionName())) {
                            sectionNames += "," + materialContractDetailSubVO.getSectionName();
                            sectionNameSet.add(materialContractDetailSubVO.getSectionName());
                        }
                    } else {
                        sectionNames += materialContractDetailSubVO.getSectionName();
                        sectionNameSet.add(materialContractDetailSubVO.getSectionName());
                    }
                }
            }
            contractChangeVO.setSectionNames(sectionNames);
            JSONObject jsonObject = archivesUtil.batchSaveArchive(materialDetailList, "materialTypeId", "materialId",
                    "materialTypeName", "materialName","materialCode", "unit", "spec", "errorMessage");
            String sourceList = JSON.toJSONString(jsonObject.get("sourceList"), SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat);
            List<ContractDetailChangeVO> contractDetailChangeVOS = JSONObject.parseArray(sourceList, ContractDetailChangeVO.class);
            contractChangeVO.setMaterialDetailList(contractDetailChangeVOS);

            // 根据物资来源ids查询物资档案
            List<Long> sourceIds = contractDetailChangeVOS.stream().map(contractDetailChangeVO -> Long.parseLong(contractDetailChangeVO.getMaterialId())).collect(Collectors.toList());
            List<List<Long>> partition = Lists.partition(sourceIds, 300);
            List<MaterialVO> materialList=new ArrayList<>();
            for (int i = 0; i < partition.size(); i++) {
                CommonResponse<List<MaterialVO>> r = materialApi.queryMaterialByIds(partition.get(i));
                if (!r.isSuccess()) {
                    throw new BusinessException("保存合同信息失败，查询物资档案失败！");
                }
                materialList.addAll(r.getData());
            }
            Map<Long, MaterialVO> materialMap = materialList.stream().collect(Collectors.toMap(MaterialVO::getId, Function.identity(), (key1, key2) -> key2));
            for (ContractDetailChangeVO detailSubVO : contractChangeVO.getMaterialDetailList()) {
                if (materialMap.containsKey(Long.parseLong(detailSubVO.getMaterialId()))) {
                    MaterialVO vo = materialMap.get(Long.parseLong(detailSubVO.getMaterialId()));
                    String assistUnitName = vo.getAssistUnitName();
                    BigDecimal unitAssistUnit = vo.getUnitAssistUnit();
                    if (StringUtils.isNotBlank(assistUnitName) && unitAssistUnit != null) {
                        detailSubVO.setAssistUnitName(assistUnitName);
                        detailSubVO.setUnitAssistUnit(unitAssistUnit);
                        detailSubVO.setNumAssistUnit(NumberUtil.mul(detailSubVO.getCount(), unitAssistUnit));
                        detailSubVO.setPriceAssistUnit(NumberUtil.div(detailSubVO.getTotalAmount(), detailSubVO.getNumAssistUnit(), 4, RoundingMode.HALF_UP));
                    }
                }
            }
        }

        ContractChangeVO resp = contractChangeService.save(contractChangeVO);
        this.queryPrice(resp);
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 获取自动生成编码并校验重复
     *
     * @return
     */
    private String getAutoCode() {
        String code = null;

        while (StringUtils.isBlank(code)) {
            CommonResponse<String> resp = billCodeApi.getCodeBatchByRuleCode(DEFAULT_RULE_CODE, InvocationInfoProxy.getTenantid());
            if (!resp.isSuccess()) {
                throw new BusinessException("保存合同变更信息失败，自动生成编码失败！");
            }
            boolean checkResult = materialContractService.codeCheck(null, resp.getData());
            if (checkResult) {
                code = resp.getData();
            }
        }

        return code;
    }

    /**
     * 合同变更详情查询
     *
     * @param id
     * @return
     */
    @GetMapping("queryDetail")
    public CommonResponse<ContractChangeVO> detail(@RequestParam(value = "id") Long id) {
        ContractChangeVO vo = null;
        ContractChangeEntity e = contractChangeService.selectById(id);
        if (e != null) {
            vo = BeanMapper.map(e, ContractChangeVO.class);
            this.queryPrice(vo);
        }

        return CommonResponse.success("查询成功！", vo);
    }

    private void queryPrice(ContractChangeVO vo) {
        //根据单据参数控制查询总计划量\价,历史价格区间
        List<ContractDetailChangeVO> materialDetailList = vo.getMaterialDetailList();

        if (CollectionUtils.isNotEmpty(materialDetailList)) {
            MaterialPriceVO priceVO = new MaterialPriceVO(vo.getProjectId());
            List<MaterialPriceVO> detail = new ArrayList<>();
            materialDetailList.forEach(item -> {
                MaterialPriceVO materialPriceVO = new MaterialPriceVO();
                materialPriceVO.setMaterialId(Long.valueOf(item.getMaterialId()));
                detail.add(materialPriceVO);
            });
            priceVO.setDetail(detail);
            Integer isJc = vo.getPurchaseMode().equals("1") ? 1 : 0;
            MaterialPriceVO materialPriceVO = materialContractService.queryPrice(isJc, vo.getMaterialContractId(), priceVO);
            List<MaterialPriceVO> dd = materialPriceVO.getDetail();
            if (CollectionUtils.isNotEmpty(dd)) {
                Map<Long, MaterialPriceVO> priceVOMap = dd.stream().collect(Collectors.toMap(MaterialPriceVO::getMaterialId, account -> account, (v1, v2) -> v2));
                for (ContractDetailChangeVO v : materialDetailList) {
                    String materialId = v.getMaterialId();
                    MaterialPriceVO p = priceVOMap.get(Long.valueOf(materialId));
                    if (null != p) {
                        v.setPlanNum(p.getPlanNum());
                        v.setPlanPrice(p.getPlanPrice());
                        v.setMinPrice(p.getMinPrice());
                        v.setMaxPrice(p.getMaxPrice());
                        v.setPriceArea(p.getPriceArea());
                    }
                }
            }
        }
    }

    /**
     * 合同变更删除
     *
     * @param vos
     * @return
     */
    @PostMapping("delete")
    public CommonResponse<String> delete(@RequestBody List<ContractChangeVO> vos) {
        contractChangeService.deleteById(vos.get(0).getId(), false);
        return CommonResponse.success("删除成功！");
    }
}
