package com.ejianc.foundation.message.controller.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.aliyuncs.CommonRequest;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.exceptions.ServerException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.ejianc.foundation.message.config.MessageConfiguration;
import com.ejianc.foundation.message.util.ProduceNumber;
import com.ejianc.foundation.message.util.Validate;
import com.ejianc.foundation.message.vo.SmsCheckParam;
import com.ejianc.foundation.message.vo.SmsMsgSendParam;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springside.modules.nosql.redis.JedisTemplate;

import java.util.Date;
import java.util.Map;

@RestController
@RequestMapping
public class SmsMessageApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private JedisTemplate jedis;
    @Autowired
    private MessageConfiguration messageConfiguration;

    /**
     * 发送短消息
     *
     * @return
     */
    @PostMapping(value = "/api/sms/sendMessage")
    public CommonResponse<String> sendMessage(@RequestBody SmsMsgSendParam param) {
        logger.debug("发送短信参数：{}", param.toString());
        try {
            if(StringUtils.isBlank(param.getTemplateCode())) {
                return CommonResponse.error("短信模板编号不能为空！");
            }
            if(!Validate.isMobile(param.getPhone())) {
                return CommonResponse.error("手机号不合法！");
            }
            Date date = new Date();
            String timeStr = jedis.get(param.getTemplateCode()+param.getPhone()+"time");
            if(StringUtils.isNotBlank(timeStr)) {
                long ltime = Long.parseLong(timeStr);
                if((date.getTime() - ltime) / 1000L < 60L) {
                    jedis.setex(param.getTemplateCode()+param.getPhone()+"time", date.getTime()+"", 600);
                    return CommonResponse.error("一分钟内不能重复发送");
                }
            }
            String rand = ProduceNumber.produce();
            DefaultProfile profile = DefaultProfile.getProfile(messageConfiguration.getRegionId(), messageConfiguration.getAccessKey(), messageConfiguration.getSecret());
            IAcsClient client = new DefaultAcsClient(profile);

            CommonRequest commonRequest = new CommonRequest();
            commonRequest.setSysMethod(MethodType.POST);
            commonRequest.setSysDomain(messageConfiguration.getSysDomain());
            commonRequest.setSysVersion(messageConfiguration.getSysVersion());
            commonRequest.setSysAction(messageConfiguration.getSysAction());
            commonRequest.putQueryParameter("RegionId", messageConfiguration.getRegionId());
            commonRequest.putQueryParameter("PhoneNumbers", param.getPhone());
            commonRequest.putQueryParameter("SignName", StringUtils.isNotBlank(param.getSignName()) ? param.getSignName() : "益企工程云");
            commonRequest.putQueryParameter("TemplateCode", param.getTemplateCode());

            Map<String, String> templateParam = param.getParams();
            templateParam.put("code", rand);
            commonRequest.putQueryParameter("TemplateParam", JSON.toJSONString(templateParam));
            try {
                com.aliyuncs.CommonResponse response = client.getCommonResponse(commonRequest);
                String result = response.getData();
                if(StringUtils.isNotBlank(result)) {
                    JSONObject resultJson = JSON.parseObject(result);
                    if("OK".equals(resultJson.getString("Code"))) {
                        jedis.setex(param.getTemplateCode()+param.getPhone(), rand, 600);
                        jedis.setex(param.getTemplateCode()+param.getPhone()+"time", date.getTime()+"", 600);

                        return CommonResponse.success("短信发送成功");
                    } else {
                        String errorMsg = resultJson.getString("Message");
                        return CommonResponse.error(errorMsg);
                    }
                }
            } catch (ServerException e) {
                logger.error("短信发送异常,", e);
            } catch (ClientException e) {
                logger.error("短信发送异常,", e);
            }
        } catch(Exception e) {
            logger.error("短信发送异常,", e);
        }
        return CommonResponse.error("短信发送失败");
    }

    /**
     * 检查验证码是否正确
     *
     * @return
     * @param param
     */
    @PostMapping(value = "/api/sms/checkMessage")
    public CommonResponse<String> checkMessage(@RequestBody SmsCheckParam param) {
        logger.debug("短信验证码校验参数 - param：{}", JSONObject.toJSONString(param));
        try {
            String phone = param.getPhone();
            String templateCode = param.getTemplateCode();
            String validate = param.getValidate();
            if(StringUtils.isBlank(templateCode)) {
                return CommonResponse.error("消息模板不能为空");
            }
            if(StringUtils.isBlank(phone)) {
                return CommonResponse.error("手机号不能为空");
            }
            if(StringUtils.isBlank(validate)) {
                return CommonResponse.error("验证码不能为空");
            }
            String ran = jedis.get(templateCode + phone);
            if(StringUtils.isBlank(ran)) {
                return CommonResponse.error("手机验证码已经过期，请重新获取短信验证码！");
            }
            if(validate.equals(ran)) {
                jedis.del(templateCode + phone);
                return CommonResponse.success("短信验证成功");
            }else{
                return CommonResponse.error("手机验证码输入错误，请确认之后重新输入！");
            }
        }catch(Exception e) {
            logger.error("短信验证码校验 - param, 异常：{}", JSONObject.toJSONString(param), e);
            return CommonResponse.error("验证失败，请检查传参是否合法");
        }
    }
}
