package com.ejianc.business.procBid.utils;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

public class HKDateUtil {

    // 使用 ThreadLocal 保证 SimpleDateFormat 的线程安全
    private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT_THREAD_LOCAL =
            ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd"));

    /**
     * 根据输入字符串获取 Excel 日期
     *
     * @param dateStr 输入字符串（可以是数字或日期格式）
     * @return 转换后的 Date 对象
     */
    public static Date getExcelDate(String dateStr) {
        if (dateStr == null || dateStr.trim().isEmpty()) {
            throw new IllegalArgumentException("入参不能为空");
        }

        Date date = null;
        // 校验是否为数字类型（Excel 日期格式）
        boolean isNumeric = isDouble(dateStr);

        try {
            if (isNumeric) {
                // 如果是数字类型（Excel 日期格式），使用 HSSFDateUtil 处理
                date = HSSFDateUtil.getJavaDate(Double.parseDouble(dateStr));
            } else {
                // 如果是日期字符串，解析并校验日期
                date = parseAndValidateDate(dateStr);
            }
        } catch (Exception e) {
            throw new RuntimeException("错误处理日期: " + dateStr, e);
        }

        return date;
    }

    /**
     * 判断输入字符串是否为数字
     *
     * @param s 输入字符串
     * @return 是否为数字
     */
    private static boolean isDouble(String s) {
        try {
            Double.parseDouble(s);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    /**
     * 解析并校验日期字符串
     *
     * @param s 输入的日期字符串
     * @return 解析后的 Date 对象
     * @throws ParseException 如果日期格式不合法
     */
    private static Date parseAndValidateDate(String s) throws ParseException {
        SimpleDateFormat dateFormat = DATE_FORMAT_THREAD_LOCAL.get();
        dateFormat.setLenient(false); // 禁用宽松模式，严格校验日期
        return dateFormat.parse(s);
    }

}
